package ru.yandex.direct.core.entity.uac.service.trackingurl

import org.springframework.stereotype.Component
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.TrackingSystem
import ru.yandex.direct.core.entity.uac.model.TrackingUrl
import ru.yandex.direct.core.entity.uac.validation.TrackingUrlDefectIds
import ru.yandex.direct.core.entity.uac.validation.invalidTrackingUrl
import ru.yandex.direct.utils.UrlUtils
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.direct.validation.constraint.StringConstraints
import ru.yandex.direct.validation.defect.CommonDefects.invalidValue
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.DefectId

// TODO Переиспользовать валидатор трекинговых ссылок и код из TrackingUrlNormalizerService
@Component
class TrackingUrlParseService(
    private val parsers: List<UrlParser>,
) {
    fun validateTracking(url: String?): Defect<Void>? =
        validate(url, ParserType.TRACKING_URL, TrackingUrlDefectIds.Gen.INVALID_TRACKING_URL)

    fun validateImpression(url: String?): Defect<Void>? =
        validate(url, ParserType.IMPRESSION_URL, TrackingUrlDefectIds.Gen.INVALID_IMPRESSION_URL)

    private fun validate(url: String?, type: ParserType, defectId: DefectId<Void>): Defect<Void>? {
        if (url.isNullOrBlank()) {
            return null
        }
        if (!StringConstraints.isValidHref(url)) {
            return invalidValue()
        }
        val urlParts = UrlUtils.laxParseUrl(url)
        val (_, err) = getParser(urlParts, type)
        if (err != null) {
            return Defect(defectId)
        }
        return null
    }

    fun parse(url: String, platform: Platform?, type: ParserType): TrackingUrl? {
        if (url.isBlank() || !StringConstraints.isValidHref(url)) {
            return null
        }
        val urlParts = UrlUtils.laxParseUrl(url)
        return parse(urlParts, platform, type)
    }

    fun parse(urlParts: UrlParts, platform: Platform?, type: ParserType): TrackingUrl? {
        val (parser, _) = getParser(urlParts, type)
        parser ?: return null
        return parser.parse(urlParts, platform)
    }

    private fun getParser(urlParts: UrlParts, type: ParserType): Pair<UrlParser?, UrlMatchResult?> {
        for (parser in parsers) {
            if (parser.type != type) {
                continue
            }
            return when (parser.matches(urlParts)) {
                UrlMatchResult.WRONG_HOST -> continue
                UrlMatchResult.WRONG_PATH -> null to UrlMatchResult.WRONG_PATH
                UrlMatchResult.MATCHES -> parser to null
            }
        }
        return null to UrlMatchResult.WRONG_HOST
    }

    fun createImpressionUrl(url: String, platform: Platform?): Pair<Defect<Void>?, TrackingUrl?> {
        if (url.isBlank() || !StringConstraints.isValidHref(url)) {
            return invalidTrackingUrl() to null
        }
        val urlParts = UrlUtils.laxParseUrl(url)
        val (parser, err) = getParser(urlParts, ParserType.TRACKING_URL)
        if (err != null) {
            return invalidTrackingUrl() to null
        }
        val (defect, impressionUrlParts) = parser!!.createImpressionUrl(urlParts)
        if (defect != null) {
            return defect to null
        }
        return null to parse(impressionUrlParts!!, platform, ParserType.IMPRESSION_URL)
    }

    fun getTrackingSystem(url: String, type: ParserType): TrackingSystem? {
        if (url.isBlank() || !StringConstraints.isValidHref(url)) {
            return null
        }
        val urlParts = UrlUtils.laxParseUrl(url)
        return getTrackingSystem(urlParts, type)
    }

    fun getTrackerId(url: String, type: ParserType): String? {
        if (url.isBlank() || !StringConstraints.isValidHref(url)) {
            return null
        }
        val urlParts = UrlUtils.laxParseUrl(url)
        return getTrackerId(urlParts, type)
    }

    fun getTrackingSystem(urlParts: UrlParts, type: ParserType): TrackingSystem? {
        val (parser, _) = getParser(urlParts, type)
        return parser?.trackingSystem
    }

    private fun getTrackerId(urlParts: UrlParts, type: ParserType): String? {
        val (parser, _) = getParser(urlParts, type)
        return parser?.trackerIdFetcher?.getTrackerId(urlParts)
    }
}
