package ru.yandex.direct.core.entity.uac.service.trackingurl

import ru.yandex.direct.core.entity.uac.model.Parameter
import ru.yandex.direct.core.entity.uac.model.Platform
import ru.yandex.direct.core.entity.uac.model.TrackingSystem
import ru.yandex.direct.core.entity.uac.model.TrackingUrl
import ru.yandex.direct.core.entity.uac.validation.cannotMakeImpressionUrl
import ru.yandex.direct.utils.model.UrlParts
import ru.yandex.direct.validation.result.Defect

enum class UrlMatchResult {
    MATCHES,
    WRONG_PATH,
    WRONG_HOST,
}

enum class ParserType {
    TRACKING_URL,
    IMPRESSION_URL,
}

interface TrackerIdFetcher {
    fun getTrackerId(urlParts: UrlParts): String?
}

class NullTrackerIdFetcher: TrackerIdFetcher {
    override fun getTrackerId(urlParts: UrlParts): String? {
        return null
    }
}

class InPathTrackerIdFetcher: TrackerIdFetcher {
    override fun getTrackerId(urlParts: UrlParts): String? {
        return urlParts.path.trimEnd('/').split('/').lastOrNull()
    }
}

data class CreateImpressionUrlResult(
    val defect: Defect<Void>? = null,
    val urlParts: UrlParts? = null,
)

abstract class UrlParser(
    val type: ParserType,
) {

    companion object {
        const val LOGID_MACROS = "{logid}"
        const val GOOGLE_AID_MACROS = "{google_aid}"
        const val IOS_IFA_MACROS = "{ios_ifa}"
        const val OAID_MACROS = "{oaid}"
        const val ANDROID_ID_MACROS = "{android_id}"
        const val CAMPAIGN_MACROS = "{campaign_id}_{campaign_name_lat}"
        const val ADGROUP_MACROS = "{gbid}"
        const val CREATIVE_MACROS = "{ad_id}_{phrase_id}{retargeting_id}_{keyword}{adtarget_name}"
    }

    abstract val hostPatterns: List<Regex>
    open val pathPattern: Regex? = null

    abstract val trackingSystem: TrackingSystem

    open val hasImpression: Boolean = false
    open val trackerIdFetcher: TrackerIdFetcher = NullTrackerIdFetcher()

    open val parameters: List<Parameter> = listOf()

    open val skadNetworkIntegrated: Boolean = false
    open val externalLink: String? = null

    fun matches(urlParts: UrlParts): UrlMatchResult {
        for (pattern in hostPatterns) {
            if (pattern.find(urlParts.domain) != null) {
                if (pathPattern == null || pathPattern?.find(urlParts.path) != null) {
                    return UrlMatchResult.MATCHES
                }
                return UrlMatchResult.WRONG_PATH
            }
        }
        return UrlMatchResult.WRONG_HOST
    }

    fun parse(urlParts: UrlParts, platform: Platform?): TrackingUrl {
        val params = urlParts.parameters
            ?.filter { (_, value) -> value?.isNotEmpty() ?: false }

        val builder = urlParts.toBuilder()
            .withParameters(params)

        applyParams(builder, platform)

        val resultUrlParts = builder.build()
        return TrackingUrl(
            resultUrlParts,
            trackingSystem,
            skadNetworkIntegrated,
            externalLink ?: "",
            hasImpression,
            trackerIdFetcher.getTrackerId(resultUrlParts),
            parameters,
        )
    }

    private fun applyParams(builder: UrlParts.Builder, platform: Platform?) {
        parameters.filter {
            it.required && (it.platform == null || platform == it.platform)
        }.forEach {
            if (it.overwrite) {
                builder.replaceOrAddParam(it.name, it.value)
            } else {
                builder.addParamIfNotExists(it.name, it.value)
            }
        }

        transformParams(builder)
    }

    open fun transformParams(builder: UrlParts.Builder) {
    }

    open fun createImpressionUrl(urlParts: UrlParts) = CreateImpressionUrlResult(
        defect = cannotMakeImpressionUrl()
    )
}
