package ru.yandex.direct.core.entity.urlmonitoring.service;


import java.net.IDN;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignForNotifyUrlMonitoring;
import ru.yandex.direct.core.entity.campaign.model.SmsFlag;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.UrlMonitoringEventNotification;
import ru.yandex.direct.core.entity.urlmonitoring.model.UrlMonitoringNotificationStatistics;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class UrlMonitoringService {
    private static final int MAX_DOMAINS_TO_IMPORT = 150_000;
    private static final Logger logger = LoggerFactory.getLogger(UrlMonitoringService.class);
    private static final YtCluster IMPORT_CLUSTER = YtCluster.MARKOV;
    private final CampaignService campaignService;
    private final NotificationService notificationService;
    private final YtProvider ytProvider;
    private final UserService userService;

    @Autowired
    public UrlMonitoringService(CampaignService campaignService,
                                UserService userService, NotificationService notificationService,
                                YtProvider ytProvider) {
        this.campaignService = campaignService;
        this.notificationService = notificationService;
        this.ytProvider = ytProvider;
        this.userService = userService;
    }

    public Map<String, UrlMonitoringNotificationStatistics> notifyUsersOnDomainsStateChange(
            Map<String, Set<Pair<String, String>>> domainsByState, boolean dryRun) {
        Map<String, UrlMonitoringNotificationStatistics> statisticsMap = new HashMap<>();
        for (String domainState : domainsByState.keySet()) {
            int campDomainPairs = 0;
            int notifiedCamps = 0;
            int notifiedDomains = 0;
            int notificationsCount = 0;
            int failedNotifications = 0;
            int smsCount = 0;

            Set<Pair<String, String>> domainsInState = domainsByState.get(domainState);
            for (List<Pair<String, String>> domainsInStateChunk : Iterables.partition(domainsInState, 5_000)) {
                Map<Long, Map<String, List<CampaignForNotifyUrlMonitoring>>> campaignsByDomainInStateByUser =
                        campaignService.getCampaignsForNotifyUrlMonitoring(domainsInStateChunk);

                Map<Long, User> users = listToMap(userService.massGetUser(campaignsByDomainInStateByUser.keySet()),
                        User::getId, identity());

                // по каждому UID
                for (Long userId : campaignsByDomainInStateByUser.keySet()) {
                    User user = users.get(userId);
                    Map<String, List<CampaignForNotifyUrlMonitoring>> userCampaignsByDomainInState =
                            campaignsByDomainInStateByUser.get(user.getUid());
                    List<CampaignForNotifyUrlMonitoring> campaigns = userCampaignsByDomainInState.values().stream()
                            .flatMap(Collection::stream)
                            .collect(Collectors.toList());
                    campDomainPairs++;
                    boolean needSms =
                            !filterList(campaigns, c -> c.getSmsFlags().contains(SmsFlag.NOTIFY_METRICA_CONTROL_SMS))
                                    .isEmpty();
                    Set<String> userDomainsInState = userCampaignsByDomainInState.keySet();
                    UrlMonitoringEventNotification notification = new UrlMonitoringEventNotification()
                            .withUid(user.getUid())
                            .withLogin(user.getLogin())
                            .withClientId(user.getClientId().asLong())
                            .withNeedSms(needSms)
                            .withState(domainState)
                            .withDomains(userDomainsInState)
                            .withCampaigns(campaigns);
                    if (!dryRun) {
                        try {
                            notificationService.addNotification(notification);
                        } catch (RuntimeException e) {
                            failedNotifications++;
                            logger.error("Notification failed " + JsonUtils.toJson(notification), e);
                        }
                    } else {
                        logger.debug("Dry run. Will not send notification: {}", notification);
                    }
                    // statistics
                    notifiedCamps += campaigns.size();
                    notifiedDomains += domainsInState.size();
                    notificationsCount++;
                    smsCount += needSms ? 1 : 0;
                }
            }
            statisticsMap.put(
                    domainState,
                    new UrlMonitoringNotificationStatistics()
                            .withCampDomainPairs(campDomainPairs)
                            .withNotifiedCamps(notifiedCamps)
                            .withNotifiedDomains(notifiedDomains)
                            .withNotificationsCount(notificationsCount)
                            .withFailedNotifications(failedNotifications)
                            .withSmsCount(smsCount)
            );
        }
        return statisticsMap;
    }

    public List<String> importDeadDomains() {
        YtDynamicOperator readOperator = ytProvider.getDynamicOperator(IMPORT_CLUSTER);
        List<UnversionedRow> tableContents = readOperator.runRpcCommand(client -> {
            UnversionedRowset rowSet = client.selectRows(
                    "R.url AS url "
                            + "FROM [//home/yabs/url_monitoring/states] AS R "
                            + "WHERE list_contains(R.sources, 'direct') AND R.status = 0 "
                            + "LIMIT " + MAX_DOMAINS_TO_IMPORT).join(); // IGNORE-BAD-JOIN DIRECT-149116
            int rows = rowSet.getRows().size();
            logger.info("Got {} rows", rows);
            if (rows == MAX_DOMAINS_TO_IMPORT) {
                logger.error("Domains number to import reached upper limit of {} rows. "
                        + "Increase the limit and ask BS support team about the reasons", MAX_DOMAINS_TO_IMPORT);
            }
            return rowSet.getRows();
        });
        return mapList(tableContents, row -> row.getValues().get(0).stringValue());
    }

    public List<String> getDeadUrls(Set<String> urlsToCheck) {
        YtDynamicOperator readOperator = ytProvider.getDynamicOperator(IMPORT_CLUSTER);
        Set<String> urlSet = urlsToCheck.stream()
                .filter(Objects::nonNull)
                .filter(url -> url.contains("://"))
                .collect(toSet());
        if (urlSet.isEmpty()) {
            return Collections.emptyList();
        }
        Map<String, String> urlEncodingVariantToUrl = urlSet.stream()
                .flatMap(url -> {
                    String[] split = url.split("://");
                    String protocol = split[0];
                    String domain = split[1];
                    return Stream.of(IDN.toASCII(domain), IDN.toUnicode(domain))
                            .map(domainVariant -> new AbstractMap.SimpleEntry<>(protocol + "://" + domainVariant, url));
                })
                .distinct()
                .collect(toMap(AbstractMap.SimpleEntry::getKey, AbstractMap.SimpleEntry::getValue));
        String urlEncodingVariantsSQLPredicate = urlEncodingVariantToUrl.keySet().stream()
                .map(url -> "is_prefix('" + url + "', R.url)")
                .collect(joining(" OR "));
        List<UnversionedRow> tableContents = readOperator.runRpcCommand(client -> {
            UnversionedRowset rowSet = client.selectRows(
                    String.format("regex_extract('(http[s]?://[^?/#]+)([?/#].*)?', R.url, '\\\\1') AS url "
                            + "FROM [//home/yabs/url_monitoring/states] AS R "
                            + "WHERE list_contains(R.sources, 'direct') AND R.status = 0 "
                            + "AND (%s)", urlEncodingVariantsSQLPredicate)).join(); // IGNORE-BAD-JOIN DIRECT-149116
            return rowSet.getRows();
        });
        Set<String> foundUrlEncodingVariants = listToSet(tableContents, row -> row.getValues().get(0).stringValue());
        Map<String, String> result = EntryStream.of(urlEncodingVariantToUrl)
                .filterKeys(foundUrlEncodingVariants::contains)
                .toMap();
        return new ArrayList<>(result.values());
    }
}
