package ru.yandex.direct.core.entity.user.repository;

import java.sql.Timestamp;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.jooq.Record;
import org.jooq.util.mysql.MySQLDSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.user.model.ApiEnabled;
import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.dbschema.ppc.enums.ClientsApiOptionsApiEnabled;
import ru.yandex.direct.dbschema.ppc.enums.UsersApiOptionsApiAllowFinanceOperations;
import ru.yandex.direct.dbschema.ppc.enums.UsersApiOptionsApiOffer;
import ru.yandex.direct.dbschema.ppc.enums.UsersApiOptionsApiSendMailNotifications;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.jooqmapper.write.JooqWriterBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;

import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromNullableYesNoEnumFieldToBoolean;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToYesNo;
import static ru.yandex.direct.core.entity.user.UserLangUtils.EXT_UKRAINIAN_LANG_SUBTAG;
import static ru.yandex.direct.core.entity.user.UserLangUtils.INT_UKRAINIAN_LANG_SUBTAG;
import static ru.yandex.direct.dbschema.ppc.Tables.API_SPECIAL_USER_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_API_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS_AUTOBAN;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_BRANDS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS_API_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.USERS_OPTIONS;
import static ru.yandex.direct.i18n.Language.fromLangString;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.CommonUtils.nvl;


@Repository
public class ApiUserRepository {
    private static final String SPECIAL_OPTION_CONCURRENT_CALLS = "concurrent_calls";

    private static final Logger logger = LoggerFactory.getLogger(ApiUserRepository.class);

    private final DslContextProvider dslContextProvider;

    // potentially, rbac will be integrated into PPC database,
    // so, it's ok to use rbacService from UserRepository
    private final RbacService rbacService;

    private static final JooqReaderWithSupplier<ApiUser> USER_READER =
            JooqReaderWithSupplierBuilder.builder(ApiUser::new)
                    .readProperty(ApiUser.UID, fromField(USERS.UID))
                    .readProperty(ApiUser.LOGIN, fromField(USERS.LOGIN))
                    .readProperty(ApiUser.CLIENT_ID, fromField(USERS.CLIENT_ID).by(ClientId::fromNullableLong))
                    .readProperty(ApiUser.FIO, fromField(USERS.FIO))
                    .readProperty(ApiUser.PHONE, fromField(USERS.PHONE))
                    .readProperty(ApiUser.EMAIL, fromField(USERS.EMAIL))
                    .readProperty(ApiUser.LANG, fromField(USERS.LANG).by(lang -> fromLangString(
                            INT_UKRAINIAN_LANG_SUBTAG.equals(lang) ? EXT_UKRAINIAN_LANG_SUBTAG : lang)))
                    .readProperty(ApiUser.CREATE_TIME,
                            fromField(USERS.CREATETIME).by(v -> new Timestamp(v).toLocalDateTime()))
                    .readProperty(ApiUser.ALLOWED_IPS, fromField(USERS.ALLOWED_IPS).by(v -> nvl(v, "")))
                    .readProperty(ApiUser.SEND_NEWS, fromNullableYesNoEnumFieldToBoolean(USERS.SEND_NEWS))
                    .readProperty(ApiUser.SEND_ACC_NEWS, fromNullableYesNoEnumFieldToBoolean(USERS.SEND_ACC_NEWS))
                    .readProperty(ApiUser.SEND_WARN, fromNullableYesNoEnumFieldToBoolean(USERS.SEND_WARN))
                    .readProperty(ApiUser.STATUS_BLOCKED, fromNullableYesNoEnumFieldToBoolean(USERS.STATUS_BLOCKED))
                    .readProperty(ApiUser.STATUS_ARCH, fromNullableYesNoEnumFieldToBoolean(USERS.STATUS_ARCH))
                    .readProperty(ApiUser.STATUS_EASY, fromNullableYesNoEnumFieldToBoolean(USERS_OPTIONS.STATUS_EASY))
                    .readProperty(ApiUser.PASSPORT_KARMA, fromField(USERS_OPTIONS.PASSPORT_KARMA).by(v -> nvl(v, 0L)))
                    .readProperty(ApiUser.API_ENABLED,
                            fromField(CLIENTS_API_OPTIONS.API_ENABLED).by(ApiUserRepository::convertApiEnabled))
                    .readProperty(ApiUser.API_GEO_ALLOWED,
                            fromNullableYesNoEnumFieldToBoolean(USERS_API_OPTIONS.API_GEO_ALLOWED))
                    .readProperty(ApiUser.API_UNITS_DAILY, fromField(USERS_API_OPTIONS.API_UNITS_DAILY))
                    .readProperty(ApiUser.API5_UNITS_DAILY, fromField(CLIENTS_API_OPTIONS.API5_UNITS_DAILY))
                    .readProperty(ApiUser.BRAND_CLIENT_ID, fromField(CLIENT_BRANDS.BRAND_CLIENT_ID))
                    .readProperty(ApiUser.API_ALLOWED_IPS, fromField(USERS_API_OPTIONS.API_ALLOWED_IPS))
                    .readProperty(ApiUser.AUTOBANNED,
                            fromNullableYesNoEnumFieldToBoolean(CLIENTS_AUTOBAN.IS_AUTOBANNED))
                    .readProperty(ApiUser.API_SPECIAL_USER_OPTIONS_CONCURRENT_CALLS,
                            fromField(API_SPECIAL_USER_OPTIONS.VALUE))
                    .readProperty(ApiUser.API_FINANCIAL_OPERATIONS_ALLOWED,
                            fromNullableYesNoEnumFieldToBoolean(USERS_API_OPTIONS.API_ALLOW_FINANCE_OPERATIONS))
                    .readProperty(ApiUser.API_OFFER_ACCEPTED,
                            fromField(USERS_API_OPTIONS.API_OFFER).by(ApiUserRepository::booleanFromApiOfferAccepted))
                    .readProperty(ApiUser.API_SEND_MAIL_NOTIFICATIONS,
                            fromNullableYesNoEnumFieldToBoolean(USERS_API_OPTIONS.API_SEND_MAIL_NOTIFICATIONS))
                    .readProperty(ApiUser.AGENCY_CLIENT_ID, fromField(CLIENTS.AGENCY_CLIENT_ID))
                    .readProperty(ApiUser.IS_READONLY_REP, fromField(USERS.REP_TYPE).by(UserMappings::isReadonlyRepFromSource))
                    .build();

    private static final JooqWriter<ApiUser> API_FINANCE_SPECIFIC_FIELDS_WRITER =
            JooqWriterBuilder.<ApiUser>builder()
                    .writeField(USERS_API_OPTIONS.UID, fromProperty(ApiUser.UID))
                    .writeField(USERS_API_OPTIONS.API_ALLOW_FINANCE_OPERATIONS,
                            fromProperty(ApiUser.API_FINANCIAL_OPERATIONS_ALLOWED)
                                    .by(boolValue -> booleanToYesNo(boolValue,
                                            UsersApiOptionsApiAllowFinanceOperations.class)))
                    .writeField(USERS_API_OPTIONS.API_OFFER,
                            fromProperty(ApiUser.API_OFFER_ACCEPTED)
                                    .by(ApiUserRepository::apiOfferAcceptedFromBoolean))
                    .writeField(USERS_API_OPTIONS.API_SEND_MAIL_NOTIFICATIONS,
                            fromProperty(ApiUser.API_SEND_MAIL_NOTIFICATIONS)
                                    .by(boolValue -> booleanToYesNo(boolValue,
                                            UsersApiOptionsApiSendMailNotifications.class)))
                    .build();

    private static UsersApiOptionsApiOffer apiOfferAcceptedFromBoolean(
            Boolean apiOfferAccepted) {
        return apiOfferAccepted ? UsersApiOptionsApiOffer.accepted : UsersApiOptionsApiOffer.rejected;
    }

    private static Boolean booleanFromApiOfferAccepted(UsersApiOptionsApiOffer apiOfferAccepted) {
        return UsersApiOptionsApiOffer.accepted == apiOfferAccepted;
    }

    @Autowired
    public ApiUserRepository(DslContextProvider dslContextProvider,
                             RbacService rbacService) {
        this.dslContextProvider = dslContextProvider;
        this.rbacService = rbacService;
    }

    /**
     * Извлекает сведения о пользователе необходимые для аутентификации и авторизации по uid-и
     *
     * @param shard Номер shard-а на котором расположены данные пользователя
     * @param uid   uid пользователя
     */
    @Nullable
    public ApiUser fetchByUid(int shard, @Nonnull Long uid) {
        Record record = dslContextProvider.ppc(shard)
                .select(USER_READER.getFieldsToRead())
                .from(USERS)
                .leftJoin(USERS_OPTIONS).on(USERS_OPTIONS.UID.eq(USERS.UID))
                .leftJoin(USERS_API_OPTIONS).on(USERS_API_OPTIONS.UID.eq(USERS.UID))
                .leftJoin(CLIENTS_API_OPTIONS).on(
                        CLIENTS_API_OPTIONS.CLIENT_ID.eq(USERS.CLIENT_ID))
                .leftJoin(CLIENT_BRANDS).on(CLIENT_BRANDS.CLIENT_ID.eq(USERS.CLIENT_ID))
                .leftJoin(CLIENTS_AUTOBAN).on(USERS.CLIENT_ID.eq(CLIENTS_AUTOBAN.CLIENT_ID))
                .leftJoin(API_SPECIAL_USER_OPTIONS).on(
                        USERS.CLIENT_ID.eq(API_SPECIAL_USER_OPTIONS.CLIENT_ID)
                                .and(API_SPECIAL_USER_OPTIONS.KEYNAME.eq(SPECIAL_OPTION_CONCURRENT_CALLS)))
                .leftJoin(CLIENTS).on(USERS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .where(USERS.UID.eq(uid)).fetchOne();

        if (record == null) {
            return null;
        }

        RbacRole rbacRole = rbacService.getUidRole(uid);
        // Такое возможно, если например, хотить super-ом в production
        if (rbacRole == null) {
            logger.error("Can't find any role for user with uid {} in RBAC", uid);
            // Мы не понимаем какую роль пользователь имеет в системе, поэтому считаем,
            // что мы не знаем данного пользователя
            return null;
        }

        ApiUser user = USER_READER.fromDb(record);
        user.setRole(rbacRole);
        user.setChiefUid(rbacService.getChief(user.getUid(), rbacRole));

        return user;
    }

    public static ApiEnabled convertApiEnabled(ClientsApiOptionsApiEnabled recordApiEnabled) {
        if (recordApiEnabled == null) {
            return ApiEnabled.DEFAULT;
        }
        switch (recordApiEnabled) {
            case Yes:
                return ApiEnabled.YES;
            case No:
                return ApiEnabled.NO;
            case Default:
                return ApiEnabled.DEFAULT;
            default:
                throw new IllegalStateException("Unknown enum value");
        }
    }

    public void allowFinancialOperationsAcceptApiOffer(int shard, Long uid) {
        ApiUser toInsert = new ApiUser()
                .withApiFinancialOperationsAllowed(true)
                .withApiOfferAccepted(true)
                .withApiSendMailNotifications(false);
        toInsert.setId(uid);
        new InsertHelper<>(dslContextProvider.ppc(shard), USERS_API_OPTIONS)
                .add(API_FINANCE_SPECIFIC_FIELDS_WRITER, toInsert)
                .onDuplicateKeyUpdate()
                .set(USERS_API_OPTIONS.API_ALLOW_FINANCE_OPERATIONS, UsersApiOptionsApiAllowFinanceOperations.Yes)
                .set(USERS_API_OPTIONS.API_OFFER, UsersApiOptionsApiOffer.accepted)
                .set(USERS_API_OPTIONS.API_SEND_MAIL_NOTIFICATIONS,
                        MySQLDSL.values(USERS_API_OPTIONS.API_SEND_MAIL_NOTIFICATIONS))
                .execute();
    }
}

