package ru.yandex.direct.core.entity.user.repository;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.util.EnumSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.user.UserLangUtils;
import ru.yandex.direct.dbschema.ppc.enums.ClientsRole;
import ru.yandex.direct.dbschema.ppc.enums.InternalUsersIsDeveloper;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsIsOfferAccepted;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsSendclientletters;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsSendclientsms;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsStatuseasy;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsUseCampDescription;
import ru.yandex.direct.dbschema.ppc.enums.UsersRepType;
import ru.yandex.direct.dbschema.ppc.enums.UsersSendaccnews;
import ru.yandex.direct.dbschema.ppc.enums.UsersSendnews;
import ru.yandex.direct.dbschema.ppc.enums.UsersSendwarn;
import ru.yandex.direct.dbschema.ppc.enums.UsersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.UsersStatusblocked;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.rbac.ClientPerm;
import ru.yandex.direct.rbac.RbacRole;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.DateTimeUtils.fromEpochMillis;
import static ru.yandex.direct.utils.DateTimeUtils.fromEpochSeconds;

class UserMappings {
    private static final String TRUE_ENUM = "Yes";

    private static final long TIME_THRESHOLD = LocalDateTime.now().plusYears(100).toEpochSecond(ZoneOffset.UTC);
    private static final ZoneId ZONE_ID = ZoneId.systemDefault();


    private UserMappings() {
    }

    static <T extends Enum<T>> Boolean fromNullableYesNoToBoolean(@Nullable T t) {
        return t != null && t.name().equalsIgnoreCase(TRUE_ENUM);
    }

    static String nullToEmptyString(@Nullable String val) {
        return nvl(val, "");
    }

    static boolean booleanFromLongOrFalse(@Nullable Long value) {
        return Optional.ofNullable(value)
                .map(RepositoryUtils::booleanFromLong)
                .orElse(false);
    }

    static Long booleanToLong(@Nullable Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? RepositoryUtils.TRUE : RepositoryUtils.FALSE;
    }

    /**
     * Костыль, вызванный тем, что в поле писались разные значения: секунды и миллисекунды от начала эпохи.
     *
     * @param value время в секундах или миллисекундах от начала эпохи
     * @return Локальное время.
     */
    static LocalDateTime localDateTimeFromSource(@Nullable Long value) {
        return Optional.ofNullable(value)
                .map(val -> value > TIME_THRESHOLD ? fromEpochMillis(value) : fromEpochSeconds(value))
                .orElse(null);
    }

    static Long localTimeToMilli(@Nullable LocalDateTime value) {
        return Optional.ofNullable(value)
                .or(() -> Optional.of(LocalDateTime.now()))
                .map(v -> v.atZone(ZONE_ID).toInstant().toEpochMilli())
                .orElseThrow();
    }


    static UsersSendnews sendNewsToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersSendnews.Yes : UsersSendnews.No;
    }

    static UsersSendaccnews sendAccNewsToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersSendaccnews.Yes : UsersSendaccnews.No;
    }

    static UsersSendwarn sendWarnToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersSendwarn.Yes : UsersSendwarn.No;
    }

    static UsersOptionsStatuseasy statusEasyToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersOptionsStatuseasy.Yes : UsersOptionsStatuseasy.No;
    }

    static UsersOptionsUseCampDescription useCampDescriptionToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersOptionsUseCampDescription.Yes :
                UsersOptionsUseCampDescription.No;
    }

    static UsersStatusblocked statusBlockedToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersStatusblocked.Yes : UsersStatusblocked.No;
    }

    static UsersStatusarch statusArchToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersStatusarch.Yes : UsersStatusarch.No;
    }

    static InternalUsersIsDeveloper internalUsersIsDeveloperToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? InternalUsersIsDeveloper.Yes : InternalUsersIsDeveloper.No;
    }

    static UsersOptionsSendclientletters sendClientLettersToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersOptionsSendclientletters.Yes : UsersOptionsSendclientletters.No;
    }

    static UsersOptionsSendclientsms sendClientSmsToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersOptionsSendclientsms.Yes : UsersOptionsSendclientsms.No;
    }

    static UsersOptionsIsOfferAccepted isOfferAcceptedToSource(Boolean val) {
        return Objects.equals(val, Boolean.TRUE) ? UsersOptionsIsOfferAccepted.Yes : UsersOptionsIsOfferAccepted.No;
    }

    static Boolean isReadonlyRepFromSource(UsersRepType repType) {
        return repType == UsersRepType.readonly;
    }

    static String langToSource(Language val) {
        return Objects.nonNull(val) ? UserLangUtils.extToInt(val.getLangString()) : Language.RU.getLangString();
    }

    static Language langFromSource(String val) {
        return Language.fromLangString(UserLangUtils.intToExt(val));
    }

    static RbacRole clientsRoleFromSource(ClientsRole clientsRole) {
        return Optional.ofNullable(clientsRole)
                .map(RbacRole::fromSource)
                .orElse(RbacRole.CLIENT);
    }

    static EnumSet<ClientPerm> clientsPermsFromSource(String val) {
        Set<ClientPerm> parsedPerms = ClientPerm.parse(val);
        return EnumSet.copyOf(parsedPerms);
    }

}
