package ru.yandex.direct.core.entity.user.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.common.util.TextUtils;
import ru.yandex.direct.core.entity.client.model.ClientWithOptions;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.model.UserOptions;
import ru.yandex.direct.core.entity.user.model.UsersBlockReasonType;
import ru.yandex.direct.core.entity.user.model.UsersOptionsOptsValues;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbschema.ppc.enums.UsersApiOptionsAllowCreateSubclients;
import ru.yandex.direct.dbschema.ppc.enums.UsersHidden;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsStatuseasy;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.UsersRepType;
import ru.yandex.direct.dbschema.ppc.enums.UsersSendaccnews;
import ru.yandex.direct.dbschema.ppc.enums.UsersSendnews;
import ru.yandex.direct.dbschema.ppc.enums.UsersSendwarn;
import ru.yandex.direct.dbschema.ppc.enums.UsersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.UsersStatusblocked;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.rbac.RbacRepType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacSubrole;
import ru.yandex.direct.rbac.RbacUserLookupService;
import ru.yandex.direct.utils.YamlUtils;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToBoolean;
import static ru.yandex.direct.common.util.RepositoryUtils.updateOrInsert;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Clients.CLIENTS;
import static ru.yandex.direct.dbschema.ppc.tables.ClientsAutoban.CLIENTS_AUTOBAN;
import static ru.yandex.direct.dbschema.ppc.tables.ClientsOptions.CLIENTS_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.InternalUsers.INTERNAL_USERS;
import static ru.yandex.direct.dbschema.ppc.tables.ManagerHierarchy.MANAGER_HIERARCHY;
import static ru.yandex.direct.dbschema.ppc.tables.UserCampaignsFavorite.USER_CAMPAIGNS_FAVORITE;
import static ru.yandex.direct.dbschema.ppc.tables.Users.USERS;
import static ru.yandex.direct.dbschema.ppc.tables.UsersAgency.USERS_AGENCY;
import static ru.yandex.direct.dbschema.ppc.tables.UsersApiOptions.USERS_API_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.UsersCaptcha.USERS_CAPTCHA;
import static ru.yandex.direct.dbschema.ppc.tables.UsersOptions.USERS_OPTIONS;
import static ru.yandex.direct.dbschema.ppcdict.tables.ShardLogin.SHARD_LOGIN;
import static ru.yandex.direct.dbschema.ppcdict.tables.ShardUid.SHARD_UID;
import static ru.yandex.direct.dbschema.ppcdict.tables.XlsReports.XLS_REPORTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.jooqmapperhelper.InsertHelper.saveModelObjectsToDbTable;
import static ru.yandex.direct.rbac.PpcRbac.NO_SUPERVISOR;
import static ru.yandex.direct.utils.CommonUtils.notEquals;
import static ru.yandex.direct.utils.DateTimeUtils.getNowEpochSeconds;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class UserRepository {

    private static final JooqMapperWithSupplier<User> MAPPER = createReadWriteMaper();
    private final DslContextProvider dslContextProvider;
    // potentially, rbac will be integrated into PPC database,
    // so, it's ok to use rbacUserLookupService from UserRepository
    private final RbacUserLookupService rbacUserLookupService;

    @Autowired
    public UserRepository(DslContextProvider dslContextProvider,
                          RbacUserLookupService rbacUserLookupService) {
        this.dslContextProvider = dslContextProvider;
        this.rbacUserLookupService = rbacUserLookupService;
    }

    private static JooqMapperWithSupplier<User> createReadWriteMaper() {
        return JooqMapperWithSupplierBuilder.builder(User::new)

                //ppc.users
                .map(property(User.UID, USERS.UID))
                .map(property(User.LOGIN, USERS.LOGIN))
                .map(clientIdProperty(User.CLIENT_ID, USERS.CLIENT_ID))
                .map(property(User.FIO, USERS.FIO))
                .map(property(User.PHONE, USERS.PHONE))
                .map(property(User.VERIFIED_PHONE_ID, USERS.VERIFIED_PHONE_ID))
                .map(property(User.EMAIL, USERS.EMAIL))
                .map(convertibleProperty(User.CAPTCHA_FREQ, USERS.CAPTCHA_FREQ, RepositoryUtils::nullToZero,
                        RepositoryUtils::nullToZero))
                .map(convertibleProperty(User.REP_TYPE, USERS.REP_TYPE, RbacRepType::fromSource,
                        RbacRepType::toSource))
                .map(convertibleProperty(User.CREATE_TIME, USERS.CREATETIME, UserMappings::localDateTimeFromSource,
                        UserMappings::localTimeToMilli))
                .map(convertibleProperty(User.LANG, USERS.LANG, UserMappings::langFromSource,
                        UserMappings::langToSource))
                .map(convertibleProperty(User.ALLOWED_IPS, USERS.ALLOWED_IPS, UserMappings::nullToEmptyString,
                        UserMappings::nullToEmptyString))
                .map(convertibleProperty(User.SEND_NEWS, USERS.SEND_NEWS, UserMappings::fromNullableYesNoToBoolean,
                        UserMappings::sendNewsToSource))
                .map(convertibleProperty(User.SEND_ACC_NEWS, USERS.SEND_ACC_NEWS,
                        UserMappings::fromNullableYesNoToBoolean,
                        UserMappings::sendAccNewsToSource))
                .map(convertibleProperty(User.SEND_WARN, USERS.SEND_WARN, UserMappings::fromNullableYesNoToBoolean,
                        UserMappings::sendWarnToSource))
                .map(convertibleProperty(User.STATUS_BLOCKED, USERS.STATUS_BLOCKED,
                        UserMappings::fromNullableYesNoToBoolean,
                        UserMappings::statusBlockedToSource))
                .map(convertibleProperty(User.BLOCK_REASON_TYPE, USERS.BLOCK_REASON_TYPE,
                        UsersBlockReasonType::fromSource,
                        val -> UsersBlockReasonType.toSource(firstNonNull(val, UsersBlockReasonType.NOT_SET))))
                .map(convertibleProperty(User.STATUS_ARCH, USERS.STATUS_ARCH, UserMappings::fromNullableYesNoToBoolean,
                        UserMappings::statusArchToSource))
                .map(property(User.DESCRIPTION, USERS.DESCRIPTION))
                .readProperty(User.IS_READONLY_REP, fromField(USERS.REP_TYPE).by(UserMappings::isReadonlyRepFromSource))

                //ppc.users_options
                .writeField(USERS_OPTIONS.UID, fromProperty(User.UID))
                .map(convertibleProperty(User.METRIKA_COUNTERS_NUM, USERS_OPTIONS.YA_COUNTERS,
                        RepositoryUtils::nullToIntegerZero,
                        value -> 0L))
                .map(convertibleProperty(User.GEO_ID, USERS_OPTIONS.GEO_ID, RepositoryUtils::nullToZero,
                        RepositoryUtils::nullToZero))
                .map(convertibleProperty(User.OPTS, USERS_OPTIONS.OPTS, StringUtils::trimToEmpty,
                        UserMappings::nullToEmptyString))
                .map(convertibleProperty(User.STATUS_EASY, USERS_OPTIONS.STATUS_EASY,
                        UserMappings::fromNullableYesNoToBoolean,
                        UserMappings::statusEasyToSource))
                .map(convertibleProperty(User.USE_CAMP_DESCRIPTION, USERS_OPTIONS.USE_CAMP_DESCRIPTION,
                        UserMappings::fromNullableYesNoToBoolean,
                        UserMappings::useCampDescriptionToSource))
                .map(convertibleProperty(User.SEND_CLIENT_LETTERS, USERS_OPTIONS.SEND_CLIENT_LETTERS,
                        UserMappings::fromNullableYesNoToBoolean, UserMappings::sendClientLettersToSource))
                .map(convertibleProperty(User.SEND_CLIENT_SMS, USERS_OPTIONS.SEND_CLIENT_SMS,
                        UserMappings::fromNullableYesNoToBoolean, UserMappings::sendClientSmsToSource))
                .map(convertibleProperty(User.PASSPORT_KARMA, USERS_OPTIONS.PASSPORT_KARMA, RepositoryUtils::nullToZero,
                        RepositoryUtils::nullToZero))
                .map(property(User.RECOMMENDATIONS_EMAIL, USERS_OPTIONS.RECOMMENDATIONS_EMAIL))
                .map(convertibleProperty(User.IS_OFFER_ACCEPTED, USERS_OPTIONS.IS_OFFER_ACCEPTED,
                        UserMappings::fromNullableYesNoToBoolean, UserMappings::isOfferAcceptedToSource))
                .map(property(User.MANAGER_OFFICE_ID, USERS_OPTIONS.MANAGER_OFFICE_ID))

                //ppc.internal_users
                .writeField(INTERNAL_USERS.UID, fromProperty(User.UID))
                .map(convertibleProperty(User.DEVELOPER, INTERNAL_USERS.IS_DEVELOPER,
                        UserMappings::fromNullableYesNoToBoolean, UserMappings::internalUsersIsDeveloperToSource))
                .map(convertibleProperty(User.SUPER_MANAGER, INTERNAL_USERS.IS_SUPER_MANAGER,
                        UserMappings::booleanFromLongOrFalse, UserMappings::booleanToLong))
                .map(property(User.DOMAIN_LOGIN, INTERNAL_USERS.DOMAIN_LOGIN))

                //ppc.clients - read only
                .readProperty(User.ROLE, fromField(CLIENTS.ROLE).by(UserMappings::clientsRoleFromSource))
                .readProperty(User.SUB_ROLE, fromField(CLIENTS.SUBROLE).by(RbacSubrole::fromSource))
                .readProperty(User.CHIEF_UID, fromField(CLIENTS.CHIEF_UID))
                .readProperty(User.AGENCY_USER_ID, fromField(CLIENTS.AGENCY_UID))
                .readProperty(User.AGENCY_CLIENT_ID, fromField(CLIENTS.AGENCY_CLIENT_ID))
                .readProperty(User.MANAGER_USER_ID, fromField(CLIENTS.PRIMARY_MANAGER_UID))
                .readProperty(User.PERMS, fromField(CLIENTS.PERMS).by(UserMappings::clientsPermsFromSource))
                .readProperty(User.CLIENT_CREATE_DATE, fromField(CLIENTS.CREATE_DATE))

                //ppc.clients_autoban - read only
                .readProperty(User.AUTOBANNED,
                        fromField(CLIENTS_AUTOBAN.IS_AUTOBANNED).by(UserMappings::fromNullableYesNoToBoolean))

                //ppc.clients_options
                .readProperty(User.CAN_MANAGE_PRICE_PACKAGES, fromLongFieldToBoolean(
                        CLIENTS_OPTIONS.CAN_MANAGE_PRICE_PACKAGES))
                .readProperty(User.CAN_APPROVE_PRICE_PACKAGES, fromLongFieldToBoolean(
                        CLIENTS_OPTIONS.CAN_APPROVE_PRICE_PACKAGES))
                .build();
    }

    private static JooqMapper<User> getShardUidMapper() {
        return JooqMapperBuilder.<User>builder()
                .map(property(User.UID, SHARD_UID.UID))
                .map(convertibleProperty(User.CLIENT_ID, SHARD_UID.CLIENT_ID, ClientId::fromLong, ClientId::asLong))
                .build();
    }

    private static JooqMapper<User> getShardLoginMapper() {
        return JooqMapperBuilder.<User>builder()
                .map(property(User.UID, SHARD_LOGIN.UID))
                .map(property(User.LOGIN, SHARD_LOGIN.LOGIN))
                .build();
    }

    /**
     * Получить несколько пользователей по списку uid из указанного шарда.
     * <p>
     * Если пользователь, с указанным uid не был найден на шарде, объект с его uid не будет присутстовать в результате
     *
     * @param uids  список пользователей для получения
     * @param shard номер шарда
     * @return пользователи, найденные в базе; порядок элементов не гарантируется
     */
    @Nonnull
    public List<User> fetchByUids(int shard, Collection<Long> uids) {
        if (uids.isEmpty()) {
            return emptyList();
        }

        List<User> users = getPpcDslContext(shard)
                .select(MAPPER.getFieldsToRead())
                .from(USERS)
                .leftJoin(CLIENTS).on(USERS.CLIENT_ID.eq(CLIENTS.CLIENT_ID))
                .leftJoin(USERS_OPTIONS).on(USERS.UID.eq(USERS_OPTIONS.UID))
                .leftJoin(INTERNAL_USERS).on(USERS.UID.eq(INTERNAL_USERS.UID))
                .leftJoin(CLIENTS_AUTOBAN).on(USERS.CLIENT_ID.eq(CLIENTS_AUTOBAN.CLIENT_ID))
                .leftJoin(CLIENTS_OPTIONS).on(USERS.CLIENT_ID.eq(CLIENTS_OPTIONS.CLIENT_ID))
                .where(USERS.UID.in(uids))
                .fetch(MAPPER::fromDb);

        if (users.isEmpty()) {
            return emptyList();
        }

        Map<ClientId, Long> chiefsUidsByClientId = rbacUserLookupService.getChiefsByClientIds(users.stream()
                .map(User::getClientId)
                .collect(toSet()));

        Map<Long, RbacRole> rolesByUids = rbacUserLookupService.getUidsRoles(users.stream()
                .map(User::getUid)
                .collect(toSet()));

        for (User u : users) {
            u.setRole(rolesByUids.get(u.getUid()));
            u.setChiefUid(chiefsUidsByClientId.get(u.getClientId()));
        }

        return users;
    }

    /**
     * Заполняет необходимые таблицы при создании клиента
     */
    public void addClient(int shard, ClientWithOptions clientWithOptions) {
        DSLContext dslContext = getPpcDslContext(shard);

        dslContext.batch(
                dslContext.insertInto(USERS)
                        .columns(
                                USERS.UID,
                                USERS.LOGIN,
                                USERS.EMAIL,
                                USERS.FIO,
                                USERS.CLIENT_ID,
                                USERS.REP_TYPE,
                                USERS.CREATETIME,
                                USERS.LANG,
                                USERS.SEND_NEWS,
                                USERS.SEND_ACC_NEWS,
                                USERS.SEND_WARN)
                        .values(
                                clientWithOptions.getUid(),
                                TextUtils.smartStrip(clientWithOptions.getLogin(), false),
                                // Так делается в perl-ом коде (См. User.pm#create_update_user)
                                clientWithOptions.getEmail(),
                                clientWithOptions.getName(),
                                clientWithOptions.getClientId().asLong(),
                                UsersRepType.chief,
                                getNowEpochSeconds(),
                                clientWithOptions.getNotificationLang().getLangString(),
                                clientWithOptions.isSendNews() ? UsersSendnews.Yes : UsersSendnews.No,
                                clientWithOptions.isSendAccNews() ? UsersSendaccnews.Yes : UsersSendaccnews.No,
                                clientWithOptions.isSendWarn() ? UsersSendwarn.Yes : UsersSendwarn.No),

                dslContext.insertInto(USERS_OPTIONS)
                        .columns(
                                USERS_OPTIONS.UID,
                                USERS_OPTIONS.YA_COUNTERS,
                                USERS_OPTIONS.GEO_ID,
                                USERS_OPTIONS.STATUS_EASY,
                                USERS_OPTIONS.OPTS)
                        .values(
                                clientWithOptions.getUid(),
                                0L,
                                0L,
                                UsersOptionsStatuseasy.No,
                                ""))
                .execute();
    }

    /**
     * Заполняет необходимые таблицы при создании нового пользователя.
     * Тестировалось только создание неглавных представителей клиентов. Для создания пользователей с другими ролями
     * нужно расширять функционал метода и допокрывать его тестами.
     * Поля Uid и Login должы соответствовать данным в Blackbox.
     * Ожидается, что у User обязательно заполнены следующие поля: uid, login, email, fio, role, repType, clientId.
     */
    public void addUsers(int shard, Collection<User> users) {
        if (users.isEmpty()) {
            return;
        }
        List<String> rbacRoleErrors = StreamEx.of(users)
                .filter(user -> notEquals(user.getRole(), RbacRole.CLIENT))
                .map(user -> format("User [uid='%1$d'] has RbacRole '%2$s' which is not supported in this method.",
                        user.getUid(), user.getRole()))
                .toList();
        List<String> rbacRepTypeErrors = StreamEx.of(users)
                .filter(user -> notEquals(user.getRepType(), RbacRepType.MAIN))
                .map(user -> format("User [uid='%1$d'] has RbacRepType '%2$s' which is not supported in this method.",
                        user.getUid(), user.getRole()))
                .toList();
        if (!rbacRoleErrors.isEmpty() || !rbacRepTypeErrors.isEmpty()) {
            String error = StreamEx.of(rbacRoleErrors)
                    .append(rbacRepTypeErrors)
                    .joining("; ");
            throw new IllegalArgumentException(error);
        }

        /* В таком порядке, чтобы гарантировать, что в случае коллизий при гонке потоков, код в одном из потоков
        упадёт, но не привяжет ещё раз уже привязанный uid к другому клиенту.*/
        saveModelObjectsToDbTable(dslContextProvider.ppcdict(), SHARD_UID, getShardUidMapper(), users);
        saveModelObjectsToDbTable(dslContextProvider.ppcdict(), SHARD_LOGIN, getShardLoginMapper(), users);
        saveModelObjectsToDbTable(dslContextProvider.ppc(shard), USERS, MAPPER, users);
        saveModelObjectsToDbTable(dslContextProvider.ppc(shard), USERS_OPTIONS, MAPPER, users);
    }

    List<User> getShardUidIds(Collection<Long> uids) {
        JooqMapper<User> shardUidMapper = getShardUidMapper();
        return dslContextProvider.ppcdict()
                .select(shardUidMapper.getFieldsToRead())
                .from(SHARD_UID)
                .where(SHARD_UID.UID.in(uids))
                .fetch(m -> shardUidMapper.fromDb(m, new User()));
    }

    public Set<Long> usersExistByUids(Collection<Long> uids) {
        if (uids.isEmpty()) {
            return emptySet();
        }
        List<User> shardUidIds = getShardUidIds(uids);
        return listToSet(shardUidIds, User::getUid);
    }

    List<User> getShardLoginIds(Collection<String> logins) {
        JooqMapper<User> shardLoginMapper = getShardLoginMapper();
        return dslContextProvider.ppcdict()
                .select(shardLoginMapper.getFieldsToRead())
                .from(SHARD_LOGIN)
                .where(SHARD_LOGIN.LOGIN.in(logins))
                .fetch(m -> shardLoginMapper.fromDb(m, new User()));
    }

    public Map<Long, String> getLoginsByUids(Collection<Long> uids) {
        return dslContextProvider.ppcdict()
                .select(SHARD_LOGIN.UID, SHARD_LOGIN.LOGIN)
                .from(SHARD_LOGIN)
                .where(SHARD_LOGIN.UID.in(uids))
                .fetchMap(SHARD_LOGIN.UID, SHARD_LOGIN.LOGIN);
    }

    public Set<String> usersExistByLogins(Collection<String> logins) {
        if (logins.isEmpty()) {
            return emptySet();
        }
        List<User> shardLoginIds = getShardLoginIds(logins);
        return listToSet(shardLoginIds, User::getLogin);
    }

    public void updateDomainLogin(int shard, Long uid, String domainLogin) {
        getPpcDslContext(shard).insertInto(INTERNAL_USERS)
                .set(INTERNAL_USERS.UID, uid)
                .set(INTERNAL_USERS.DOMAIN_LOGIN, domainLogin)
                .onDuplicateKeyUpdate()
                .set(INTERNAL_USERS.DOMAIN_LOGIN, MySQLDSL.values(INTERNAL_USERS.DOMAIN_LOGIN))
                .execute();
    }

    public void addManagerInHierarchy(int shard, ClientId managerClientId, Long managerUid) {
        getPpcDslContext(shard).insertInto(MANAGER_HIERARCHY)
                .set(MANAGER_HIERARCHY.MANAGER_CLIENT_ID, managerClientId.asLong())
                .set(MANAGER_HIERARCHY.MANAGER_UID, managerUid)
                .set(MANAGER_HIERARCHY.SUPERVISOR_CLIENT_ID, NO_SUPERVISOR)
                .set(MANAGER_HIERARCHY.SUPERVISOR_UID, NO_SUPERVISOR)
                .onDuplicateKeyIgnore()
                .execute();
    }

    public void setManagerOfficeId(int shard, Long uid, Long managerOfficeId) {
        getPpcDslContext(shard)
                .update(USERS_OPTIONS)
                .set(USERS_OPTIONS.MANAGER_OFFICE_ID, managerOfficeId)
                .where(USERS_OPTIONS.UID.eq(uid))
                .execute();
    }

    public void unblockUser(int shard, Long uid) {
        updateStatusBlocked(shard, uid, UsersStatusblocked.No);
    }

    public void unblockUserByClientIds(int shard, Collection<Long> clientIds) {
        getPpcDslContext(shard)
                .update(USERS)
                .set(USERS.STATUS_BLOCKED, UsersStatusblocked.No)
                .setNull(USERS.DESCRIPTION)
                .set(USERS.BLOCK_REASON_TYPE, UsersBlockReasonType.toSource(UsersBlockReasonType.NOT_SET))
                .where(USERS.CLIENT_ID.in(clientIds))
                .execute();
    }

    public void blockUser(int shard, Long uid) {
        updateStatusBlocked(shard, uid, UsersStatusblocked.Yes);
    }

    public void blockUserByClientIds(int shard, Collection<Long> clientIds,
                                     UsersBlockReasonType blockReasonType,
                                     @Nullable String blockComment) {
        getPpcDslContext(shard)
                .update(USERS)
                .set(USERS.STATUS_BLOCKED, UsersStatusblocked.Yes)
                .set(USERS.BLOCK_REASON_TYPE, UsersBlockReasonType.toSource(blockReasonType))
                .set(USERS.DESCRIPTION, blockComment)
                .where(USERS.CLIENT_ID.in(clientIds))
                .execute();
    }

    private void updateStatusBlocked(int shard, Long uid, UsersStatusblocked statusBlocked) {
        getPpcDslContext(shard)
                .update(USERS)
                .set(USERS.STATUS_BLOCKED, statusBlocked)
                .where(USERS.UID.eq(uid))
                .execute();
    }

    /**
     * Блокирует всех клиентов для пользователя, если сам пользователь заблокирован
     */
    public Integer blockAllClientsForUser(int shard, Collection<Long> uids) {
        var blockedUsers = USERS.as("blockedUsers");
        var allUsersForClientId = USERS.as("allUsersForClientId");
        return getPpcDslContext(shard)
                .update(blockedUsers
                        .join(allUsersForClientId).on(blockedUsers.CLIENT_ID.eq(allUsersForClientId.CLIENT_ID)))
                .set(allUsersForClientId.STATUS_BLOCKED, UsersStatusblocked.Yes)
                .where(blockedUsers.UID.in(uids).and(blockedUsers.STATUS_BLOCKED.eq(UsersStatusblocked.Yes)))
                .execute();
    }

    /**
     * Проверяет по таблице {@link ru.yandex.direct.dbschema.ppc.Tables#USERS} - существует клиент, или нет.
     */
    public boolean clientExists(int shard, long clientId) {
        return getPpcDslContext(shard).select(DSL.val(1))
                .from(USERS)
                .where(USERS.CLIENT_ID.eq(clientId))
                .limit(1)
                .fetchAny() != null;
    }

    /**
     * По списку ClientID возвращает списки uid'ов, соответствующих каждому ClientID
     *
     * @param clientIds коллекция id клиентов
     * @param shard     шард
     * @return мапа (ClientID -> [uid1, ..., uidN])
     */
    public Map<ClientId, List<Long>> getUidsByClientIds(int shard, Collection<ClientId> clientIds) {
        if (clientIds.isEmpty()) {
            return emptyMap();
        }
        return getPpcDslContext(shard)
                .select(USERS.UID, USERS.CLIENT_ID)
                .from(USERS)
                .where(USERS.CLIENT_ID.in(mapList(clientIds, ClientId::asLong)))
                .fetchGroups(r -> ClientId.fromLong(r.get(USERS.CLIENT_ID)), r -> r.get(USERS.UID));
    }

    public Set<Long> getUidsByDomainLogin(int shard, String domainLogin) {
        return getPpcDslContext(shard)
                .select(USERS.UID)
                .from(USERS)
                .join(INTERNAL_USERS)
                .on(INTERNAL_USERS.UID.eq(USERS.UID))
                .where(INTERNAL_USERS.DOMAIN_LOGIN.eq(domainLogin))
                .fetchSet(USERS.UID);
    }

    /**
     * Обновляет таблицы с информацией по пользователю.
     * Вносятся изменения в таблицы: users, users_options, internal_users.
     * Если в appliedChanges есть изменения полей хранящихся в таблицах users_options или internal_users,
     * а у пользователя ещё нет соответствующих строк в этих таблицах, то они будут созданы из модели
     * хранящейся в AppliedChanges#model.
     */
    public void update(int shard, Collection<AppliedChanges<User>> appliedChanges) {
        DSLContext context = dslContextProvider.ppc(shard);
        new UpdateHelper<>(context, USERS.UID)
                .processUpdateAll(MAPPER, appliedChanges)
                .execute();
        updateOrInsert(context, appliedChanges, MAPPER, USERS_OPTIONS.UID);
        updateOrInsert(context, appliedChanges, MAPPER, INTERNAL_USERS.UID);
    }

    /**
     * Обновить поле LastChange для пользователей
     *
     * @param shard      шард
     * @param uids       список uid-ов
     * @param lastChange время, на которое нужно обновить LastChange
     */
    public void updateLastChange(int shard, Collection<Long> uids, LocalDateTime lastChange) {
        getPpcDslContext(shard)
                .update(USERS)
                .set(USERS.LAST_CHANGE, lastChange)
                .where(USERS.UID.in(uids))
                .execute();
    }

    /**
     * Разархивировать заданных пользователей
     *
     * @param shard шард
     * @param uids  список uid-ов
     */
    public void unarchiveUsers(int shard, Collection<Long> uids) {
        updateStatusArch(shard, uids, UsersStatusarch.No);
    }

    /**
     * Получить почтовый ящик пользователя
     *
     * @param shard шард
     * @param uid   идентификатор пользователя
     */
    public String getUserEmail(int shard, Long uid) {
        return getPpcDslContext(shard)
                .select(USERS.EMAIL)
                .from(USERS)
                .where(USERS.UID.eq(uid))
                .fetchOne(USERS.EMAIL);
    }

    public UserOptions getUserOptions(int shard, Long uid){
        var options = getPpcDslContext(shard)
                .select(USERS_OPTIONS.OPTIONS)
                .from(USERS_OPTIONS)
                .where(USERS_OPTIONS.UID.eq(uid))
                .fetchOne(USERS_OPTIONS.OPTIONS);

        if (options == null) return new UserOptions();

        return YamlUtils.fromYaml(options, UserOptions.class);
    }

    public Long getChiefUidByClientId(int shard, Long clientId) {
        return getPpcDslContext(shard)
                .select(USERS.UID)
                .from(USERS)
                .where(USERS.CLIENT_ID.eq(clientId))
                .and(USERS.REP_TYPE.eq(UsersRepType.chief))
                .fetchOne(USERS.UID);
    }

    public int updateRepTypeByUid(int shard, List<Long> uids, UsersRepType repType) {
        return getPpcDslContext(shard)
                .update(USERS)
                .set(USERS.REP_TYPE, repType)
                .where(USERS.UID.in(uids))
                .execute();
    }

    /**
     * Обновить статус архивации для пользователей
     *
     * @param shard      шард
     * @param uids       список uid-ов
     * @param statusArch статус архивации
     */
    private void updateStatusArch(int shard, Collection<Long> uids,
                                  @SuppressWarnings("SameParameterValue") UsersStatusarch statusArch) {
        getPpcDslContext(shard)
                .update(USERS)
                .set(USERS.STATUS_ARCH, statusArch)
                .where(USERS.UID.in(uids))
                .execute();
    }

    /**
     * Проверить может ли агенство создавать клиентов
     *
     * @param agencyMainChiefUid Uid главного представителя агенства
     */
    public Optional<Boolean> isAgencyCanCreateSubclient(int shard, Long agencyMainChiefUid) {
        return Optional.ofNullable(
                getPpcDslContext(shard)
                        .select(USERS_API_OPTIONS.ALLOW_CREATE_SUBCLIENTS)
                        .from(USERS_API_OPTIONS)
                        .where(USERS_API_OPTIONS.UID.eq(agencyMainChiefUid))
                        .fetchOne(USERS_API_OPTIONS.ALLOW_CREATE_SUBCLIENTS))
                .map(e -> e == UsersApiOptionsAllowCreateSubclients.Yes);
    }

    /**
     * Вернуть uid-ы пользователей, которые содержат только mcb или geo кампании
     *
     * @param shard шард
     * @param uids  uid-ы пользователей
     */
    @QueryWithoutIndex("Выборка по ключу в подзапросе, в основном возвращается минимум записей")
    public List<Long> getUserUidsHavingOnlyMcbOrGeoCampaigns(int shard, Collection<Long> uids) {
        Field<Long> campaignUidRef = DSL.field(CAMPAIGNS.UID.getName(), CAMPAIGNS.UID.getType());
        return getPpcDslContext(shard)
                .select(campaignUidRef)
                .from(
                        DSL.select(
                                CAMPAIGNS.UID,
                                // Кол-во mcb и geo кампаний у пользователя с заданным uid-ом
                                DSL.sum(
                                        DSL.when(CAMPAIGNS.TYPE.in(CampaignsType.mcb, CampaignsType.geo), 1)
                                                .otherwise(0))
                                        .cast(Integer.class).as("mcb_and_geo_count"),
                                // Общее кол-во кампаний у пользователя с заданным uid-ом
                                DSL.count().cast(Integer.class).as("total_count"))
                                .from(CAMPAIGNS)
                                .where(CAMPAIGNS.UID.in(uids))
                                .groupBy(CAMPAIGNS.UID))
                .where(DSL.field("mcb_and_geo_count").eq(DSL.field("total_count")))
                .fetch(campaignUidRef);
    }

    /**
     * Установить/сбросить флаг тестового пользователя
     *
     * @param shard  шард
     * @param logins логины пользователей
     * @param hidden значение флага тестового пользователя
     */
    public void setHidden(int shard, Collection<String> logins, UsersHidden hidden) {
        getPpcDslContext(shard)
                .update(USERS)
                .set(USERS.HIDDEN, hidden)
                .where(USERS.LOGIN.in(logins)).and(USERS.HIDDEN.notEqual(hidden))
                .execute();
    }

    /**
     * Для списка ID кампаний возвращает соответствие ID кампании и разрешения на пост-модерацию для ее владельца.
     */
    public Map<Long, UsersOptionsStatuspostmoderate> getUserStatusPostModerateByCampaignId(
            int shard, Collection<Long> campaignIds) {
        return getPpcDslContext(shard)
                .select(CAMPAIGNS.CID, USERS_OPTIONS.STATUS_POSTMODERATE)
                .from(CAMPAIGNS)
                .join(USERS_OPTIONS)
                .on(CAMPAIGNS.UID.eq(USERS_OPTIONS.UID))
                .where(CAMPAIGNS.CID.in(campaignIds))
                .fetchMap(CAMPAIGNS.CID, USERS_OPTIONS.STATUS_POSTMODERATE);
    }

    public List<Long> getCampaignIdsForUsers(int shard, List<Long> uids) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.UID.in(uids))
                .fetch(CAMPAIGNS.CID);
    }

    public void updateUserRelatedTables(int shard, Collection<Long> sourceUids, Long targetUid) {
        updateUserCampaignsFavoriteUids(shard, sourceUids, targetUid);

        updateXlsReportsUids(sourceUids, targetUid);
    }

    private void updateUserCampaignsFavoriteUids(int shard, Collection<Long> sourceUids, Long targetUid) {
        getPpcDslContext(shard)
                .update(USER_CAMPAIGNS_FAVORITE)
                .set(USER_CAMPAIGNS_FAVORITE.UID, targetUid)
                .where(USER_CAMPAIGNS_FAVORITE.UID.in(sourceUids))
                .execute();
    }

    private void updateXlsReportsUids(Collection<Long> sourceUids, Long targetUid) {
        dslContextProvider.ppcdict()
                .update(XLS_REPORTS)
                .set(XLS_REPORTS.UID, targetUid)
                .where(XLS_REPORTS.UID.in(sourceUids))
                .execute();
    }

    /**
     * Удаляет данные пользователя из PPC [dangerous!]
     *
     * @param shard
     * @param uid
     */
    public void deleteUserFromPpc(int shard, Long uid) {
        DSLContext dslContext = getPpcDslContext(shard);

        dslContext.deleteFrom(USERS_API_OPTIONS).where(USERS_API_OPTIONS.UID.eq(uid)).execute();
        dslContext.deleteFrom(USERS_AGENCY).where(USERS_AGENCY.UID.eq(uid)).execute();
        dslContext.deleteFrom(USERS_CAPTCHA).where(USERS_CAPTCHA.UID.eq(uid)).execute();
        dslContext.deleteFrom(USERS_OPTIONS).where(USERS_OPTIONS.UID.eq(uid)).execute();

        dslContext.deleteFrom(USERS).where(USERS.UID.eq(uid)).execute();
    }

    /**
     * Удаляет данные пользователя из PPCDICT [dangerous!]
     *
     * @param uid
     */
    public void deleteUserFromPpcDict(Long uid) {
        dslContextProvider.ppcdict().deleteFrom(SHARD_LOGIN).where(SHARD_LOGIN.UID.eq(uid)).execute();
        dslContextProvider.ppcdict().deleteFrom(SHARD_UID).where(SHARD_UID.UID.eq(uid)).execute();
    }

    private DSLContext getPpcDslContext(int shard) {
        return dslContextProvider.ppc(shard);
    }

    /**
     * Ищет всех пользователей, заблокированных после {@code fromTime}, и возвращает их {@code ClientID}.
     * Для каждого {@code ClientID} также возвращается timestamp последнего изменения среди всех
     * заблокированных представителей этого клиента.
     * <p>
     * Точное время блокировки пользователя мы не знаем, но можем гарантировать,
     * что если {@code LastChange < fromTime}, то пользователя заблокировали до интересующего нас времени,
     * следовательно, он нам точно не интересен.
     * </p>
     *
     * @param shard    Шард, на котором будет выполнен поиск
     * @param fromTime Минимальное значение {@code LastChange} заблокированных пользователей.
     * @return Отображение {@code ClientID => максимальный среди всех представителей клиента LastUpdate}
     */
    public Map<ClientId, LocalDateTime> getRecentlyBlockedClientIds(int shard, LocalDateTime fromTime) {
        Field maxLastChange = DSL.field(DSL.max(USERS.LAST_CHANGE).as("maxLastChange"));

        return getPpcDslContext(shard)
                .select(USERS.CLIENT_ID, maxLastChange)
                .from(USERS)
                .where(USERS.STATUS_BLOCKED.eq(UsersStatusblocked.Yes)
                        .and(USERS.LAST_CHANGE.gt(fromTime)))
                .groupBy(USERS.CLIENT_ID)
                .fetchMap(r -> ClientId.fromLong(r.get(USERS.CLIENT_ID)), r -> r.get(maxLastChange));
    }

    /**
     * Прочитать значение поля USERS_OPTIONS.OPTS
     *
     * @param shard — шард
     * @param uid — id пользователя
     * @return множество взведённых в OPTS флагов
     */
    public Set<UsersOptionsOptsValues> getOpts(int shard, Long uid) {
        var opts = getPpcDslContext(shard)
                .select(USERS_OPTIONS.OPTS)
                .from(USERS_OPTIONS)
                .where(USERS_OPTIONS.UID.eq(uid))
                .fetchOne(USERS_OPTIONS.OPTS);
        return RepositoryUtils.setFromDb(opts, UsersOptionsOptsValues::fromTypedValue);
    }

    /**
     * Записать новое значение поля USERS_OPTIONS.OPTS. В Java сейчас поле используется только внутренним инструментом
     *
     * @param shard — шард
     * @param uid — id пользователя
     * @param values — всё множество флагов в OPTS, которые должны быть взведены
     */
    public void setOpts(int shard, Long uid, Set<UsersOptionsOptsValues> values) {
        var dbValue = RepositoryUtils.setToDb(values, UsersOptionsOptsValues::getTypedValue);
        getPpcDslContext(shard)
                .update(USERS_OPTIONS)
                .set(USERS_OPTIONS.OPTS, dbValue)
                .where(USERS_OPTIONS.UID.eq(uid))
                .execute();
    }

    public Map<Long, String> getFiosByUids(int shard, Collection<Long> uids) {
        return dslContextProvider.ppc(shard)
                .select(USERS.UID, USERS.FIO)
                .from(USERS)
                .where(USERS.UID.in(uids))
                .fetchMap(USERS.UID, USERS.FIO);
    }

}
