package ru.yandex.direct.core.entity.user.repository;

import java.util.Collection;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.USERS_AGENCY;

/**
 * Репозиторий для таблицы users_agency - для специфических свойств пользователей представляющих агентства
 */
@Repository
public class UsersAgencyRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public UsersAgencyRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Возвращает для пользователей представляющих агентства флаг - возможность переносить деньги
     * Если в базе не будет записи по uid, то в результате тоже не будет записи для этого uid
     *
     * @see Tables#USERS_AGENCY#getDisallowMoneyTransfers
     */
    public Map<Long, Boolean> getDisallowMoneyTransfers(int shard, Collection<Long> uids) {
        return dslContextProvider.ppc(shard)
                .select(USERS_AGENCY.UID, USERS_AGENCY.DISALLOW_MONEY_TRANSFER)
                .from(USERS_AGENCY)
                .where(USERS_AGENCY.UID.in(uids))
                .fetchMap(USERS_AGENCY.UID, record -> record.value2() > 0);
    }

    /**
     * Возвращает для пользователей представляющих агентства флаг - показывать ли им контакты агентства
     */
    public Map<Long, Boolean> getShowAgencyContacts(int shard, Collection<Long> uids) {
        return dslContextProvider.ppc(shard)
                .select(USERS_AGENCY.UID, USERS_AGENCY.SHOW_AGENCY_CONTACTS)
                .from(USERS_AGENCY)
                .where(USERS_AGENCY.UID.in(uids))
                .fetchMap(USERS_AGENCY.UID, record -> record.value2() > 0);
    }

    /**
     * Возвращает флаг запрета оплаты для пользователей представляющих агентства
     */
    public Map<Long, Boolean> getIsNoPay(int shard, Collection<Long> uids) {
        return dslContextProvider.ppc(shard)
                .select(USERS_AGENCY.UID, USERS_AGENCY.IS_NO_PAY)
                .from(USERS_AGENCY)
                .where(USERS_AGENCY.UID.in(uids))
                .fetchMap(USERS_AGENCY.UID, record -> record.value2() > 0);
    }
}
