package ru.yandex.direct.core.entity.user.service;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.core.entity.user.model.BlackboxUser;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.entity.user.service.validation.AddUserValidationService;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.rbac.RbacRepType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.singletonList;

/**
 * Сервис для добавления новых пользователей в систему.
 * Управляет хождением во внешние системы + непосредственно сохранением пользователя в базе Direct-а
 */
@ParametersAreNonnullByDefault
@Service
public class AddUserService {

    private final ShardHelper shardHelper;
    private final PpcRbac rbacService;
    private final BlackboxUserService blackboxUserService;
    private final AddUserValidationService addUserValidationService;
    private final TurboLandingService turboLandingService;
    private final UserRepository userRepository;
    private final BalanceService balanceService;

    @Autowired
    public AddUserService(ShardHelper shardHelper, PpcRbac rbacService, BlackboxUserService blackboxUserService,
                          AddUserValidationService addUserValidationService, TurboLandingService turboLandingService, UserRepository userRepository,
                          BalanceService balanceService) {
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.blackboxUserService = blackboxUserService;
        this.addUserValidationService = addUserValidationService;
        this.turboLandingService = turboLandingService;
        this.userRepository = userRepository;
        this.balanceService = balanceService;
    }

    public Result<User> addUserFromBlackbox(Long uid, ClientId clientId) {
        return addUsersFromBlackbox(singletonList(uid), clientId).get(0);
    }

    /**
     * Создаёт нового пользователя по данным из BlackBox.
     * Пока что умеет создавать только НЕ главного представителя для клиента с ролью Client. Для создания пользователй
     * с другими ролями метод нужно дорабатывать.
     */
    public MassResult<User> addUsersFromBlackbox(List<Long> uids, ClientId clientId) {
        Map<Long, BlackboxUser> usersByUids = blackboxUserService.getUsersInfo(uids);
        int shard = shardHelper.getShardByClientId(clientId);
        //передаётся в баланс как оператор связывающий нового пользователя и клиента
        Long chiefUserId = userRepository.getChiefUidByClientId(shard, clientId.asLong());
        List<User> users = StreamEx.of(uids)
                .map(usersByUids::get)
                .map(user -> {
                    if (user != null) {
                        return new User()
                                .withUid(user.getUid())
                                .withLogin(user.getLogin())
                                .withEmail(user.getEmail())
                                .withFio(user.getFio())
                                .withLang(user.getLang())
                                .withClientId(clientId)
                                .withChiefUserId(chiefUserId)
                                .withRole(RbacRole.CLIENT)
                                .withRepType(RbacRepType.MAIN);
                    }

                    return null;
                })
                .toList();
        ValidationResult<List<User>, Defect> validateResult = addUserValidationService.validate(users);
        if (validateResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(users, validateResult);
        }
        balanceService.createUserClientAssociations(users);
        userRepository.addUsers(shard, users);
        rbacService.clearCaches();
        turboLandingService.refreshTurbolandingMetrikaGrants(chiefUserId, clientId);

        return MassResult.successfulMassAction(users, validateResult);
    }


}
