package ru.yandex.direct.core.entity.user.service;

import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.model.ApiUser;
import ru.yandex.direct.core.entity.user.repository.ApiUserRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.rbac.RbacService;

@Service
public class ApiUserService {
    private final ApiUserRepository apiUserRepository;

    private final RbacService rbacService;

    private final ShardHelper shardHelper;

    @Autowired
    public ApiUserService(
            ApiUserRepository apiUserRepository,
            RbacService rbacService,
            ShardHelper shardHelper) {
        this.apiUserRepository = apiUserRepository;
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;
    }

    @Nullable
    public ApiUser getUser(Long uid) {
        int shard = shardHelper.getShardByUserId(uid);
        if (shard == ShardSupport.NO_SHARD) {
            return null;
        }
        return apiUserRepository.fetchByUid(shard, uid);
    }

    @Nullable
    public ApiUser getUser(String login) {
        Long uid = shardHelper.getUidByLogin(login);
        if (uid == null) {
            return null;
        }
        return getUser(uid);
    }

    @Nonnull
    public ApiUser getChiefRepFor(ApiUser user) {
        Long chiefRepUid = rbacService.getChief(user.getUid());
        if (Objects.equals(chiefRepUid, user.getUid())) {
            return user;
        } else {
            ApiUser chiefRep = getUser(chiefRepUid);
            if (chiefRep == null) {
                // FIXME: что делать если chiefRep == null?
                throw new RuntimeException(
                        "Не найден главный представитель для uid=" + user.getUid());
            }
            return chiefRep;
        }
    }

    @Nullable
    public ApiUser getBrandChiefRepFor(ApiUser user) {
        Long brandClientId = user.getBrandClientId();
        if (brandClientId == null) {
            return null;
        }
        long brandChiefRepUid = rbacService.getChiefByClientId(ClientId.fromLong(brandClientId));
        if (Objects.equals(brandChiefRepUid, user.getUid())) {
            return user;
        }
        return getUser(brandChiefRepUid);
    }

    public void allowFinancialOperationsAcceptApiOffer(Long uid) {
        int shard = shardHelper.getShardByClientUidStrictly(uid);
        apiUserRepository.allowFinancialOperationsAcceptApiOffer(shard, uid);
    }
}
