package ru.yandex.direct.core.entity.user.service;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.entity.user.model.BlackboxUser;
import ru.yandex.direct.core.validation.defects.Defects;
import ru.yandex.direct.dbutil.model.LoginOrUid;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.i18n.I18NException;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.inside.passport.AbstractPassportUid;
import ru.yandex.inside.passport.PassportDomain;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.protocol.BlackboxException;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.AliasesParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.EmailsParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.request.params.PhoneAttributesParameterValue;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAbstractResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAddress;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAliases;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxAvatar;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxCorrectResponse;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxDbFields;
import ru.yandex.misc.email.Email;
import ru.yandex.misc.ip.IpAddress;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class BlackboxUserService {
    private static final Logger logger = LoggerFactory.getLogger(BlackboxUserService.class);
    private static final AliasesParameterValue CHECK_USER_ALIASES =
            AliasesParameterValue.types(BlackboxAliases.PORTAL, BlackboxAliases.PDD, BlackboxAliases.PDD_ALIAS);

    private final BlackboxClient blackboxClient;
    private final TvmIntegration tvmIntegration;
    private final TvmService tvmService;

    @Autowired
    public BlackboxUserService(
            BlackboxClient blackboxClient,
            EnvironmentType environmentType,
            TvmIntegration tvmIntegration
    ) {
        this.blackboxClient = blackboxClient;
        this.tvmIntegration = tvmIntegration;
        this.tvmService = environmentType.isProductionOrPrestable()
                ? TvmService.BLACKBOX_PROD
                : TvmService.BLACKBOX_MIMINO;
    }

    private static Language fromLangString(String langString) {
        try {
            return Language.fromLangString(langString);
        } catch (I18NException ex) {
            return null;
        }
    }

    private static BlackboxUser getUser(Long uid, BlackboxAbstractResponse abstractResponse) {
        Optional<BlackboxCorrectResponse> optionalResponse = abstractResponse.getO().toOptional();
        if (optionalResponse.isEmpty()) {
            return null;
        }
        BlackboxCorrectResponse response = optionalResponse.get();

        String login = response.getLogin().toOptional()
                .filter(StringUtils::isNotBlank)
                .orElse(null);

        String avatarId = response.getDisplayName().toOptional()
                .flatMap(e -> e.getAvatar().toOptional())
                .map(BlackboxAvatar::getDefaultAvatarId)
                .orElse(null);

        String email = response.getDefaultEmail().toOptional()
                .map(Email::toString)
                .filter(StringUtils::isNotBlank)
                .orElse(null);

        String fio = response.getDbFields().getOptional(BlackboxDbFields.FIO)
                .filter(StringUtils::isNotBlank)
                .orElse(null);

        Language lang = response.getDbFields().getOptional(BlackboxDbFields.LANG)
                .filter(StringUtils::isNotBlank)
                .map(BlackboxUserService::fromLangString)
                .orElse(null);

        return new BlackboxUser(uid, login, avatarId, email, fio, lang);
    }

    @SuppressWarnings("WeakerAccess")
    public BlackboxUser getUserInfo(Long uid) {
        return getUsersInfo(singletonList(uid)).get(uid);
    }

    public Map<Long, BlackboxUser> getUsersInfo(Collection<Long> uids) {
        List<PassportUid> passportUids = mapList(uids, PassportUid::new);
        /* Используется dbFields, вместо arguments, т.к. на момент написания кода в BlackboxQueryable ещё нет
        перегруженных методов использующих arguments, но не использующих в качестве входных аргументов bolts-коллекции.
        А bolts-коллекции в Директе постепенно становится нельзя:
        https://a.yandex-team.ru/arc/trunk/arcadia/direct/libs/python/style-tests/test_banned_strings
        .py?rev=5648953&blame=true#L38 */
        List<String> dbFields = List.of(BlackboxDbFields.FIO, BlackboxDbFields.LANG);
        Optional<EmailsParameterValue> emails = Optional.of(EmailsParameterValue.GET_DEFAULT);
        Optional<AliasesParameterValue> aliases = Optional.empty();
        Optional<List<PhoneAttributesParameterValue>> phoneAttributes = Optional.empty();
        boolean regName = true;

        String tvmTicket = tvmIntegration.getTicket(tvmService);
        Map<PassportUid, BlackboxAbstractResponse> responseByPassportUids =
                blackboxClient.userInfoBulk(getUserAddress(), passportUids,
                        dbFields, emails, aliases, regName, phoneAttributes, tvmTicket);
        return EntryStream.of(responseByPassportUids)
                .mapKeys(AbstractPassportUid::getUid)
                .mapToValue(BlackboxUserService::getUser)
                .toNavigableMap();
    }

    public Optional<Long> getUidByLogin(String passportLogin) {
        String tvmTicket = tvmIntegration.getTicket(tvmService);
        BlackboxCorrectResponse response = blackboxClient.userInfo(getUserAddress(),
                passportLogin, emptyList(), tvmTicket);
        return response.getUid().toOptional().map(PassportUid::getUid);
    }

    /**
     * Получает информацию о стране пользователя в паспорте.
     *
     * @param uid - паспортный uid, для которого запрашивается страна
     */
    public String getCountryByUid(Long uid) {
        String tvmTicket = tvmIntegration.getTicket(tvmService);
        PassportUid passportUid = new PassportUid(uid);
        BlackboxCorrectResponse response = blackboxClient.userInfo(getUserAddress(),
                passportUid, singletonList(BlackboxDbFields.COUNTRY), tvmTicket);
        return response.getDbFields().getOptional(BlackboxDbFields.COUNTRY)
                .filter(StringUtils::isNotBlank)
                .orElse(null);
    }

    public List<BlackboxAddress> getAllEmailsByUid(Long uid) {
        String tvmTicket = tvmIntegration.getTicket(tvmService);
        BlackboxCorrectResponse response = blackboxClient.userInfo(getUserAddress(),
                new PassportUid(uid), EmailsParameterValue.GET_ALL, tvmTicket);
        return response.getAddresses();
    }

    public Map<String, Long> getUidsByLogins(List<String> passportLogins) {
        String tvmTicket = tvmIntegration.getTicket(tvmService);
        Map<String, Long> result = new HashMap<>(passportLogins.size());
        for (String passportLogin : passportLogins) {
            BlackboxCorrectResponse response = blackboxClient.userInfo(getUserAddress(),
                    passportLogin, emptyList(), tvmTicket);
            Optional<Long> uid = response.getUid().toOptional().map(PassportUid::getUid);
            uid.ifPresent(it -> result.put(passportLogin, it));
        }
        return result;
    }

    public Result<Long> getAndCheckUserId(String userTicket) {
        String tvmTicket = tvmIntegration.getTicket(tvmService);
        BlackboxCorrectResponse user;
        try {
            user = blackboxClient.userTicket(userTicket, CHECK_USER_ALIASES, tvmTicket);
        } catch (BlackboxException e) {
            logger.error("Error while blackbox.userTicket", e);
            return Result.broken(ValidationResult.failed(null, CommonDefects.objectNotFound()));
        }
        long uid;
        if (user.getUid().isPresent()) {
            uid = user.getUid().get().getUid();
        } else {
            logger.error("UID not found in blackbox response. Status: {}, error: {}",
                    user.getStatus(), user.getError());
            return Result.broken(ValidationResult.failed(null, CommonDefects.objectNotFound()));
        }
        ValidationResult<Object, Defect> validationResult = checkUser(uid, user);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        return Result.successful(uid, validationResult);
    }

    public Result<BlackboxCorrectResponse> getAndCheckUser(LoginOrUid loginOrUid) {
        String tvmTicket = tvmIntegration.getTicket(tvmService);
        BlackboxCorrectResponse user;
        try {
            IpAddress address = HttpUtil.getRemoteAddressForBlackbox();
            user = loginOrUid.map(login -> blackboxClient.userInfo(address, login, CHECK_USER_ALIASES, tvmTicket),
                    uid -> blackboxClient.userInfo(address, new PassportUid(uid), CHECK_USER_ALIASES, tvmTicket));
        } catch (BlackboxException e) {
            logger.error("Error while blackbox.userInfo", e);
            return Result.broken(ValidationResult.failed(loginOrUid.get(), CommonDefects.objectNotFound()));
        }
        ValidationResult<Object, Defect> validationResult = checkUser(loginOrUid.get(), user);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        return Result.successful(user, validationResult);
    }

    private <T> ValidationResult<T, Defect> checkUser(T userInfo, BlackboxCorrectResponse blackboxResponse) {
        if (!blackboxResponse.getUid().isPresent()) {
            logger.warn("User {} not found", userInfo);
            return ValidationResult.failed(userInfo, CommonDefects.objectNotFound());
        }
        long uid = blackboxResponse.getUid().get().getUid();
        Set<Integer> userAliases = blackboxResponse.getAliases() != null ?
                blackboxResponse.getAliases().keySet() : emptySet();
        if (userAliases.contains(BlackboxAliases.PDD) || userAliases.contains(BlackboxAliases.PDD_ALIAS)) {
            // Логин из Почты для доменов (ПДД) не может быть зарегистрирован в Директе.
            logger.warn("User {} can not be PDD user", uid);
            return ValidationResult.failed(userInfo, Defects.pddLogin());
        }
        if (!userAliases.contains(BlackboxAliases.PORTAL)) {
            logger.warn("User {} is not a portal user", uid);
            return ValidationResult.failed(userInfo, CommonDefects.invalidValue());
        }
        if (getValidEmail(blackboxResponse) == null) {
            logger.warn("User {} has invalid email", uid);
            return ValidationResult.failed(userInfo, CommonDefects.invalidValue());
        }
        return ValidationResult.success(userInfo);
    }

    @Nullable
    public static String getValidEmail(BlackboxCorrectResponse user) {
        // Если есть валидный дефолтный эмейл, то возвращаем его.
        if (user.getDefaultEmail().isPresent()) {
            String defaultEmail = user.getDefaultEmail().get().getEmail();
            if (Predicates.validEmail().test(defaultEmail)) {
                return defaultEmail;
            }
        }
        // Если логин - валидный эмейл, то возвращаем его (это лайт аккаунт).
        String login = PassportUtils.normalizeLogin(user.getLogin().get());
        if (Predicates.validEmail().test(login)) {
            return login;
        }
        // Если можем сформировать валидный эмейл из логина, то возвращаем его
        String email = login + "@" + user.getUidDomain().map(Tuple2::get2).getOrElse(PassportDomain.YANDEX_RU);
        if (Predicates.validEmail().test(email)) {
            return email;
        }
        // Если ничего из этого не получилось, то возвращаем null
        return null;
    }

    IpAddress getUserAddress() {
        return HttpUtil.getRemoteAddressForBlackbox();
    }
}
