package ru.yandex.direct.core.entity.user.service;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.feature.FeatureName;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Service
@ParametersAreNonnullByDefault
public class UcUserService {
    public static final int CHUNK_SIZE = 100;
    private final UserService userService;
    private final FeatureService featureService;
    private final ShardHelper shardHelper;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignRepository campaignRepository;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public UcUserService(UserService userService, FeatureService featureService,
                         ShardHelper shardHelper, CampaignTypedRepository campaignTypedRepository,
                         CampaignRepository campaignRepository) {
        this.userService = userService;
        this.featureService = featureService;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
    }

    /**
     * Получить нескольких пользователей без включенной фичи универсальных кампаний
     * Пользователи могут находиться в разных шардах.
     *
     * @return отображение uid на объект пользователя User
     */
    @Nonnull
    public Map<Long, User> massGetNonUcUser(@Nullable Collection<Long> uids) {
        if (uids == null || uids.isEmpty()) {
            return Collections.emptyMap();
        }
        Collection<User> users = userService.massGetUser(uids);
        Set<ClientId> clientsIds = users.stream().map(User::getClientId).collect(toSet());
        Map<ClientId, Boolean> enabledForClientIds = featureService.isEnabledForClientIds(clientsIds,
                FeatureName.UNIVERSAL_CAMPAIGNS_ENABLED_FOR_UAC.getName());
        Set<ClientId> clientsIdsWithUCEnabled =
                EntryStream.of(enabledForClientIds).filterValues(enabled -> enabled).keys().toSet();
        if (clientsIdsWithUCEnabled.isEmpty()) {
            return listToMap(users, User::getId);
        } else {
            Map<Long, List<CommonCampaign>> campaignsByClientIds = shardHelper
                    .groupByShard(clientsIdsWithUCEnabled, ShardKey.CLIENT_ID, ClientId::asLong)
                    .chunkedBy(CHUNK_SIZE)
                    .stream()
                    .mapKeyValue((shard, clients)
                            -> getCommonCampaignsByClientIds(shard, clients, CampaignTypeKinds.BASE))
                    .flatMapToEntry(Function.identity())
                    .toMap();
            return users.stream().filter(u -> {
                ClientId clientId = u.getClientId();
                if (!clientsIdsWithUCEnabled.contains(clientId)) {
                    return true;
                }
                List<CommonCampaign> campaigns = campaignsByClientIds.get(clientId.asLong());
                return StreamEx.of(campaigns)
                        .anyMatch(campaign -> !campaign.getIsUniversal());
            }).collect(toMap(User::getId, identity()));
        }
    }

    /**
     * Возвращает: id клиента -> список базовых кампаний клиента (без кошельков, билинговых аггрегатов и т. п.)
     */
    private Map<Long, List<CommonCampaign>> getCommonCampaignsByClientIds(int shard,
                                                                          Collection<ClientId> clientIds,
                                                                          Set<CampaignType> campaignTypes) {
        Set<Long> campaignIds = campaignRepository.getCampaignIdsByClientIds(shard, clientIds, campaignTypes);
        List<CommonCampaign> campaigns = campaignTypedRepository.getStrictly(shard, campaignIds, CommonCampaign.class);
        return StreamEx.of(campaigns)
                .filter(c -> !c.getStatusEmpty())
                .groupingBy(CommonCampaign::getClientId);
    }
}
