package ru.yandex.direct.core.entity.user.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.blackbox.client.BlackboxClient;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.mailnotification.service.MailNotificationEventService;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.payment.service.AutopayService;
import ru.yandex.direct.core.entity.turbolanding.service.TurboLandingService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.model.UsersBlockReasonType;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.entity.user.repository.UsersAgencyRepository;
import ru.yandex.direct.core.entity.user.service.validation.BlockUserValidationService;
import ru.yandex.direct.core.entity.user.utils.BlackboxGetPhoneQuery;
import ru.yandex.direct.core.entity.user.utils.BlackboxGetPhonesQuery;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.dbschema.ppc.enums.UsersHidden;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.rbac.model.Representative;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.utils.PassportUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxPhone;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.mailnotification.model.CampaignEvent.campaignStopped;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.balanceUserAssociatedWithAnotherClient;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.chiefDeletionProhibited;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userHasActiveAutoPay;
import static ru.yandex.direct.core.entity.user.utils.BlockedUserUtil.sendBlockedClientNotifications;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class UserService {
    private final ShardHelper shardHelper;
    private final UserRepository userRepository;
    private final UsersAgencyRepository usersAgencyRepository;
    private final ClientRepository clientRepository;
    private final RbacService rbacService;
    private final BlackboxClient blackboxClient;
    private final BalanceService balanceService;
    private final VcardRepository vcardRepository;
    private final CampaignRepository campaignRepository;
    private final WalletRepository walletRepository;
    private final TurboLandingService turboLandingService;
    private final TvmIntegration tvmIntegration;
    private final EnvironmentType environmentType;
    private final FeatureService featureService;
    private final BlockUserValidationService blockUserValidationService;
    private final AutopayService autopayService;
    private final MailNotificationEventService mailNotificationEventService;
    private final NotificationService notificationService;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public UserService(UserRepository userRepository, UsersAgencyRepository usersAgencyRepository,
                       ShardHelper shardHelper, ClientRepository clientRepository, RbacService rbacService,
                       BlackboxClient blackboxClient, BalanceService balanceService, VcardRepository vcardRepository,
                       CampaignRepository campaignRepository,
                       WalletRepository walletRepository,
                       TurboLandingService turboLandingService, TvmIntegration tvmIntegration,
                       EnvironmentType environmentType, FeatureService featureService,
                       BlockUserValidationService blockUserValidationService, AutopayService autopayService,
                       MailNotificationEventService mailNotificationEventService,
                       NotificationService notificationService) {
        this.userRepository = userRepository;
        this.usersAgencyRepository = usersAgencyRepository;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
        this.rbacService = rbacService;
        this.blackboxClient = blackboxClient;
        this.balanceService = balanceService;
        this.vcardRepository = vcardRepository;
        this.campaignRepository = campaignRepository;
        this.walletRepository = walletRepository;
        this.turboLandingService = turboLandingService;
        this.tvmIntegration = tvmIntegration;
        this.environmentType = environmentType;
        this.featureService = featureService;
        this.blockUserValidationService = blockUserValidationService;
        this.autopayService = autopayService;
        this.mailNotificationEventService = mailNotificationEventService;
        this.notificationService = notificationService;
    }

    @Nullable
    public User getUser(Long uid) {
        return Iterables.getFirst(massGetUser(singletonList(uid)), null);
    }

    /**
     * Получить нескольких пользователей. Пользователи могут находиться в разных шардах.
     *
     * @return отображение uid на объект пользователя User
     */
    @Nonnull
    public List<User> massGetUser(Collection<Long> uids) {
        return shardHelper.groupByShard(uids, ShardKey.UID).stream()
                .map(e -> userRepository.fetchByUids(e.getKey(), e.getValue()))
                .toFlatList(Function.identity());
    }

    /**
     * Получить пользователя по {@code login}
     *
     * @see #massGetUserByLogin(Collection)
     */
    @Nullable
    public User getUserByLogin(String login) {
        return Iterables.getFirst(massGetUserByLogin(singletonList(login)), null);
    }

    /**
     * Получить нескольких пользователей по login. Пользователи могут находиться в разных шардах.
     *
     * @return отображение login на объект пользователя User
     */
    @Nonnull
    public List<User> massGetUserByLogin(Collection<String> logins) {
        List<Long> uids = filterList(shardHelper.getUidsByLogin(new ArrayList<>(logins)), Objects::nonNull);
        return shardHelper.groupByShard(uids, ShardKey.UID).stream()
                .map(e -> userRepository.fetchByUids(e.getKey(), e.getValue()))
                .toFlatList(Function.identity());
    }

    /**
     * Вернуть словарь из id клиента в главного представителя (класса User).
     * Не даёт гарантий, что все входные id будут в ключах ответа
     *
     * @param clientIds — id клиентов
     * @return словарь из id клиента в главного представителя
     */
    @Nonnull
    public Map<ClientId, User> getChiefUserByClientIdMap(Collection<ClientId> clientIds) {
        Map<ClientId, Long> chiefsByClientIds = rbacService.getChiefsByClientIds(clientIds);
        return shardHelper.groupByShard(chiefsByClientIds.values(), ShardKey.UID).stream()
                .map(e -> userRepository.fetchByUids(e.getKey(), e.getValue()))
                .flatMap(Collection::stream)
                .toMap(User::getClientId, Function.identity());
    }

    /**
     * Записывает применённые изменения в базу
     * <p>
     * Перед тем как вызывать этот метод нужно убедиться, что изменения в {@code appliedChanges} корректные
     *
     * @param appliedChanges применённые изменения
     */
    public void update(AppliedChanges<User> appliedChanges) {
        massUpdate(singleton(appliedChanges));
    }

    /**
     * Записывает коллекцию применённых изменения в базу
     * <p>
     * Перед тем как вызывать этот метод нужно убедиться, что изменения в {@code appliedChanges} корректные
     *
     * @param appliedChanges коллекция применённых изменения
     */
    @SuppressWarnings("WeakerAccess")
    public void massUpdate(Collection<AppliedChanges<User>> appliedChanges) {
        shardHelper.groupByShard(appliedChanges, ShardKey.UID, ac -> ac.getModel().getId())
                .forEach(userRepository::update);
    }

    /**
     * Get uid(passport-id) from user by login
     * Login can be not normalized
     *
     * @param login - login of user
     * @return uid of user, null if no such user in metabase
     */
    @Nullable
    public Long getUidByLogin(String login) {
        return massGetUidByLogin(singletonList(login)).get(login);
    }

    /**
     * Mass get uid(passport-id) from user by login
     * Login can be not normalized
     *
     * @param logins - logins of users
     * @return uids of users by their logins
     */
    @Nonnull
    public Map<String, Long> massGetUidByLogin(List<String> logins) {
        List<String> normalizedLogins = mapList(logins, PassportUtils::normalizeLogin);
        return StreamEx.of(logins).zipWith(StreamEx.of(shardHelper.getUidsByLogin(normalizedLogins)))
                .distinctKeys()
                .filterValues(Objects::nonNull)
                .toMap();
    }

    /**
     * Возвращает ClientID по логину.
     */
    @Nullable
    public Long getClientIdByLogin(String login) {
        return massGetClientIdsByLogin(singletonList(login)).get(login);
    }

    /**
     * Возвращает соответствие ClientID их логинам.
     */
    @Nonnull
    public Map<String, Long> massGetClientIdsByLogin(List<String> logins) {
        Map<String, String> normalizedLogins = listToMap(logins, PassportUtils::normalizeLogin);
        return EntryStream
                .of(shardHelper.getClientIdsByLogins(new ArrayList<>(normalizedLogins.keySet())))
                .mapKeys(normalizedLogins::get)
                .toMap();
    }

    public boolean loginExists(String login) {
        return getUidByLogin(login) != null;
    }

    public boolean clientExists(long clientId) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(clientId));
        return shard != NO_SHARD && userRepository.clientExists(shard, clientId);
    }

    /**
     * По списку ClientID возвращает списки uid'ов, соответствующих каждому ClientID.
     * Запрос чанкуется. Размер чанка - {@link SqlUtils#TYPICAL_SELECT_CHUNK_SIZE}
     *
     * @param clientIds - список ClientID
     * @return мапа (ClientID -> [uid1, ..., uidN])
     */
    @Nonnull
    public Map<ClientId, List<Long>> massGetUidsByClientIds(Collection<ClientId> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID, ClientId::asLong)
                .chunkedBy(SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .stream()
                .map(e -> userRepository.getUidsByClientIds(e.getKey(), e.getValue()))
                .flatMapToEntry(Function.identity())
                .toMap();
    }

    /**
     * Проверить может ли агенство создавать клиентов
     *
     * @param agencyMainChief Главный представитель агенства
     */
    public boolean canAgencyCreateSubclient(UidAndClientId agencyMainChief) {
        int shard = shardHelper.getShardByClientId(agencyMainChief.getClientId());
        return userRepository.isAgencyCanCreateSubclient(shard, agencyMainChief.getUid())
                .orElse(false);
    }

    /**
     * Вернуть только uid пользователей, у которых либо нет кампаний, либо есть хотя одна отличная от
     * mcb и geo
     *
     * @param uids uid-ы пользователей
     */
    public Set<Long> getUserUidsWithoutHavingOnlyGeoOrMcbCampaigns(Collection<Long> uids) {
        Set<Long> userUidsHavingOnlyGeoOrMcbCampaigns = shardHelper.groupByShard(uids, ShardKey.UID)
                .stream()
                .flatMap(e -> userRepository.getUserUidsHavingOnlyMcbOrGeoCampaigns(e.getKey(), e.getValue())
                        .stream())
                .toSet();

        return uids.stream()
                .filter(uid -> !userUidsHavingOnlyGeoOrMcbCampaigns.contains(uid))
                .collect(toSet());
    }

    /**
     * Для заданных логинов проставляет тестовый флаг users.hidden = Yes
     *
     * @param logins логины пользователей
     * @return логины пользователей которые нашли при получении списка шардов
     */
    public List<String> massSetHidden(Collection<String> logins) {
        ShardedData<String> foundLogins = shardHelper.groupByShard(logins, ShardKey.LOGIN);

        foundLogins.forEach((shard, shardLogins) -> userRepository.setHidden(shard, shardLogins, UsersHidden.Yes));

        return foundLogins.getData();
    }

    /**
     * Для заданных ClientID возвращает соответствующие им логины представителей-шефов
     *
     * @param clientIds список ClientID
     * @return мапа - ClientID -> логин представителя-шефа
     */
    public Map<ClientId, String> getChiefsLoginsByClientIds(Collection<ClientId> clientIds) {
        Map<ClientId, Long> chiefsByClientIds = new LinkedHashMap<>(rbacService.getChiefsByClientIds(clientIds));
        List<List<String>> loginsByUids = shardHelper.getLoginsByUids(
                new ArrayList<>(chiefsByClientIds.values()));
        return StreamEx.of(chiefsByClientIds.keySet())
                .zipWith(StreamEx.of(loginsByUids)
                        .map(chiefLogins -> !chiefLogins.isEmpty() ? chiefLogins.get(0) : null))
                .toMap();
    }

    /**
     * Получить email ползователя
     *
     * @param userId uid пользователя
     * @return email ползователя
     */
    public String getUserEmail(Long userId) {
        int shard = shardHelper.getShardByUserId(userId);
        return userRepository.getUserEmail(shard, userId);
    }

    /**
     * Возвращает для пользователей представляющих агенства флаг - возможность переносить деньги
     */
    public Map<Long, Boolean> getUsersAgencyDisallowMoneyTransfer(Collection<Long> uids) {
        return shardHelper.groupByShard(uids, ShardKey.UID).stream()
                .mapKeyValue(usersAgencyRepository::getDisallowMoneyTransfers)
                .flatMapToEntry(Function.identity())
                .toMap();
    }

    /**
     * Возвращает флаг запрета оплаты для пользователей представляющих агенства
     */
    public Map<Long, Boolean> getUsersAgencyIsNoPay(Collection<Long> uids) {
        return shardHelper.groupByShard(uids, ShardKey.UID).stream()
                .mapKeyValue(usersAgencyRepository::getIsNoPay)
                .flatMapToEntry(Function.identity())
                .toMap();
    }

    /**
     * Возвращает для пользователей представляющих агенства флаг - показывать ли контакты агентства клиенту
     */
    public Map<Long, Boolean> getUsersAgencyShowAgencyContacts(Collection<Long> uids) {
        return shardHelper.groupByShard(uids, ShardKey.UID).stream()
                .mapKeyValue(usersAgencyRepository::getShowAgencyContacts)
                .flatMapToEntry(Function.identity())
                .toMap();
    }

    /**
     * Возвращает для пользователей представляющих агенства флаг - показывать ли контакты агентства клиенту,
     * по умолчанию — false
     */
    public Map<Long, Boolean> getUsersAgencyShowAgencyContactsWithDefaultFalse(Collection<Long> uids) {
        Map<Long, Boolean> userIdToShowAgencyContact = getUsersAgencyShowAgencyContacts(uids);
        uids.forEach(uid -> {
            userIdToShowAgencyContact.putIfAbsent(uid, false);
        });
        return userIdToShowAgencyContact;
    }

    /**
     * Возвращает для пользователя представляющего агенства флаг - возможность переносить деньги
     */
    public boolean getUserAgencyDisallowMoneyTransfer(Long userId) {
        return getUsersAgencyDisallowMoneyTransfer(singleton(userId))
                .getOrDefault(userId, false);
    }

    /**
     * Возвращает флаг запрета оплаты для пользователя представляющего агенство
     */
    public boolean getUserAgencyIsNoPay(Long userId) {
        return getUsersAgencyIsNoPay(singleton(userId))
                .getOrDefault(userId, false);
    }

    @Nullable
    public String getSmsPhone(Long uid) {
        return getSmsPhones(List.of(uid)).get(uid);
    }

    /**
     * Получение телефона пользователей для смс уведомлений
     * Если у пользователя нет в паспорте телефона, то его uid'а не будет в результирующей мапе
     *
     * @return Мапа (Uid -> smsPhone); Телефон будет замаскирован: +7905*****00
     */
    public Map<Long, String> getSmsPhones(Collection<Long> uids) {
        return EntryStream.of(getSmsBlackboxPhoneByUid(uids))
                .mapValues(blackboxPhone -> blackboxPhone.getMaskedE164Number().getOrNull())
                .nonNullValues()
                .toImmutableMap();
    }

    @Nullable
    public BlackboxPhone getSmsBlackboxPhone(Long uid) {
        return getSmsBlackboxPhoneByUid(singletonList(uid)).get(uid);
    }

    public List<BlackboxPhone> getAllBlackboxPhones(Long uid) {
        Map<Long, List<BlackboxPhone>> allBlackboxPhonesByUid = getAllBlackboxPhonesByUid(List.of(uid));
        return allBlackboxPhonesByUid.getOrDefault(uid, emptyList());
    }

    public Map<Long, List<BlackboxPhone>> getAllBlackboxPhonesByUid(Collection<Long> uids) {
        if (uids.isEmpty()) {
            return emptyMap();
        }

        List<PassportUid> passportUids = mapList(uids, PassportUid::new);
        Map<PassportUid, List<BlackboxPhone>> result =
                new BlackboxGetPhonesQuery(blackboxClient, tvmIntegration, environmentType, passportUids).call();

        return EntryStream.of(result)
                .mapKeys(PassportUid::getUid)
                .toMap();
    }

    @Nullable
    public Long getSmsPhoneId(Long uid) {
        BlackboxPhone smsBlackboxPhone = getSmsBlackboxPhoneByUid(singletonList(uid)).get(uid);

        return smsBlackboxPhone != null ? smsBlackboxPhone.getPhoneId() : null;
    }

    private Map<Long, BlackboxPhone> getSmsBlackboxPhoneByUid(Collection<Long> uids) {
        List<PassportUid> passportUids = mapList(uids, PassportUid::new);
        Map<PassportUid, Optional<BlackboxPhone>> result = new BlackboxGetPhoneQuery(blackboxClient, tvmIntegration,
                environmentType, passportUids)
                .call();

        return EntryStream.of(result)
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .mapKeys(PassportUid::getUid)
                .toImmutableMap();
    }

    @Nullable
    public UidAndClientId getAgencyUidAndClientId(Long operatorUid,
                                                  @Nullable Long agencyUserId,
                                                  @Nullable Long agencyClientId) {
        if (rbacService.getUidRole(operatorUid) == RbacRole.AGENCY) {
            Long operatorClientId = shardHelper.getClientIdByUid(operatorUid);

            UidAndClientId agencyOperator = UidAndClientId.of(operatorUid, ClientId.fromLong(operatorClientId));
            // если у клиента назначен ограниченный представитель агентства,
            // то создаем кампанию для этого ограниченного представителя
            List<Long> representativeUids = massGetUidsByClientIds(Set.of(agencyOperator.getClientId()))
                    .get(agencyOperator.getClientId());
            List<Representative> agencyLimitedRepresentatives =
                    rbacService.getAgencyLimitedRepresentatives(representativeUids);
            return agencyLimitedRepresentatives.stream()
                    .filter(representative -> Objects.equals(agencyUserId, representative.getUserId()))
                    .findFirst()
                    .map(rep -> UidAndClientId.of(rep.getUserId(), rep.getClientId()))
                    .orElse(agencyOperator);
        } else if (agencyUserId != null && agencyClientId != null) {
            return UidAndClientId.of(agencyUserId, ClientId.fromLong(agencyClientId));
        }

        return null;
    }

    /**
     * Удаление представителя клиента.
     * Сейчас вызывается только при удалении представителя из Connect'а.
     * В отличие от перловой реализации, не сохраняет данные пользователя в ppc.clients.deleted_reps, т.к.
     * при управлении представителями через Connect это не требуется.
     *
     * @param user
     * @return
     */
    public Result<User> dropClientRep(@Nullable User user) {
        if (user == null) {
            return Result.successful(user);
        }

        ValidationResult<User, Defect> vr = preValidateUserForDelete(user);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        //Проверяем, что представитель не привязан в Балансе к другому клиенту
        Optional<ClientId> balanceClient = balanceService.findClientIdByUid(user.getUid());
        if (balanceClient.isPresent() && !balanceClient.get().equals(user.getClientId())) {
            return error(user, balanceUserAssociatedWithAnotherClient());
        }

        int shard = shardHelper.getShardByClientId(user.getClientId());
        //Достаем автоплатежи, плательщиком по которым установлен удаляемый пользователь и проверяем,
        //что он не привязан к активным автоплатежам
        Map<Long, Boolean> autoPayIsActiveFlagByWalletCid = walletRepository
                .findWalletsWithSameAutoPayPayerUid(shard, user.getUid(), user.getClientId());
        if (autoPayIsActiveFlagByWalletCid.containsValue(true)) {
            return error(user, userHasActiveAutoPay());
        }

        //Удаляем привязку пользователя к клиенту в Балансе
        if (balanceClient.isPresent()) {
            balanceService.removeUserClientAssociation(user);
        }

        //Удаляем неактивные автоплатежи, привязанные к представителю.
        //Что среди них нет активных мы проверили выше.
        walletRepository.deleteInactiveAutoPays(shard, autoPayIsActiveFlagByWalletCid.keySet());

        deleteUserFromDatabase(shard, user.getUid(), user.getChiefUid());

        //Обновляем доступ к счетчикам турболендингов;
        turboLandingService.refreshTurbolandingMetrikaGrants(user.getChiefUid(), user.getClientId());


        return Result.successful(user);
    }

    private Result<User> error(User user, Defect defect) {
        return Result.broken(ValidationResult.failed(user, defect));
    }

    private void deleteUserFromDatabase(int shard, Long uid, Long chiefUid) {
        //В таблицах, в которых теоретически могут быть привязки к удаляемому uid,
        // меняем на привязку к осеновному представителю клиента
        updateRelatedTables(shard, uid, chiefUid);

        //Удаляем пользователя из базы
        userRepository.deleteUserFromPpc(shard, uid);
        userRepository.deleteUserFromPpcDict(uid);

        rbacService.clearCaches();
    }

    private ValidationResult<User, Defect> preValidateUserForDelete(User user) {
        if (user.getUid().equals(user.getChiefUid())) {
            return ValidationResult.failed(user, chiefDeletionProhibited());
        }

        return ValidationResult.success(user);
    }

    private void updateRelatedTables(int shard, Long sourceUid, Long targetUid) {
        vcardRepository.updateVcardUids(shard, singletonList(sourceUid), targetUid);
        vcardRepository.updateOrgDetailsUids(shard, singletonList(sourceUid), targetUid);
        campaignRepository.updateCampOwners(shard, singletonList(sourceUid), targetUid);

        userRepository.updateUserRelatedTables(shard, singletonList(sourceUid), targetUid);
    }

    public Map<Long, List<Long>> getCampaignIdsForStopping(Collection<ClientId> usersClientIds) {
        Map<Long, List<Long>> result = new HashMap<>();
        shardHelper.groupByShard(usersClientIds, ShardKey.CLIENT_ID)
                .forEach((shard, clientIds) -> result.putAll(campaignRepository.getCampaignIdsForStopping(shard,
                        mapList(clientIds, ClientId::asLong))));
        return result;
    }

    public void unblockUser(ClientId clientId, Long uid) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        userRepository.unblockUser(shard, uid);
    }

    public void blockUser(ClientId clientId, Long uid) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        userRepository.blockUser(shard, uid);
    }

    public void unblockUserByClientIds(Collection<ClientId> clientIds) {
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .forEach((shard, ids) -> userRepository.unblockUserByClientIds(shard, mapList(ids, ClientId::asLong)));
    }

    public void blockUserByClientIds(Collection<ClientId> clientIds,
                                     UsersBlockReasonType blockReasonType,
                                     @Nullable String blockComment) {
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .forEach((shard, ids) -> userRepository.blockUserByClientIds(
                        shard, mapList(ids, ClientId::asLong), blockReasonType, blockComment));
    }

    /**
     * Выставляет представителю клиента флаг isOfferAccepted.
     * Предполагается, что права на изменения этого флага проверяются в вызывающем коде.
     * С юридической точки зрения важно, чтобы флаг принятия оферы мог поставить только лично сам представитель клиента
     * и НЕ могли за него поставить менеджеры или супер-пользователи.
     */
    public void setOfferAccepted(Long uid) {
        User user = getUser(uid);
        checkNotNull(user, "User not found");
        ModelChanges<User> changes = new ModelChanges<>(uid, User.class);
        changes.process(true, User.IS_OFFER_ACCEPTED);
        AppliedChanges<User> userAppliedChanges = changes.applyTo(user);
        update(userAppliedChanges);
    }

    /**
     * Установить флаг разработчика
     *
     * @param uid  uid пользователя
     * @param flag флаг для установки
     */
    public void setDeveloperFlag(Long uid, boolean flag) {
        User user = getUser(uid);
        checkNotNull(user, "User not found");
        ModelChanges<User> changes = new ModelChanges<>(uid, User.class);
        changes.process(flag, User.DEVELOPER);
        AppliedChanges<User> userAppliedChanges = changes.applyTo(user);
        update(userAppliedChanges);
    }

    /**
     * @param uid Идентификатор представителя клиента.
     * @return При включенной фиче MODERATION_OFFER_ENABLED_FOR_DNA возвращает эвристику - принял или нет пользователь
     * оферту.
     * Исходим из предположения, что пользователь стопудово принял оферту при заведении клиента,
     * если это самоходный клиент, а сам пользователь единственный представитель и других представителей никогда не
     * было.
     * <p>
     * Если эвристика не сработала, то проверяет значение из базы. Эвристику пердполагается удалить из кода, когда её
     * логика будет перенесена в код заведения новых клиентов и значение поля ppc.users_options.is_offer_accepted
     * будет правильно заполняться сразу при заведении клиента.
     */
    public boolean getIsOfferAccepted(@Nonnull Long uid) {
        return Optional.of(uid)
                .map(shardHelper::getClientIdByUid)
                .map(ClientId::fromLong)
                .filter(id1 -> featureService.isEnabledForClientId(id1, FeatureName.MODERATION_OFFER_ENABLED_FOR_DNA))
                .filter(this::isSamohod)
                .filter(this::isOnlyOneRep)
                .map(l -> true)
                .orElse(isOfferAccepted(uid));
    }

    private boolean isOnlyOneRep(@Nonnull ClientId clientId) {
        List<Long> uids = shardHelper.getUidsByClientId(clientId.asLong());
        if (uids.size() != 1) {
            return false;
        }
        int shard = shardHelper.getShardByClientId(clientId);
        Client client = clientRepository.get(shard, singleton(clientId)).get(0);
        return Optional.ofNullable(client.getDeletedReps())
                .filter(l -> l.contains("{"))
                .isEmpty();
    }

    private boolean isSamohod(@Nonnull ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<Client> clients = clientRepository.get(shard, singleton(clientId));
        return clients.get(0).getAgencyClientId() == null;
    }

    /**
     * Заменит метод getIsOfferAccepted после переходного периода, когда логика эвристик будет перенесена в код
     * заведения новых клиентов.
     */
    private boolean isOfferAccepted(Long uid) {
        int shard = shardHelper.getShardByUserId(uid);
        List<User> users = userRepository.fetchByUids(shard, singleton(uid));
        return users.get(0).getIsOfferAccepted();
    }

    public ValidationResult<List<User>, Defect> validateUsersToBlock(List<User> users,
                                                                     Map<Long, List<Long>> campaignIdsByClientId,
                                                                     boolean checkCampsCount) {
        return blockUserValidationService.validateCampaignsToBlock(users, campaignIdsByClientId, checkCampsCount);
    }

    public ValidationResult<List<User>, Defect> blockUsers(User operator,
                                                           List<User> users,
                                                           Map<Long, List<Long>> campaignIdsByClientId,
                                                           UsersBlockReasonType blockReasonType,
                                                           @Nullable String blockComment,
                                                           boolean checkCampsCount) {

        ValidationResult<List<User>, Defect> vr = blockUserValidationService.validateCampaignsToBlock(users,
                campaignIdsByClientId, checkCampsCount);

        if (vr.hasAnyErrors()) {
            return vr;
        }
        var clientIdUsersMap = users.stream().collect(groupingBy(User::getClientId));

        for (ClientId clientId : clientIdUsersMap.keySet()) {
            int shard = shardHelper.getShardByClientIdStrictly(clientId);
            List<Long> campaignIds = campaignIdsByClientId.getOrDefault(clientId.asLong(), emptyList());
            autopayService.removeAutopay(shard, clientId);
            campaignRepository.stopCampaigns(shard, campaignIds);
            var campaignEvents = mapList(campaignIds, cid -> campaignStopped(operator.getUid(),
                    nvl((clientIdUsersMap.get(clientId).get(0)).getChiefUid(),
                            userRepository.getChiefUidByClientId(shard, clientId.asLong())), cid));
            mailNotificationEventService.queueEvents(operator.getUid(), clientId, campaignEvents);
        }

        blockUserByClientIds(clientIdUsersMap.keySet(), blockReasonType, blockComment);

        sendBlockedClientNotifications(users, notificationService);

        return vr;
    }

    public ValidationResult<List<User>, Defect> validateUsersToUnblock(List<User> users) {
        return blockUserValidationService.validateCanUnblock(users);
    }

    public ValidationResult<List<User>, Defect> unblockUsers(List<User> users) {
        var vr = validateUsersToUnblock(users);
        if (vr.hasAnyErrors()) {
            return vr;
        }
        unblockUserByClientIds(mapList(users, User::getClientId));
        return vr;
    }

    /**
     * Получить ФИО пользователей по их UID
     *
     * @param uids uid-ы пользователей
     */
    public Map<Long, String> getUserFiosByUids(Collection<Long> uids) {
        return shardHelper.groupByShard(uids, ShardKey.UID)
                .stream()
                .mapKeyValue(userRepository::getFiosByUids)
                .flatMapToEntry(Function.identity())
                .toMap();
    }
}
