package ru.yandex.direct.core.entity.user.service;

import java.util.Collection;
import java.util.List;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.entity.user.service.validation.UpdateUserValidationService;
import ru.yandex.direct.core.entity.user.service.validation.UserValidationService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class UserUpdateOperation extends SimpleAbstractUpdateOperation<User, Long> {
    private final UserRepository userRepository;
    private final UpdateUserValidationService updateUserValidationService;
    private final UserValidationService userValidationService;
    private final UserService userService;
    private final ShardHelper shardHelper;
    private final long operatorUid;

    public UserUpdateOperation(Applicability applicability, UserRepository userRepository,
                               UpdateUserValidationService updateUserValidationService,
                               UserValidationService userValidationService, UserService userService, ShardHelper shardHelper,
                               List<ModelChanges<User>> modelChanges, long operatorUid) {
        super(applicability, modelChanges, id -> new User().withUid(id));
        this.userRepository = userRepository;
        this.updateUserValidationService = updateUserValidationService;
        this.userValidationService = userValidationService;
        this.userService = userService;
        this.shardHelper = shardHelper;
        this.operatorUid = operatorUid;
    }

    @Override
    protected ValidationResult<List<ModelChanges<User>>, Defect> validateModelChanges(
            List<ModelChanges<User>> modelChanges) {
        return updateUserValidationService.preValidate(operatorUid, modelChanges);
    }

    @Override
    protected Collection<User> getModels(Collection<Long> ids) {
        return userService.massGetUser(ids);
    }

    @Override
    protected ValidationResult<List<User>, Defect> validateAppliedChanges(
            ValidationResult<List<User>, Defect> validationResult) {
        return userValidationService.validate(validationResult);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<User>> validAppliedChanges) {
        shardHelper.groupByShard(validAppliedChanges, ShardKey.UID, ac -> ac.getModel().getId())
                .forEach((shard, changesForShard) -> userRepository.update(shard, changesForShard));
        return mapList(validAppliedChanges, a -> a.getModel().getId());
    }
}
