package ru.yandex.direct.core.entity.user.service.validation;

import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.core.service.integration.balance.model.PaymentMethodType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.frequency;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.balanceUserAssociatedWithAnotherClient;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userHasCardPaymentMethod;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userHasNotValidEmail;
import static ru.yandex.direct.core.entity.user.service.validation.UserDefects.userHasNotValidName;
import static ru.yandex.direct.core.validation.constraints.Constraints.validLogin;
import static ru.yandex.direct.utils.CommonUtils.notEquals;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validEmail;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedElement;

@ParametersAreNonnullByDefault
@Service
public class AddUserValidationService {

    private final UserRepository userRepository;
    private final BalanceService balanceService;

    @Autowired
    public AddUserValidationService(UserRepository userRepository, BalanceService balanceService) {
        this.userRepository = userRepository;
        this.balanceService = balanceService;
    }

    public ValidationResult<List<User>, Defect> validate(List<User> users) {
        var validationInfo = new ValidationInfoContainer(users).init();

        ListValidationBuilder<User, Defect> lvb = ListValidationBuilder.of(users);
        lvb.checkEachBy(getUserDefectValidator(validationInfo), When.isValid());
        return lvb.getResult();
    }

    private Validator<User, Defect> getUserDefectValidator(ValidationInfoContainer validationInfo) {
        return user -> validateUser(user, validationInfo);
    }

    private ValidationResult<User, Defect> validateUser(@Nullable User user, ValidationInfoContainer validationInfo) {
        /*возвращаем UserDefectIds.Gen.USER_NOT_FOUND т.к. чтобы здесь оказался null выше по стеку пользователь
        должен был
        не найтись во внешних системах*/
        if (user == null) {
            return ValidationResult.failed(user, UserDefects.userNotFound());
        }

        ModelItemValidationBuilder<User> vb = ModelItemValidationBuilder.of(user);
        vb.item(User.UID)
                .check(notNull())
                .check(validId())
                .check(notInSet(validationInfo.getDuplicatedUids()), duplicatedElement())
                .check(notInSet(validationInfo.getExistingUids()), CommonDefects.inconsistentStateAlreadyExists())
                .check(notInSet(validationInfo.getHasOtherClientInBalanceUids()),
                        balanceUserAssociatedWithAnotherClient())
                .check(notInSet(validationInfo.getHasCardUids()), userHasCardPaymentMethod());
        vb.item(User.LOGIN)
                .check(notNull())
                .check(validLogin())
                .check(notInSet(validationInfo.getExistingLogins()), CommonDefects.inconsistentStateAlreadyExists());
        vb.item(User.FIO)
                .check(notNull(), userHasNotValidName())
                .check(notBlank(), userHasNotValidName());
        vb.item(User.EMAIL)
                .check(notNull(), userHasNotValidEmail())
                .check(validEmail(), userHasNotValidEmail());
        return vb.getResult();
    }

    public class ValidationInfoContainer {
        private final List<User> users;
        private Set<Long> existingUids;
        private Set<String> existingLogins;
        private Set<Long> duplicatedUids;
        private Set<Long> hasOtherClientInBalanceUids;
        private Set<Long> hasCardUids;

        ValidationInfoContainer(List<User> users) {
            this.users = users;
        }

        Set<Long> getExistingUids() {
            return existingUids;
        }

        Set<String> getExistingLogins() {
            return existingLogins;
        }

        Set<Long> getDuplicatedUids() {
            return duplicatedUids;
        }

        Set<Long> getHasOtherClientInBalanceUids() {
            return hasOtherClientInBalanceUids;
        }

        Set<Long> getHasCardUids() {
            return hasCardUids;
        }

        public ValidationInfoContainer init() {
            List<Long> uids = StreamEx.of(users)
                    .filter(Objects::nonNull)
                    .map(User::getUid)
                    .filter(Objects::nonNull)
                    .toList();
            initExistingUids(uids);
            initExistingLogins();
            initDuplicatedUids(uids);
            initAlreadyAssociatedInBalance(
                    StreamEx.of(users)
                            .filter(Objects::nonNull)
                            .toList()
            );
            initHasCardUids(uids);
            return this;
        }

        private void initExistingUids(List<Long> uids) {
            existingUids = userRepository.usersExistByUids(uids);
        }

        private void initDuplicatedUids(List<Long> uids) {
            duplicatedUids = StreamEx.of(uids)
                    .filter(i -> frequency(uids, i) > 1)
                    .toSet();
        }

        private void initExistingLogins() {
            List<String> logins = StreamEx.of(users)
                    .filter(Objects::nonNull)
                    .map(User::getLogin)
                    .filter(Objects::nonNull)
                    .toList();
            existingLogins = userRepository.usersExistByLogins(logins);
        }

        private void initAlreadyAssociatedInBalance(List<User> users) {
            hasOtherClientInBalanceUids = new HashSet<>();
            for (User user : users) {
                Long uid = user.getUid();
                Optional<ClientId> returnedClientId = balanceService.findClientIdByUid(uid);
                /* Проверям, что у пользователя нет связанных клиентов в балансе или он тот же, к которому
                собираемся привязать пользователя.
                В Балансе операция установления связи между пользователем и клиентом - идемпотентна.*/
                if (returnedClientId.isPresent() && notEquals(returnedClientId.get(), user.getClientId())) {
                    hasOtherClientInBalanceUids.add(uid);
                }
            }
        }

        private void initHasCardUids(List<Long> uids) {
            hasCardUids = new HashSet<>();
            for (Long uid : uids) {
                Set<PaymentMethodType> userPaymentMethodTypeTypes = balanceService.getUserPaymentMethodTypes(uid);
                if (userPaymentMethodTypeTypes.contains(PaymentMethodType.CARD)) {
                    hasCardUids.add(uid);
                }
            }
        }

    }

}
