package ru.yandex.direct.core.entity.user.service.validation;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.client.model.ClientsOptions;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.validation.defects.RightsDefects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static ru.yandex.direct.core.entity.client.Constants.CANT_UNBLOCK_CLIENT_FLAG;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Component
@ParametersAreNonnullByDefault
public class BlockUserValidationService {
    private static final Logger logger = LoggerFactory.getLogger(BlockUserValidationService.class);

    public final static int MAX_CAMPAIGNS_COUNT_FOR_STOPPING = 200;

    /**
     * Таблица, где хранятся данные по тирам
     */
    public static final String CLIENT_TIERS_TABLE_HAHN = "//home/comdep-analytics/public/client_tiers/fact/latest";
    /**
     * Тиры, которые считаем важными (согласовано с vsuvorov@)
     */
    private static final Set<String> IMPORTANT_TIERS = Set.of("Tier 0", "LE", "NBD", "INT LE");
    /**
     * Индустрии, которые считаем важными
     */
    private static final Set<String> IMPORTANT_INDUSTRIES = Set.of("ICO");

    private final ClientService clientService;
    private final YtProvider ytProvider;

    @Autowired
    public BlockUserValidationService(ClientService clientService,
                                      YtProvider ytProvider) {
        this.clientService = clientService;
        this.ytProvider = ytProvider;
    }

    public ValidationResult<List<User>, Defect> validateAccess(List<User> users) {
        return ListValidationBuilder.of(users, Defect.class)
                .checkEachBy(this::validateUserAccess)
                .getResult();
    }

    private ValidationResult<User, Defect> validateUserAccess(User user) {
        return ModelItemValidationBuilder.of(user)
                .check(notNull(), CommonDefects.objectNotFound())
                .check(fromPredicate(u -> hasOneOfRoles(u, CLIENT, AGENCY), RightsDefects.noRights()))
                .getResult();
    }

    public ValidationResult<List<User>, Defect> validateCampaignsToBlock(List<User> users,
                                                                         Map<Long, List<Long>> usersToCampaignId,
                                                                         boolean checkCampsCount) {
        ListValidationBuilder<User, Defect> vr = ListValidationBuilder.of(users, Defect.class);
        if (checkCampsCount) {
            vr.checkEachBy(user -> validateCampaignsCount(user, usersToCampaignId));
        }
        return vr.getResult();
    }

    private ValidationResult<User, Defect> validateCampaignsCount(User user,
                                                                  Map<Long, List<Long>> campaignIdsByClientId) {
        ModelItemValidationBuilder<User> vb = ModelItemValidationBuilder.of(user);
        vb.check(fromPredicate(u -> user.getClientCreateDate().isAfter(LocalDateTime.now().minusMonths(3)) ||
                        campaignIdsByClientId.getOrDefault(u.getClientId().asLong(), Collections.emptyList()).size() <=
                                MAX_CAMPAIGNS_COUNT_FOR_STOPPING,
                CollectionDefects.maxCollectionSize(MAX_CAMPAIGNS_COUNT_FOR_STOPPING)));
        return vb.getResult();
    }

    public ValidationResult<List<User>, Defect> validateCanUnblock(List<User> usersToValidate) {
        List<Long> clientIds = mapList(usersToValidate, user -> user.getClientId().asLong());
        List<ClientsOptions> clientsOptions = clientService.massGetClientOptions(clientIds);
        List<Long> clientIdsCantUnblock = filterAndMapList(clientsOptions,
                co -> co.getClientFlags() != null && co.getClientFlags().contains(CANT_UNBLOCK_CLIENT_FLAG),
                ClientsOptions::getId);
        return ListValidationBuilder.of(usersToValidate, Defect.class)
                .checkEachBy(user -> validateUserCanBeUnblocked(user, clientIdsCantUnblock))
                .getResult();
    }

    private ValidationResult<User, Defect> validateUserCanBeUnblocked(User user, List<Long> clientIdsCantUnblock) {
        ModelItemValidationBuilder<User> vb = ModelItemValidationBuilder.of(user);
        vb.check(fromPredicate(u -> !clientIdsCantUnblock.contains(u.getClientId().asLong()),
                UserDefects.userCantBeUnblocked()));
        return vb.getResult();
    }

    /**
     * Среди переданных clientIds ищет важных клиентов, которых нельзя банить автоматикой без флага --force
     * Таблица есть только на кластере Hahn, поэтому при недоступности кластера функция не будет работать
     */
    public Set<ClientId> getImportantClients(Collection<ClientId> clientIds) {
        Yt yt = ytProvider.getOperator(YtCluster.HAHN).getYt();
        var ranges = StreamEx.of(clientIds)
                .map(ClientId::asLong)
                .map(clientId -> Pair.of(clientId, clientId + 1))
                .toList();
        Set<ClientId> importantClients = new HashSet<>();
        YtUtils.readTableByKeyRanges(yt, YPath.simple(CLIENT_TIERS_TABLE_HAHN),
                ranges,
                row -> {
                    long clientId = row.getOrThrow("client_id").longValue();
                    String tier = YtUtils.getYsonField(row, "curr_counterparty_crm_tier", "name");
                    String industry = YtUtils.getYsonField(row, "curr_counterparty_industry", "industry");
                    if (tier == null) {
                        logger.info("Client {} has no tier detected", clientId);
                    }
                    if (industry == null) {
                        logger.info("Client {} has no industry detected", clientId);
                    }
                    if (tier != null && IMPORTANT_TIERS.contains(tier)
                            || industry != null && IMPORTANT_INDUSTRIES.contains(industry)) {
                        logger.info("Found important client {} (tier {}, industry {})", clientId, tier, industry);
                        importantClients.add(ClientId.fromLong(clientId));
                    }
                });
        return importantClients;
    }
}
