package ru.yandex.direct.core.entity.user.service.validation;

import java.util.List;
import java.util.regex.Pattern;

import javax.annotation.Nonnull;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.inside.passport.blackbox2.protocol.response.BlackboxPhone;

import static ru.yandex.direct.core.validation.defects.Defects.phoneMustBeVerified;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.admissibleChars;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty;
import static ru.yandex.direct.validation.constraint.StringConstraints.validEmail;

@Component
public class UserValidationService {
    private static final Pattern VALID_PHONE_CHARS = Pattern.compile("^[-+ 0-9)(]+$");
    private static final Pattern AT_LEAST_TWO_DIGIT = Pattern.compile("\\D*\\d\\D*\\d.*");

    public final UserService userService;

    public UserValidationService(UserService userService) {
        this.userService = userService;
    }

    public ValidationResult<List<User>, Defect> validate(
            ValidationResult<List<User>, Defect> validationResult) {
        return new ListValidationBuilder<>(validationResult)
                .checkEachBy(this::validateUser, When.isValid())
                .getResult();
    }

    /**
     * Валидировать применённые изменения
     * <p>
     * Проверяет не нарушены ли требования к модели клиента после применения измененений
     *
     * @param appliedChanges применённые изменения
     * @return результат валидации клиента
     */
    @Nonnull
    public ValidationResult<User, Defect> validate(AppliedChanges<User> appliedChanges) {
        return validateUser(appliedChanges.getModel());
    }

    @Nonnull
    public ValidationResult<User, Defect> validateUser(User user) {
        ModelItemValidationBuilder<User> vb = ModelItemValidationBuilder.of(user);
        vb.item(User.EMAIL)
                .check(notNull())
                .check(notEmpty())
                .check(maxStringLength(255))
                .check(validEmail(), When.isValid());
        vb.item(User.FIO)
                .check(notNull())
                .check(notEmpty())
                .check(maxStringLength(255))
                .check(admissibleChars(), When.isValid());
        vb.item(User.PHONE)
                .check(notEmpty())
                .check(maxStringLength(255));
        vb.item(User.VERIFIED_PHONE_ID)
                .check(validId())
                .check(phoneIsVerified(user), phoneMustBeVerified(), When.notNull());

        return vb.getResult();
    }

    // TODO(dimitrovsd): нужно ходить за телефонами для всех пользователей сразу, а не по одному
    private Constraint<Long, Defect> phoneIsVerified(User user) {
        return inSet(() -> {
            List<BlackboxPhone> blackboxPhones = userService.getAllBlackboxPhones(user.getUid());

            return filterAndMapToSet(blackboxPhones,
                    blackboxPhone -> blackboxPhone.getIsConfirmedNumber().getOrElse(false), BlackboxPhone::getPhoneId);
        });
    }
}
