package ru.yandex.direct.core.entity.user.utils;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.campaign.StopCampLogReport;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.container.BlockedClientNotification;
import ru.yandex.direct.core.entity.user.BlockUserLogRecord;
import ru.yandex.direct.core.entity.user.UnblockUserLogRecord;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isAgency;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.isLimitedSupport;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class BlockedUserUtil {

    private static final String LOGGER_NAME = "PPCLOG_CMD.log";
    private static final Logger changeLogger = LoggerFactory.getLogger(LOGGER_NAME);
    /**
     * Проверка оператора на заблокированность для web-api.
     * Используется так же для расчета флагов доступности объектов на гриде
     * Логика взята из перла: https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DoCmd.pm?rev=6676464#L972
     *
     * @param methodAllowedForBlockedOperator - доступен ли вызываемый контроллер/резолвер заблокированному оператору
     */
    public static boolean checkOperatorIsBlocked(User operator, boolean methodAllowedForBlockedOperator) {
        if (!nvl(operator.getStatusBlocked(), false)) {
            return false;
        }

        return (operator.getRole().isInternal() || isLimitedSupport(operator))
                || (isAgency(operator) && !methodAllowedForBlockedOperator);
    }

    /**
     * Проверка пользователя на заблокированность для web-api
     * Используется так же для расчета флагов доступности объектов на гриде
     * Логика взята из перла: https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/DoCmd.pm?rev=6676464#L972
     *
     * @param methodAllowedForBlockedUser - доступен ли вызываемый контроллер/резолвер заблокированному пользователю
     */
    public static boolean checkSubjectUserIsBlocked(User subjectUser, boolean methodAllowedForBlockedUser,
                                                    User operator) {
        if (!nvl(subjectUser.getStatusBlocked(), false)
                || hasOneOfRoles(operator, RbacRole.SUPER, RbacRole.SUPERREADER, RbacRole.SUPPORT, RbacRole.PLACER,
                RbacRole.INTERNAL_AD_ADMIN, RbacRole.INTERNAL_AD_MANAGER, RbacRole.INTERNAL_AD_SUPERREADER,
                RbacRole.LIMITED_SUPPORT)) {
            return false;
        }

        return !methodAllowedForBlockedUser;
    }

    /**
     * Проверяет доступно ли редактирование объектов клиента
     * Считаем, что редактирование недоступно заблокированному оператору или пользователю
     */
    public static boolean checkClientIsBlockedForEdit(User subjectUser, User operator) {
        return checkOperatorIsBlocked(operator, false) || checkSubjectUserIsBlocked(subjectUser, false, operator);
    }

    public static void logUserBlock(User user, User operator, List<Long> campaignIds) {
        var blockUserLogRecord = getBlockUserLogRecord(user.getUid(), operator.getUid(), campaignIds);
        changeLogger.info("{} {}", blockUserLogRecord.getPrefixLogTime(), JsonUtils.toJson(blockUserLogRecord));
        for (Long cid : campaignIds) {
            var blockUserCampaignsLogRecord = getStopCampLogReport(user.getUid(), operator.getUid(), cid);
            changeLogger.info("{} {}", blockUserLogRecord.getPrefixLogTime(), JsonUtils.toJson(blockUserCampaignsLogRecord));
        }
    }

    public static void logUserUnblock(User user, User operator) {
        var unblockUserLogRecord = getUnblockUserLogRecord(user.getUid(), operator.getUid());
        changeLogger.info("{} {}", unblockUserLogRecord.getPrefixLogTime(), JsonUtils.toJson(unblockUserLogRecord));
    }

    public static void sendBlockedClientNotifications(List<User> users, NotificationService notificationService) {
        StreamEx.of(users)
                .filterBy(User::getRole, RbacRole.CLIENT)
                .map(User::getClientId)
                .distinct()
                .forEach(clientId -> {
                    var notification = new BlockedClientNotification()
                            .withClientId(clientId);

                    notificationService.addNotification(notification);
                });
    }

    private static BlockUserLogRecord getBlockUserLogRecord(Long userId, Long operatorUid, List<Long> campaignIds) {
        var blockUserLogRecord = new BlockUserLogRecord(Trace.current().getTraceId());
        blockUserLogRecord.setPath(BlockUserLogRecord.LOG_BLOCK_USER_CHANGE_CMD_NAME);
        blockUserLogRecord.setOperatorId(operatorUid);

        Map<String, Object> params = blockUserLogRecord.getParam();
        params.put("cluid", userId);
        params.put("stopped_camps", JsonUtils.toJson(campaignIds));
        return blockUserLogRecord;
    }

    private static StopCampLogReport getStopCampLogReport(Long userId, Long operatorUid, Long cid) {
        var stopCampLogReport = new StopCampLogReport(Trace.current().getTraceId());
        stopCampLogReport.setPath(StopCampLogReport.LOG_STOP_CAMP_CMD_NAME);
        stopCampLogReport.setOperatorId(operatorUid);
        stopCampLogReport.setCids(singletonList(cid));

        Map<String, Object> params = stopCampLogReport.getParam();
        params.put("cluid", userId);
        return stopCampLogReport;
    }

    private static UnblockUserLogRecord getUnblockUserLogRecord(Long userId, Long operatorUid) {
        var unblockUserLogRecord = new UnblockUserLogRecord(Trace.current().getTraceId());
        unblockUserLogRecord.setPath(UnblockUserLogRecord.LOG_UNBLOCK_USER_CHANGE_CMD_NAME);
        unblockUserLogRecord.setOperatorId(operatorUid);

        Map<String, Object> params = unblockUserLogRecord.getParam();
        params.put("cluid", userId);
        return unblockUserLogRecord;
    }

}
