package ru.yandex.direct.core.entity.user.validator;

import java.util.regex.Pattern;

import org.springframework.stereotype.Component;

import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;

import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.notContains;

/**
 * Валидатор login-а (См. TextTools.pm::validate_login для не lite режима)
 * <p>
 * Проверяет правильность написания логина. Логин должен начинаться с буквы, и заканчиваться буквой, либо цифрой
 * Допускается использование `-`(тире). Не допускается повторение подряд точки или тире
 */
@Component
public class LoginValidator implements DefaultValidator<String> {
    static final int MAX_SIZE = 30;

    private static final Pattern LOGIN_PATTERN = Pattern.compile("[a-zA-Z][a-zA-Z0-9\\-.]*[a-zA-Z0-9]");
    // По хорошему надо запрещать и -. и .-, но perl-ый код так не делает,
    // поэтому для совместимости оставим также
    private static final Pattern DBLDASH_OR_DBLDOT_PATTERN = Pattern.compile("--|\\.\\.");

    @Override
    public ValidationResult<String, Defect> apply(String login) {
        return ItemValidationBuilder.<String, Defect>of(login)
                .check(notBlank())
                .check(maxStringLength(MAX_SIZE))
                .check(matchPattern(LOGIN_PATTERN), When.isValid())
                .check(notContains(DBLDASH_OR_DBLDOT_PATTERN), When.isValid())
                .getResult();
    }
}
