package ru.yandex.direct.core.entity.userphone;

import java.time.Duration;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.redis.LettuceStorage;

import static java.lang.String.format;

/**
 * Storage для хранения количества вызовов методов для подтверждения телефона.
 */
@Component
public class UserPhoneStorage {

    private static final String KEY_FORMAT = "user-phone:%s:%s";

    private static final String CALLS_COUNT_FIELD = "calls-count";

    private static final int SLIDING_WINDOW_DURATION_IN_SECONDS = (int) Duration.ofDays(1).toSeconds();

    private final LettuceStorage lettuceStorage;
    private final String keyPrefix;

    @Autowired
    public UserPhoneStorage(LettuceStorage lettuceStorage,
                            @Value("${user-phone.key_prefix:}") String keyPrefix) {
        this.lettuceStorage = lettuceStorage;
        this.keyPrefix = keyPrefix;
    }

    public int incrementAndGetCallsCount(Long operatorUid) {
        String redisKey = getRedisKey(operatorUid, CALLS_COUNT_FIELD);

        int callCount = (int) lettuceStorage.incrementAndGet(redisKey);
        if (callCount == 1) {
            lettuceStorage.expire(redisKey, SLIDING_WINDOW_DURATION_IN_SECONDS);
        }

        return callCount;
    }

    private String getRedisKey(Long operatorUid, String fieldName) {
        return keyPrefix + format(KEY_FORMAT, operatorUid, fieldName);
    }
}
