package ru.yandex.direct.core.entity.userssegments.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.model.AdShowType;
import ru.yandex.direct.core.entity.adgroup.model.ExternalAudienceStatus;
import ru.yandex.direct.core.entity.adgroup.model.InternalStatus;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.dbschema.ppc.enums.VideoSegmentGoalsExternalAudienceStatus;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperUtils;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppc.enums.VideoSegmentGoalsExternalAudienceStatus.few_data;
import static ru.yandex.direct.dbschema.ppc.enums.VideoSegmentGoalsExternalAudienceStatus.processed;
import static ru.yandex.direct.dbschema.ppc.enums.VideoSegmentGoalsInternalStatus.new_;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.VideoSegmentGoals.VIDEO_SEGMENT_GOALS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class UsersSegmentRepository {

    private static final Long MAX_ERR_COUNT_FOR_UPDATE_SEGMENT = 50L;

    public static final JooqMapperWithSupplier<UsersSegment> MAPPER = createMapper();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public UsersSegmentRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    private static JooqMapperWithSupplier<UsersSegment> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(UsersSegment::new)
                .map(property(UsersSegment.AD_GROUP_ID, VIDEO_SEGMENT_GOALS.PID))
                .map(property(UsersSegment.TIME_CREATED, VIDEO_SEGMENT_GOALS.TIME_CREATED))
                .map(property(UsersSegment.LAST_SUCCESS_UPDATE_TIME, VIDEO_SEGMENT_GOALS.LAST_SUCCESSFUL_UPDATE_TIME))
                .map(property(UsersSegment.EXTERNAL_AUDIENCE_ID, VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_ID))
                .map(property(UsersSegment.SEGMENT_OWNER_UID, VIDEO_SEGMENT_GOALS.SEGMENT_OWNER_UID))
                .map(property(UsersSegment.ERROR_COUNT, VIDEO_SEGMENT_GOALS.SEGMENT_ERROR_COUNT))
                .map(convertibleProperty(UsersSegment.TYPE, VIDEO_SEGMENT_GOALS.AUDIENCE_TYPE,
                        AdShowType::fromSource, AdShowType::toSource))
                .map(convertibleProperty(UsersSegment.INTERNAL_STATUS, VIDEO_SEGMENT_GOALS.INTERNAL_STATUS,
                        InternalStatus::fromSource, InternalStatus::toSource))
                .map(convertibleProperty(UsersSegment.EXTERNAL_AUDIENCE_STATUS,
                        VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_STATUS,
                        ExternalAudienceStatus::fromSource, ExternalAudienceStatus::toSource))
                .map(booleanProperty(UsersSegment.IS_DISABLED, VIDEO_SEGMENT_GOALS.IS_DISABLED))
                .build();
    }

    public UsersSegment getSegmentByPrimaryKey(int shard, Long adGroupId, AdShowType type) {
        return getSegmentByPrimaryKey(dslContextProvider.ppc(shard), adGroupId, type);
    }

    public UsersSegment getSegmentByPrimaryKey(DSLContext ctx, Long adGroupId, AdShowType type) {
        Record record = ctx
                .select(MAPPER.getFieldsToRead())
                .from(VIDEO_SEGMENT_GOALS)
                .where(VIDEO_SEGMENT_GOALS.PID.eq(adGroupId))
                .and(VIDEO_SEGMENT_GOALS.AUDIENCE_TYPE.eq(AdShowType.toSource(type)))
                .fetchOne();
        if (record == null) {
            return null;
        }
        return MAPPER.fromDb(record);
    }

    public void addSegments(int shard, List<UsersSegment> segments) {
        addSegments(dslContextProvider.ppc(shard), segments);
    }

    public void addSegments(DSLContext dslContext, List<UsersSegment> segments) {
        new InsertHelper<>(dslContext, VIDEO_SEGMENT_GOALS)
                .addAll(MAPPER, segments)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    void updateSegmentsDisabledFlag(DSLContext dslContext, List<UsersSegment> segments, boolean isDisabled) {
        Condition allOrConditions = DSL.falseCondition();
        for (UsersSegment segment : segments) {
            Condition type = VIDEO_SEGMENT_GOALS.AUDIENCE_TYPE.eq(AdShowType.toSource(segment.getType()));
            Condition id = VIDEO_SEGMENT_GOALS.PID.eq(segment.getAdGroupId());
            Condition typeAndId = type.and(id);
            allOrConditions = allOrConditions.or(typeAndId);
        }
        dslContext.update(VIDEO_SEGMENT_GOALS)
                .set(VIDEO_SEGMENT_GOALS.IS_DISABLED, isDisabled ? 1L : 0L)
                .where(allOrConditions)
                .execute();
    }

    public void updateSegments(DSLContext dslContext,
                               List<UsersSegment> segmentsToEnable,
                               List<UsersSegment> segmentsToDisable) {
        if (!segmentsToEnable.isEmpty()) {
            updateSegmentsDisabledFlag(dslContext, segmentsToEnable, false);
        }
        if (!segmentsToDisable.isEmpty()) {
            updateSegmentsDisabledFlag(dslContext, segmentsToDisable, true);
        }
    }

    public List<Long> getExternalAudienceIds(int shard, LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_ID)
                .from(VIDEO_SEGMENT_GOALS)
                .orderBy(VIDEO_SEGMENT_GOALS.PID, VIDEO_SEGMENT_GOALS.AUDIENCE_TYPE)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_ID);
    }

    public void updateSegmentStatusesByExternalAudienceId(
            int shard,
            Map<Long, VideoSegmentGoalsExternalAudienceStatus> externalIdToStatusMap) {

        Field<VideoSegmentGoalsExternalAudienceStatus> choose =
                JooqMapperUtils.makeCaseStatement(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_ID,
                        VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_STATUS, externalIdToStatusMap);
        dslContextProvider.ppc(shard)
                .update(VIDEO_SEGMENT_GOALS)
                .set(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_STATUS, choose)
                .where(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_ID.in(externalIdToStatusMap.keySet()))
                .execute();
    }

    public void updateSegment(int shard, UsersSegment segment) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        dslContext.update(VIDEO_SEGMENT_GOALS)
                .set(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_ID, segment.getExternalAudienceId())
                .set(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_STATUS,
                        ExternalAudienceStatus.toSource(segment.getExternalAudienceStatus()))
                .set(VIDEO_SEGMENT_GOALS.SEGMENT_OWNER_UID, segment.getSegmentOwnerUid())
                .set(VIDEO_SEGMENT_GOALS.LAST_SUCCESSFUL_UPDATE_TIME, segment.getLastSuccessUpdateTime())
                .set(VIDEO_SEGMENT_GOALS.SEGMENT_ERROR_COUNT, segment.getErrorCount())
                .set(VIDEO_SEGMENT_GOALS.INTERNAL_STATUS, InternalStatus.toSource(segment.getInternalStatus()))
                .where(VIDEO_SEGMENT_GOALS.PID.eq(segment.getAdGroupId()))
                .and(VIDEO_SEGMENT_GOALS.AUDIENCE_TYPE.eq(AdShowType.toSource(segment.getType())))
                .execute();
    }

    public int updateSegmentLastSuccessfulUpdateTimeAndResetSegmentErrorCount(
            DSLContext dslContext,
            Long adgroupId,
            AdShowType adShowType,
            LocalDateTime lastSuccessfulUpdateTime) {
        return dslContext.update(VIDEO_SEGMENT_GOALS)
                .set(VIDEO_SEGMENT_GOALS.LAST_SUCCESSFUL_UPDATE_TIME, lastSuccessfulUpdateTime)
                .set(VIDEO_SEGMENT_GOALS.SEGMENT_ERROR_COUNT, 0L)
                .where(VIDEO_SEGMENT_GOALS.PID.eq(adgroupId))
                .and(VIDEO_SEGMENT_GOALS.AUDIENCE_TYPE.eq(AdShowType.toSource(adShowType)))
                .execute();
    }

    public List<UsersSegment> getOldestSegmentsForUpdate(int shard, AdGroupType adGroupType, int limit) {
        return dslContextProvider.ppc(shard)
                .select(MAPPER.getFieldsToRead())
                .from(VIDEO_SEGMENT_GOALS)
                .join(PHRASES).on(PHRASES.PID.eq(VIDEO_SEGMENT_GOALS.PID))
                .where(PHRASES.ADGROUP_TYPE.eq(AdGroupType.toSource(adGroupType)))
                .and(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_STATUS.in(processed, few_data))
                .and(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_ID.notEqual(0L))
                .and(VIDEO_SEGMENT_GOALS.IS_DISABLED.eq(0L))
                .and(VIDEO_SEGMENT_GOALS.SEGMENT_ERROR_COUNT.le(MAX_ERR_COUNT_FOR_UPDATE_SEGMENT))
                .orderBy(VIDEO_SEGMENT_GOALS.LAST_SUCCESSFUL_UPDATE_TIME)
                .limit(limit)
                .fetch(MAPPER::fromDb);
    }

    public List<UsersSegment> getOldestSegmentsForCreate(int shard, AdGroupType adGroupType, int limit) {
        return dslContextProvider.ppc(shard)
                .select(MAPPER.getFieldsToRead())
                .from(VIDEO_SEGMENT_GOALS)
                .join(PHRASES).on(PHRASES.PID.eq(VIDEO_SEGMENT_GOALS.PID))
                .where(PHRASES.ADGROUP_TYPE.eq(AdGroupType.toSource(adGroupType)))
                .and(VIDEO_SEGMENT_GOALS.INTERNAL_STATUS.eq(new_))
                .and(VIDEO_SEGMENT_GOALS.IS_DISABLED.eq(0L))
                .and(VIDEO_SEGMENT_GOALS.EXTERNAL_AUDIENCE_ID.eq(0L))
                .orderBy(VIDEO_SEGMENT_GOALS.LAST_SUCCESSFUL_UPDATE_TIME)
                .limit(limit)
                .fetch(MAPPER::fromDb);
    }

    /**
     * Возвращает отображения Id группы в список сегментов пользователей
     *
     * @param shard       шард для запроса
     * @param adGroupsIds номера групп
     */
    @Nonnull
    public List<UsersSegment> getSegments(int shard, Collection<Long> adGroupsIds) {
        return dslContextProvider.ppc(shard)
                .select(MAPPER.getFieldsToRead())
                .from(VIDEO_SEGMENT_GOALS)
                .where(VIDEO_SEGMENT_GOALS.PID.in(adGroupsIds))
                .and(VIDEO_SEGMENT_GOALS.IS_DISABLED.eq(0L))
                .orderBy(VIDEO_SEGMENT_GOALS.PID, VIDEO_SEGMENT_GOALS.AUDIENCE_TYPE)
                .fetch(MAPPER::fromDb);
    }

    @Nonnull
    public Map<Long, List<UsersSegment>> getSegmentsAsMap(int shard, Collection<Long> adGroupsIds) {
        List<UsersSegment> segments = getSegments(shard, adGroupsIds);
        return StreamEx.of(segments).groupingBy(UsersSegment::getAdGroupId);
    }

    /**
     * Удалить из базы сегменты при удалении группы.
     * Если группа существует, то правильно использовать updateSegmentsDisabledFlag
     */
    public void deleteSegmentsWithoutAdGroup(DSLContext dslContext, Collection<Long> adGroupIds) {
        SelectConditionStep<Record1<Long>> selectStep = dslContext.select(VIDEO_SEGMENT_GOALS.PID)
                .from(VIDEO_SEGMENT_GOALS)
                .leftJoin(PHRASES).on(PHRASES.PID.eq(VIDEO_SEGMENT_GOALS.PID))
                .where(VIDEO_SEGMENT_GOALS.PID.in(adGroupIds)
                        .and(PHRASES.PID.isNull()));
        dslContext.deleteFrom(VIDEO_SEGMENT_GOALS)
                .where(VIDEO_SEGMENT_GOALS.PID.in(adGroupIds)
                        .and(VIDEO_SEGMENT_GOALS.PID.in(selectStep)))
                .execute();

    }
}
