package ru.yandex.direct.core.entity.userssegments.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdShowType;
import ru.yandex.direct.core.entity.adgroup.model.ExternalAudienceStatus;
import ru.yandex.direct.core.entity.adgroup.model.InternalStatus;
import ru.yandex.direct.core.entity.adgroup.model.UsersSegment;
import ru.yandex.direct.core.entity.userssegments.repository.UsersSegmentRepository;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.tables.CampOptions.CAMP_OPTIONS;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class UsersSegmentService {
    private final UsersSegmentRepository usersSegmentRepository;
    private final DslContextProvider ppcDslContextProvider;

    @Autowired
    public UsersSegmentService(UsersSegmentRepository usersSegmentRepository, DslContextProvider ppcDslContextProvider) {
        this.usersSegmentRepository = usersSegmentRepository;
        this.ppcDslContextProvider = ppcDslContextProvider;
    }

    public void addSegments(int shard, Map<Long, List<UsersSegment>> adGroupsIdWithSegments) {
        addSegments(ppcDslContextProvider.ppc(shard), adGroupsIdWithSegments);
    }

    public void addSegments(DSLContext dslContext, Map<Long, List<UsersSegment>> adGroupsIdWithSegments) {
        List<UsersSegment> allSegments = new ArrayList<>();
        for (Map.Entry<Long, List<UsersSegment>> adGroupIdWithSegments : adGroupsIdWithSegments.entrySet()) {
            List<UsersSegment> segmentListToAdd = adGroupIdWithSegments.getValue();
            if (segmentListToAdd == null) {
                continue;
            }
            Long adGroupId = adGroupIdWithSegments.getKey();
            for (UsersSegment segment : segmentListToAdd) {
                segment.setAdGroupId(adGroupId);
                segment.setLastSuccessUpdateTime(LocalDateTime.now());
            }
            allSegments.addAll(segmentListToAdd);
        }
        addSegmentsInner(dslContext, allSegments);
    }

    private void addSegmentsInner(DSLContext dslContext, List<UsersSegment> segments) {
        if (segments.isEmpty()) {
            return;
        }
        prepareSystemFields(segments);
        usersSegmentRepository.addSegments(dslContext, segments);
    }

    private static void prepareSystemFields(List<UsersSegment> segments) {
        LocalDateTime now = LocalDateTime.now();
        for (UsersSegment segment : segments) {
            segment.setTimeCreated(now);
            segment.setErrorCount(0L);
            segment.setExternalAudienceId(0L);
            segment.setSegmentOwnerUid(0L);
            segment.setIsDisabled(false);
            segment.setInternalStatus(InternalStatus.NEW_);
            segment.setExternalAudienceStatus(ExternalAudienceStatus.FEW_DATA);
        }
    }

    public void updateAdGroupsSegments(int shard, Long campaignId,
                                       Map<Long, List<UsersSegment>> adGroupsWithSegments) {
        updateAdGroupsSegments(ppcDslContextProvider.ppc(shard), campaignId, adGroupsWithSegments);
    }

    public void updateAdGroupsSegments(DSLContext dslContext, Long campaignId,
                                       Map<Long, List<UsersSegment>> adGroupsWithSegments) {
        if (adGroupsWithSegments.isEmpty()) {
            return;
        }

        LocalDateTime campaignCreateDate = dslContext.select(CAMP_OPTIONS.CREATE_TIME)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.eq(campaignId))
                .fetchOne(CAMP_OPTIONS.CREATE_TIME);

        List<UsersSegment> segmentsToEnable = new ArrayList<>();
        List<UsersSegment> segmentsToDisable = new ArrayList<>();
        for (Map.Entry<Long, List<UsersSegment>> adGroupIdWithSegments : adGroupsWithSegments.entrySet()) {
            List<UsersSegment> segments = nvl(adGroupIdWithSegments.getValue(), Collections.emptyList());
            for (UsersSegment segment : segments) {
                segment.setLastSuccessUpdateTime(campaignCreateDate);
            }
            segmentsToEnable.addAll(segments);
            Set<AdShowType> typesToDisable = EnumSet.allOf(AdShowType.class);
            typesToDisable.removeAll(mapList(segments, UsersSegment::getType));
            for (AdShowType type : typesToDisable) {
                segmentsToDisable.add(new UsersSegment()
                        .withType(type)
                        .withAdGroupId(adGroupIdWithSegments.getKey()));
            }
        }
        addSegmentsInner(dslContext, segmentsToEnable);
        usersSegmentRepository.updateSegments(dslContext, segmentsToEnable, segmentsToDisable);
    }
}
