package ru.yandex.direct.core.entity.vcard;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;

/**
 * Телефон из визитки
 */
@ParametersAreNonnullByDefault
public class VcardPhone {
    private String countryCode = "";
    private String cityCode = "";
    private String phone = "";
    private String ext = "";

    /**
     * @return код страны
     */
    public String getCountryCode() {
        return countryCode;
    }

    /**
     * @return код города/оператора - часть телефона, которая обычно находится в скобках
     */
    public String getCityCode() {
        return cityCode;
    }

    /**
     * @return номер телефона
     */
    public String getPhone() {
        return phone;
    }

    /**
     * @return добавочный код
     */
    public String getExt() {
        return ext;
    }

    /**
     * Преобразует строку с закодированным телефоном с решетками в объект {@link VcardPhone}.
     * Если у закодированной строки неправильный формат, кидается исключение {@link IllegalArgumentException}
     * Если строка пустая или null, возвращается пустой объект
     *
     * @param encodedString закодированная строка
     * @return объект VCardPhone
     * @throws IllegalArgumentException если закодированный телефон имеет невалидный формат.
     */
    public static VcardPhone fromEncodedString(@Nullable String encodedString) {
        VcardPhone result = new VcardPhone();

        if (!Strings.isNullOrEmpty(encodedString)) {
            String[] phoneParts = encodedString.split("#", 4);

            // в базе Директа есть телефоны самых разных видов, и в том числе такие, у которых меньше чем 4 части,
            // разделенные решеткой. Но это либо старые визитки, либо копии старых визиток, которых совсем мало
            // (около 0,045%). Все новые визитки создаются как минимум с четырьмя частями
            // Все телефоны у которых меньше чем 4 части считаем невалидными и кидаем исключение
            if (phoneParts.length < 4) {
                throw new IllegalArgumentException(String.format("Encoded phone has invalid format %s", encodedString));
            }

            result.countryCode = phoneParts[0];
            result.cityCode = phoneParts[1];
            result.phone = phoneParts[2];
            result.ext = phoneParts[3];
        }
        return result;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        VcardPhone that = (VcardPhone) o;

        if (!countryCode.equals(that.countryCode)) {
            return false;
        }
        if (!cityCode.equals(that.cityCode)) {
            return false;
        }
        if (!phone.equals(that.phone)) {
            return false;
        }
        return ext.equals(that.ext);

    }

    @Override
    public int hashCode() {
        int result = countryCode.hashCode();
        result = 31 * result + cityCode.hashCode();
        result = 31 * result + phone.hashCode();
        result = 31 * result + ext.hashCode();
        return result;
    }
}
