package ru.yandex.direct.core.entity.vcard;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import one.util.streamex.StreamEx;

/**
 * Расписание рабочих часов из визитки
 */
@ParametersAreNonnullByDefault
public class VcardWorktime {
    /**
     * @return массив расписаний рабочего времени для дня, или диапазона дней недели.
     */
    public DailySchedule[] getDailySchedules() {
        return dailySchedules;
    }

    private final DailySchedule[] dailySchedules;

    public VcardWorktime(DailySchedule[] dailySchedules) {
        this.dailySchedules = dailySchedules;
    }

    /**
     * Декодирует строку {@code encodedString} из формата с решетками и точками с запятыми, которым представлены
     * расписания в базе, и в каком они приходят из верстки, в объект {@link VcardWorktime}.
     * <p>
     * Строка разбивается по точкам с запятыми и полученные закодированные расписания по дням декодируются через
     * {@link DailySchedule#fromEncodedString(String)}
     * Если закодированная строка имеет неправильный формат, выкидывается исключение {@link IllegalArgumentException}
     * <p>
     * Если строка {@code encodedString} пустая или равна null, возвращается пустой объект
     *
     * @param encodedString закодированная строка с расписаниями
     * @return объект {@link VcardWorktime}
     * @throws IllegalArgumentException если строка имеет невалидный формат
     * @see VcardWorktime#fromDailySchedules(List)
     */
    public static VcardWorktime fromEncodedString(@Nullable String encodedString) {
        if (Strings.isNullOrEmpty(encodedString)) {
            return new VcardWorktime(new DailySchedule[0]);
        }

        // передаем trimEmpty=false в split, чтобы отловить точку с запятой в конце строки
        List<DailySchedule> dailySchedules = StreamEx.split(encodedString, ';', false)
                .map(DailySchedule::fromEncodedString)
                .toList();

        return fromDailySchedules(dailySchedules);
    }

    /**
     * Преобразовывает список расписаний в объект {@link VcardWorktime}
     * Сортирует расписания в хронологическом порядке (расписание за вторник идет первее, чем расписание за четверг)
     * и склеивает соседние расписания, если это возможно
     *
     * @see DailySchedule#canBeGluedWith(DailySchedule)
     */
    public static VcardWorktime fromDailySchedules(List<DailySchedule> dailySchedules) {
        List<DailySchedule> sortedSchedules = StreamEx.of(dailySchedules)
                .sorted(Comparator.comparingInt(DailySchedule::getDaySince))
                .toList();

        List<DailySchedule> gluedSchedulesList = new ArrayList<>();

        for (DailySchedule schedule : sortedSchedules) {
            if (gluedSchedulesList.size() > 0) {
                int lastIndex = gluedSchedulesList.size() - 1;
                DailySchedule prevDailySchedule = gluedSchedulesList.get(lastIndex);

                if (prevDailySchedule.canBeGluedWith(schedule)) {
                    gluedSchedulesList.set(lastIndex, prevDailySchedule.glueWith(schedule));
                    continue;
                }
            }

            gluedSchedulesList.add(schedule);
        }

        return new VcardWorktime(gluedSchedulesList.toArray(new DailySchedule[0]));
    }

    /**
     * Преобразовывает в закодированную строку для записи в базу
     */
    public String toEncodedString() {
        return StreamEx.of(dailySchedules)
                .sorted(Comparator.comparingInt(DailySchedule::getDaySince))
                .map(DailySchedule::toEncodedString)
                .joining(";");
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        VcardWorktime that = (VcardWorktime) o;

        return Arrays.equals(dailySchedules, that.dailySchedules);
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(dailySchedules);
    }

    @Override
    public String toString() {
        return "VCardWorktime{" +
                "dailySchedules=" + Arrays.toString(dailySchedules) +
                '}';
    }

    /**
     * Расписание рабочего времени из визитки в какой-то день недели или диапазон дней недели.
     */
    public static class DailySchedule {
        /**
         * @return Номер дня недели, включительно, с которого действует расписание. 0 = понедельник
         */
        public int getDaySince() {
            return daySince;
        }

        /**
         * @return Номер дня недели, включительно, до которого действует расписание. 0 = понедельник
         */
        public int getDayTill() {
            return dayTill;
        }

        /**
         * @return Час начала работы
         */
        public int getHourSince() {
            return hourSince;
        }

        /**
         * @return Минута начала работы
         */
        public int getMinuteSince() {
            return minuteSince;
        }

        /**
         * @return Час окончания работы
         */
        public int getHourTill() {
            return hourTill;
        }

        /**
         * @return Минута окончания работы
         */
        public int getMinuteTill() {
            return minuteTill;
        }

        /**
         * @param daySince    Номер дня недели, включительно, с которого действует расписание. 0 = понедельник
         * @param dayTill     Номер дня недели, включительно, до которого действует расписание. 0 = понедельник
         * @param hourSince   Час начала работы
         * @param minuteSince Минута начала работы
         * @param hourTill    Час окончания работы
         * @param minuteTill  Минута окончания работы
         */
        private DailySchedule(int daySince, int dayTill, int hourSince, int minuteSince, int hourTill, int minuteTill) {
            this.daySince = daySince;
            this.dayTill = dayTill;
            this.hourSince = hourSince;
            this.minuteSince = minuteSince;
            this.hourTill = hourTill;
            this.minuteTill = minuteTill;
        }

        private final int daySince;
        private final int dayTill;
        private final int hourSince;
        private final int minuteSince;
        private final int hourTill;
        private final int minuteTill;


        /**
         * Декодирует строку с расписанием работы в определенный день недели или диапазон дней недели.
         * Закодированная строка имеет вид
         * {@code (день начала)#(день окончания)#(час начала)#(минута начала)#(час окончания)#(минута окончания)}
         * <p>
         * день начала и окончания указываются включительно, и являются номером дня недели, где 0 = понедельник.
         * <p>
         * Если закодированная строка равна null или имеет невалидный формат, выкидывается исключение
         * {@link IllegalArgumentException}
         *
         * @param encodedString закодированная строка
         * @return {@link DailySchedule} с декодированным расписанием
         * @throws IllegalArgumentException если строка имеет невалидный формат
         */
        public static DailySchedule fromEncodedString(@Nullable String encodedString) {
            if (encodedString == null) {
                throw new IllegalArgumentException("encoded daily schedule string can't be null");
            }

            String[] parts = encodedString.split("#");
            if (parts.length != 6) {
                throw new IllegalArgumentException("Invalid encoded daily schedule string");
            }

            return new DailySchedule(Integer.valueOf(parts[0]),
                    Integer.valueOf(parts[1]),
                    Integer.valueOf(parts[2]),
                    Integer.valueOf(parts[3]),
                    Integer.valueOf(parts[4]),
                    Integer.valueOf(parts[5])
            );
        }

        /**
         * Проверяет, можно ли данное расписание склеить с расписанием {@code otherSchedule}.
         * Расписания можно склеить, если они относятся к идущим подряд дням недели, у которых время начала и время
         * окончания совпадают.
         *
         * @param otherSchedule расписание, с которым проверяется возможность склейки
         * @return {@code true}, если данное расписание и {@code otherSchedule} можно склеить
         */
        public boolean canBeGluedWith(DailySchedule otherSchedule) {
            if (!(this.daySince > 0 && this.daySince == otherSchedule.dayTill + 1) &&
                    !(this.dayTill < 6 && this.dayTill == otherSchedule.daySince - 1)) {
                return false;
            }

            return this.hourSince == otherSchedule.hourSince &&
                    this.hourTill == otherSchedule.hourTill &&
                    this.minuteSince == otherSchedule.minuteSince &&
                    this.minuteTill == otherSchedule.minuteTill;
        }

        /**
         * Склеивает данное расписание с указанным в {@code otherSchedule} и возвращает склеенное расписание.
         * При этом склеиваемые расписания не изменяются.
         *
         * @param otherSchedule расписание, с которым нужно склеить данное
         * @return результат склейки
         */
        public DailySchedule glueWith(DailySchedule otherSchedule) {
            if (!canBeGluedWith(otherSchedule)) {
                throw new IllegalArgumentException(
                        String.format("can't glue %s to %s", this.toString(), otherSchedule.toString()));
            }

            int newDayTill = this.dayTill;
            int newDaySince = this.daySince;
            if (dayTill < otherSchedule.daySince) {
                newDayTill = otherSchedule.dayTill;
            } else {
                newDaySince = otherSchedule.daySince;
            }

            return new DailySchedule(newDaySince, newDayTill, hourSince, minuteSince, hourTill, minuteTill);
        }

        /**
         * Вычисляет признак круглосуточности расписания.
         * Расписание считается круглосуточным, если время работы равно 00:00 - 00:00, или если расписание одно на всю
         * неделю, у которого время начала и окончания совпадают.
         *
         * @return true, если расписание является круглосуточным
         */
        public boolean isRoundTheClock() {
            return hourSince == 0 && minuteSince == 0 && hourTill == 0 && minuteTill == 0 ||
                    daySince == 0 && dayTill == 6 && hourSince == hourTill && minuteSince == minuteTill;
        }

        /**
         * Преобразовывает в закодированную строку для записи в базу
         */
        private String toEncodedString() {
            return String.format("%d#%d#%s#%s", daySince, dayTill,
                    timeToEncodedString(hourSince, minuteSince), timeToEncodedString(hourTill, minuteTill));
        }

        private static String timeToEncodedString(int hour, int minute) {
            return String.format("%02d#%02d", hour, minute);
        }

        @Override
        public String toString() {
            return Stream.of(daySince, dayTill, hourSince, minuteSince, hourTill, minuteTill)
                    .map(String::valueOf)
                    .collect(Collectors.joining("#"));
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            DailySchedule that = (DailySchedule) o;

            if (daySince != that.daySince) {
                return false;
            }
            if (dayTill != that.dayTill) {
                return false;
            }
            if (hourSince != that.hourSince) {
                return false;
            }
            if (minuteSince != that.minuteSince) {
                return false;
            }
            if (hourTill != that.hourTill) {
                return false;
            }
            return minuteTill == that.minuteTill;
        }

        @Override
        public int hashCode() {
            int result = daySince;
            result = 31 * result + dayTill;
            result = 31 * result + hourSince;
            result = 31 * result + minuteSince;
            result = 31 * result + hourTill;
            result = 31 * result + minuteTill;
            return result;
        }
    }


    /**
     * Builder для {@link DailySchedule}
     */
    public static class DailyScheduleBuilder {
        private int daySince;
        private int dayTill;
        private int hourSince;
        private int minuteSince;
        private int hourTill;
        private int minuteTill;

        public static DailyScheduleBuilder builder() {
            return new DailyScheduleBuilder();
        }

        private DailyScheduleBuilder() {
        }

        public DailyScheduleBuilder withDaySince(int daySince) {
            this.daySince = daySince;
            return this;
        }

        public DailyScheduleBuilder withDayTill(int dayTill) {
            this.dayTill = dayTill;
            return this;
        }

        public DailyScheduleBuilder withHourSince(int hourSince) {
            this.hourSince = hourSince;
            return this;
        }

        public DailyScheduleBuilder withMinuteSince(int minuteSince) {
            this.minuteSince = minuteSince;
            return this;
        }

        public DailyScheduleBuilder withHourTill(int hourTill) {
            this.hourTill = hourTill;
            return this;
        }

        public DailyScheduleBuilder withMinuteTill(int minuteTill) {
            this.minuteTill = minuteTill;
            return this;
        }

        public DailySchedule build() {
            return new DailySchedule(daySince, dayTill, hourSince, minuteSince, hourTill, minuteTill);
        }

    }

}
