package ru.yandex.direct.core.entity.vcard.repository;

import java.util.Set;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.vcard.model.InstantMessenger;
import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.core.entity.vcard.model.PointOnMap;

/**
 * Вспомогательный класс для mapping-га полей Vcard-а
 */
public class VcardMappings {
    private static final Logger logger = LoggerFactory.getLogger(VcardMappings.class);
    /**
     * Список кодов стран для которых необходима нормализация номеров
     * телефонов
     */
    private static final Set<String> SPECIAL_COUNTRIES = ImmutableSet.of(
            "+7", // Россия, Казахстан
            "+375", // Беларусь
            "+380" // Украина
    );
    /**
     * Макс. размер нормализованного номера телефона
     */
    private static final int MAX_NORMALIZED_PHONE_NUMBER_LENGTH = 3 + 1 + 2 + 1 + 2;

    /**
     * Разбиваем номер на группы по две цифры. Перед преобразованием из номера удаляются все символы '-'
     * Группы отделяются друг от друга символом '-'. Если в полученном разбиении первая группа состоит из одной цифры и
     * кол-во групп больше 3, то первая группа объединяется со второй группой
     * Примеры:
     * 1 -> 1
     * 12 -> 12
     * 123 -> 1-23
     * 1234 -> 12-34
     * 12345 -> 1-23-45
     * 123456 -> 12-34-56
     * 1234567 -> 123-45-67
     *
     * @param phoneNumber Телефонный номер в формате \d+(-\d+)*
     * @return Преобразованный номер
     */
    public static String normalizePhoneNumber(String phoneNumber) {
        if (StringUtils.isEmpty(phoneNumber)) {
            return phoneNumber;
        }

        // Берем наименьший из размеров:
        // Размер строки + кол-во дефисов которые мы вставим
        // Макс. размер номализованного телефонного номера + один дефис на месте склейки
        int maxBufferSize = Math.min(
                phoneNumber.length() + phoneNumber.length() / 2, MAX_NORMALIZED_PHONE_NUMBER_LENGTH + 1);
        StringBuilder builder = new StringBuilder(maxBufferSize);
        int numberOfDigit = 0;
        for (int i = 1; i <= phoneNumber.length(); i++) {
            char ch = phoneNumber.charAt(phoneNumber.length() - i);
            if ('0' <= ch && ch <= '9') {
                builder.append(ch);
                numberOfDigit++;
                if (numberOfDigit % 2 == 0) {
                    builder.append('-');
                }
            }
        }
        // Если добавили лишний '-', то убираем его
        if (builder.charAt(builder.length() - 1) == '-') {
            builder.setLength(builder.length() - 1);
        }

        // Склеиваем первую группу со второй, если кол-во групп > 3 и она содержит
        // одну цифру
        if (builder.length() >= 10 && builder.charAt(builder.length() - 2) == '-') {
            builder.deleteCharAt(builder.length() - 2);
        }

        builder.reverse();

        return builder.toString();
    }

    public static String phoneToDb(Phone phone) {
        if (phone == null) {
            return null;
        }
        String phoneNumber = phone.getPhoneNumber();
        if (SPECIAL_COUNTRIES.contains(phone.getCountryCode())) {
            phoneNumber = normalizePhoneNumber(phoneNumber);
        }
        String phoneStr = phone.getCountryCode() + "#" + phone.getCityCode() + "#" + phoneNumber + "#";
        if (phone.getExtension() != null) {
            phoneStr += phone.getExtension();
        }
        return phoneStr;
    }

    public static Phone phoneFromDb(String phoneStr) {
        if (phoneStr == null) {
            return null;
        }
        String[] parts = phoneStr.split("#");
        if (parts.length < 3) {
            // DIRECT-80968
            logger.warn("Invalid phone number returned from DB: {}", phoneStr);
            return new Phone().withCountryCode("").withCityCode("").withPhoneNumber(phoneStr);
        }
        Phone phone = new Phone()
                .withCountryCode(parts[0])
                .withCityCode(parts[1])
                .withPhoneNumber(parts[2]);
        // Extension не может быть пуст, если он пустой значит, то должен быть null
        if (parts.length > 3 && !parts[3].isEmpty()) {
            phone.withExtension(parts[3]);
        }
        return phone;
    }

    static InstantMessenger instantMessengerByClientAndLoginFromDb(String client, String login) {
        if (client == null && login == null) {
            return null;
        }
        return new InstantMessenger().withType(client).withLogin(login);
    }

    static String loginByImToDb(InstantMessenger im) {
        return im != null ? im.getLogin() : null;
    }

    static String clientByImToDb(InstantMessenger im) {
        return im != null ? im.getType() : null;
    }

    static PointOnMap pointOnMapByIdFromDb(Long pointId) {
        return pointId != null ? new PointOnMap().withId(pointId) : null;
    }
}
