package ru.yandex.direct.core.entity.vcard.repository;

import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BinaryOperator;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Select;
import org.jooq.SelectConditionStep;
import org.jooq.Table;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.vcard.model.PointOnMap;
import ru.yandex.direct.core.entity.vcard.model.PointPrecision;
import ru.yandex.direct.core.entity.vcard.model.PointType;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.repository.internal.AddressesRepository;
import ru.yandex.direct.core.entity.vcard.repository.internal.MapsRepository;
import ru.yandex.direct.core.entity.vcard.repository.internal.OrgDetailsRepository;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.tables.records.VcardsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.AddedModelId;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static org.jooq.impl.DSL.cast;
import static org.jooq.impl.DSL.concat;
import static org.jooq.impl.DSL.count;
import static org.jooq.impl.DSL.ifnull;
import static org.jooq.impl.DSL.md5;
import static ru.yandex.direct.core.entity.vcard.repository.VcardMappings.phoneToDb;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDRESSES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.MEDIAPLAN_BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.ORG_DETAILS;
import static ru.yandex.direct.dbschema.ppc.Tables.VCARDS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromFields;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.HashingUtils.getMd5HashAsHexString;

@Repository
@ParametersAreNonnullByDefault
public class VcardRepository {

    private static final String PLACEHOLDER = "%";
    private static final String SEPARATOR = "~";

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final MapsRepository mapsRepository;
    private final OrgDetailsRepository orgDetailsRepository;
    private final AddressesRepository addressesRepository;

    private final JooqMapperWithSupplier<Vcard> vcardMapper;
    private final Field<String> md5Field;
    private final Collection<Field<?>> vcardFieldsToRead;

    @Autowired
    public VcardRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper,
                           MapsRepository mapsRepository, AddressesRepository addressesRepository,
                           OrgDetailsRepository orgDetailsRepository) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.mapsRepository = mapsRepository;
        this.orgDetailsRepository = orgDetailsRepository;
        this.addressesRepository = addressesRepository;

        this.vcardMapper = createVcardMapper();
        this.md5Field = createMd5Field();
        this.vcardFieldsToRead = vcardMapper.getFieldsToRead();
    }

    public List<Vcard> getVcards(int shard, long uid) {
        return getVcards(shard, uid, null, maxLimited());
    }

    public List<Vcard> getVcards(int shard, long uid, @Nullable Collection<Long> vcardIds) {
        return getVcards(shard, uid, vcardIds, maxLimited());
    }

    public List<Vcard> getVcards(int shard, long uid, @Nullable Collection<Long> vcardIds,
                                 LimitOffset limitOffset) {
        List<Vcard> vcardsWithoutPoints = selectVcardsWithoutPoints(shard, uid, vcardIds, null, limitOffset);
        return fillVcardsPoints(shard, vcardsWithoutPoints);
    }

    public List<Vcard> getVcards(int shard, Collection<Long> vcardIds) {
        List<Vcard> vcardsWithoutPoints = selectVcardsWithoutPoints(shard, null, vcardIds, null, maxLimited());
        return fillVcardsPoints(shard, vcardsWithoutPoints);
    }

    public List<Vcard> getVcardsByAddressIds(int shard, Collection<Long> addressIds) {
        List<Vcard> vcardsWithoutPoints = selectVcardsWithoutPoints(shard, null, null, addressIds, maxLimited());
        return fillVcardsPoints(shard, vcardsWithoutPoints);
    }

    public List<Vcard> selectVcardsWithoutPoints(int shard,
                                                  @Nullable Long uid,
                                                  @Nullable Collection<Long> vcardIds,
                                                  @Nullable Collection<Long> addressIds,
                                                  LimitOffset limitOffset) {
        if (uid == null && vcardIds == null && addressIds == null) {
            throw new IllegalArgumentException("uid, vcardIds and addressIds are null");
        }
        if (vcardIds != null && vcardIds.isEmpty()) {
            return emptyList();
        }
        if (addressIds != null && addressIds.isEmpty()) {
            return emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(vcardFieldsToRead)
                .from(VCARDS)
                .leftJoin(ADDRESSES).on(ADDRESSES.AID.eq(VCARDS.ADDRESS_ID))
                .leftJoin(ORG_DETAILS).on(ORG_DETAILS.ORG_DETAILS_ID.eq(VCARDS.ORG_DETAILS_ID))
                .where(uid != null ? VCARDS.UID.eq(uid) : DSL.trueCondition())
                .and(vcardIds != null ? VCARDS.VCARD_ID.in(vcardIds) : DSL.trueCondition())
                .and(addressIds != null ? VCARDS.ADDRESS_ID.in(addressIds) : DSL.trueCondition())
                .orderBy(VCARDS.VCARD_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch()
                .map(vcardMapper::fromDb);
    }

    private List<Vcard> fillVcardsPoints(int shard, List<Vcard> vcards) {
        Set<Long> pointsIdsToFetch = new HashSet<>(vcards.size() * 2);

        for (Vcard vcard : vcards) {
            if (vcard.getManualPoint() != null) {
                pointsIdsToFetch.add(vcard.getManualPoint().getId());
            }
            if (vcard.getAutoPoint() != null) {
                pointsIdsToFetch.add(vcard.getAutoPoint().getId());
            }
        }

        Map<Long, PointOnMap> points = mapsRepository.getPoints(shard, pointsIdsToFetch);

        for (Vcard vcard : vcards) {
            if (vcard.getAutoPoint() != null) {
                vcard.withAutoPoint(points.get(vcard.getAutoPoint().getId()));
            }
            if (vcard.getManualPoint() != null) {
                vcard.withManualPoint(points.get(vcard.getManualPoint().getId()));
            }
        }

        return vcards;
    }

    /**
     * Получение отношения id визитки - id кампании для всех визиток, принадлежащих указанному пользователю
     *
     * @param shard шард
     * @param uid   id пользователя
     * @return map id визитки - id кампании для визиток, принадлежащих указанному пользователю
     */
    public Map<Long, Long> getVcardIdsToCampaignIds(int shard, long uid) {
        return dslContextProvider.ppc(shard)
                .select(VCARDS.VCARD_ID, VCARDS.CID)
                .from(VCARDS)
                .where(VCARDS.UID.eq(uid))
                .fetchMap(VCARDS.VCARD_ID, VCARDS.CID);
    }

    /**
     * Получает по id кампаний созданные на них визитки
     *
     * @param shard       шард
     * @param campaignIds коллекция id кампаний
     * @return мапа id компании - список id визиток этой кампании
     */
    public Map<Long, List<Long>> getVcardIdsByCampaignIds(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(VCARDS.VCARD_ID, VCARDS.CID, VCARDS.LAST_CHANGE)
                .from(VCARDS)
                .where(VCARDS.CID.in(campaignIds))
                .fetchGroups(VCARDS.CID, VCARDS.VCARD_ID);
    }

    public Map<Long, Long> getLastVcardIdByCampaignId(int shard, Collection<Long> campaignIds) {
        Map<Long, List<Vcard>> map = dslContextProvider.ppc(shard)
                .select(VCARDS.VCARD_ID, VCARDS.CID, VCARDS.LAST_CHANGE)
                .from(VCARDS)
                .where(VCARDS.CID.in(campaignIds))
                .fetchGroups(VCARDS.CID, vcardMapper::fromDb);

        return EntryStream.of(map)
                .mapValues(vcard -> vcard.stream()
                        .max(Comparator.comparing(vcard1 -> nvl(vcard1.getLastChange(), LocalDateTime.MIN)))
                        .get().getId())
                .toMap();
    }

    /**
     * Получение отношения id баннера - id визитки для визиток, принадлежащих указанному пользователю
     *
     * @param shard     шард
     * @param clientUid uid пользователя
     * @param bannerIds коллекция проверяемых id баннеров (если null то нет фильтрации по bid)
     * @param vcardIds  коллекция проверяемых id визиток (если null то нет фильтрации по vcard_id)
     * @return map id баннера - id визитки для визиток, принадлежащих указанному пользователю
     */
    public Map<Long, Long> getBannerIdsToVcardIds(int shard, long clientUid,
                                                  @Nullable Collection<Long> bannerIds,
                                                  @Nullable Collection<Long> vcardIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, VCARDS.VCARD_ID)
                .from(BANNERS)
                .innerJoin(VCARDS).on(VCARDS.VCARD_ID.eq(BANNERS.VCARD_ID))
                .where(VCARDS.UID.eq(clientUid))
                .and(vcardIds != null ? VCARDS.VCARD_ID.in(vcardIds) : DSL.trueCondition())
                .and(bannerIds != null ? BANNERS.BID.in(bannerIds) : DSL.trueCondition())
                .fetchMap(BANNERS.BID, VCARDS.VCARD_ID);
    }

    /**
     * Получение id визиток пользователя uid, которые используются в баннерах
     *
     * @param shard     шард
     * @param clientUid uid клиента
     * @return id визиток
     */
    public Set<Long> getVcardsIdsFromBannersOnly(int shard, long clientUid) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.VCARD_ID)
                .from(BANNERS)
                .innerJoin(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(CAMPAIGNS.UID.eq(clientUid))
                .and(BANNERS.BANNER_TYPE.in(BannersBannerType.text, BannersBannerType.dynamic))
                .and(BANNERS.VCARD_ID.isNotNull())
                .fetchSet(BANNERS.VCARD_ID);
    }

    /**
     * Для каждого id визитки вычисляет сколько раз она используется
     *
     * @param shard     шард
     * @param clientUid uid пользователя
     * @param vcardIds  список id визиток
     * @return map id визитки - сколько раз она используется
     */
    public Map<Long, Long> getVcardsUsesCount(int shard, long clientUid, Collection<Long> vcardIds) {
        String bannersCount = "banners_count";
        Map<Long, Long> bannersCountByVcardId = dslContextProvider.ppc(shard)
                .select(VCARDS.VCARD_ID, count(BANNERS.BID).as(bannersCount))
                .from(VCARDS)
                .innerJoin(BANNERS).on(BANNERS.VCARD_ID.eq(VCARDS.VCARD_ID))
                .where(VCARDS.UID.eq(clientUid))
                .and(VCARDS.VCARD_ID.in(vcardIds))
                .groupBy(VCARDS.VCARD_ID)
                .fetchMap(VCARDS.VCARD_ID, r -> r.getValue(bannersCount, Long.class));

        Map<Long, Long> mediaplanBannersCountByVcardId = dslContextProvider.ppc(shard)
                .select(VCARDS.VCARD_ID, count(MEDIAPLAN_BANNERS.MBID).as(bannersCount))
                .from(VCARDS)
                .innerJoin(MEDIAPLAN_BANNERS).on(MEDIAPLAN_BANNERS.VCARD_ID.eq(VCARDS.VCARD_ID))
                .where(VCARDS.UID.eq(clientUid))
                .and(VCARDS.VCARD_ID.in(vcardIds))
                .groupBy(VCARDS.VCARD_ID)
                .fetchMap(VCARDS.VCARD_ID, r -> r.getValue(bannersCount, Long.class));

        return EntryStream.of(bannersCountByVcardId)
                .append(mediaplanBannersCountByVcardId).toMap(Long::sum);
    }

    public List<AddedModelId> addVcards(int shard, long clientUid, ClientId clientId, List<Vcard> vcards) {
        addVcardsToMapsTable(shard, vcards);
        List<Boolean> isAddressChanged = addVcardsToAddressesTable(shard, clientId, vcards);
        addVcardsToOrgDetailsTable(shard, clientUid, clientId, vcards);
        List<AddedModelId> vcardIds = addVcardsToVcardsTable(shard, clientUid, clientId, vcards);
        return StreamEx.zip(isAddressChanged, vcardIds,
                        (addressChanged, modelId) -> addressChanged && modelId.isExisting() ?
                                AddedModelId.ofChanged(modelId.getId()) : modelId)
                .toList();
    }

    private void addVcardsToMapsTable(int shard, List<Vcard> vcards) {
        List<PointOnMap> allPoints = StreamEx.of(vcards)
                .flatMap(vcard -> StreamEx.of(vcard.getAutoPoint(), vcard.getManualPoint()))
                .filter(Objects::nonNull)
                .toList();
        if (!allPoints.isEmpty()) {
            List<Long> pointsIds = mapsRepository.getOrCreatePointOnMap(shard, allPoints);
            for (int i = 0; i < allPoints.size(); i++) {
                allPoints.get(i).setId(pointsIds.get(i));
            }
        }
    }

    private List<Boolean> addVcardsToAddressesTable(int shard, ClientId clientId, List<Vcard> vcards) {
        List<AddedModelId> addressesIds = addressesRepository.getOrCreateAddresses(shard, clientId, vcards);
        for (int i = 0; i < vcards.size(); i++) {
            vcards.get(i).setAddressId(addressesIds.get(i) == null ? null : addressesIds.get(i).getId());
        }
        return addressesIds.stream()
                .map(modelId -> {
                    if (modelId == null) {
                        return false;
                    }
                    return modelId.isChanged();
                }).collect(toList());
    }

    private void addVcardsToOrgDetailsTable(int shard, long clientUid, ClientId clientId, List<Vcard> vcards) {
        List<Long> orgDetailsIds = orgDetailsRepository.getOrCreateOrgDetails(shard, clientUid, clientId, vcards);
        for (int i = 0; i < vcards.size(); i++) {
            vcards.get(i).setOrgDetailsId(orgDetailsIds.get(i));
        }
    }

    private List<AddedModelId> addVcardsToVcardsTable(int shard, long clientUid, ClientId clientId,
                                                      List<Vcard> vcards) {
        List<String> vcardsHashes = mapList(vcards, this::calcVcardMd5Hash);

        Map<String, Long> existingVcardsByHashes = getExistingVcardsByHashes(shard, clientUid, vcardsHashes);

        int idCount = (int) StreamEx.of(vcardsHashes).distinct().count() - existingVcardsByHashes.size();
        Iterator<Long> ids = shardHelper.generateVcardIds(clientId.asLong(), idCount).iterator();

        DSLContext dslContext = dslContextProvider.ppc(shard);
        InsertHelper<VcardsRecord> insertHelper = new InsertHelper<>(dslContext, VCARDS);

        List<AddedModelId> vcardIds = StreamEx.zip(vcards, vcardsHashes, (vcard, vcardHash) -> {

            Long existingVcardId = existingVcardsByHashes.get(vcardHash);
            if (existingVcardId != null) {
                // Для копирования сущностей важно, чтобы по факту скопированные (в том числе дедуплицированные)
                // сущности обладали корректным id, null считается не скопированным
                vcard.setId(existingVcardId);
                return AddedModelId.ofExisting(existingVcardId);
            }

            Long id = ids.next();
            vcard.withId(id);

            // добавляем сгенерированный id в мапу существующих визиток,
            // чтобы не создавать одинаковые визитки дважды
            existingVcardsByHashes.put(vcardHash, id);

            insertHelper.add(vcardMapper, vcard).newRecord();

            return AddedModelId.ofNew(id);
        }).toList();

        insertHelper.executeIfRecordsAdded();

        return vcardIds;
    }

    private Map<String, Long> getExistingVcardsByHashes(int shard, long clientUid, Collection<String> vcardsHashes) {
        Field<String> md5Alias = DSL.field("vcards_md5", String.class);
        Field<Long> idAlias = DSL.field("vcard_id", Long.class);

        Table<Record> innerTable = dslContextProvider.ppc(shard)
                .select(asList(md5Field.as(md5Alias.getName()), VCARDS.VCARD_ID.as(idAlias.getName())))
                .from(VCARDS)
                .where(VCARDS.UID.eq(clientUid))
                .asTable("tmp");

        return dslContextProvider.ppc(shard)
                .select(asList(md5Alias, idAlias))
                .from(innerTable)
                .where(md5Alias.in(vcardsHashes))
                .stream()
                .collect(
                        toMap(
                                rec -> rec.getValue(md5Alias),
                                rec -> rec.getValue(idAlias),
                                BinaryOperator.minBy(Comparator.<Long>naturalOrder())));
    }

    public Set<Long> getUnusedVcards(int shard, long clientUid, Collection<Long> vcardIds) {
        return createSelectStepForUnusedVcards(shard, clientUid, vcardIds)
                .fetchSet(VCARDS.VCARD_ID);
    }

    /**
     * Установка даты отвязки визитки. Также обновляет lastChange
     *
     * @param shard    шард
     * @param vcardIds список id визиток
     */
    public void setDissociateLastChange(int shard, Collection<Long> vcardIds) {
        setDissociateLastChange(dslContextProvider.ppc(shard), vcardIds);
    }

    /**
     * Установка даты отвязки визитки. Также обновляет lastChange
     *
     * @param dsl      контекст базы
     * @param vcardIds список id визиток
     */
    public void setDissociateLastChange(DSLContext dsl, Collection<Long> vcardIds) {
        if (vcardIds.isEmpty()) {
            return;
        }
        dsl
                .update(VCARDS)
                .set(VCARDS.LAST_CHANGE, LocalDateTime.now())
                .set(VCARDS.LAST_DISSOCIATION, LocalDateTime.now())
                .where(VCARDS.VCARD_ID.in(vcardIds))
                .execute();
    }

    public Set<Long> deleteUnusedVcards(int shard, Collection<Long> vcardIds) {
        return deleteUnusedVcards(shard, null, vcardIds);
    }

    /**
     * Удаляет неиспользуемые визитки и возвращает Set id успешно удалённых визиток.
     *
     * @return Набор id успешно удалённых визиток.
     */
    public Set<Long> deleteUnusedVcards(int shard, @Nullable Long clientUid, Collection<Long> vcardIds) {
        deleteVcardsFromVcardsTable(shard, clientUid, vcardIds);

        Set<Long> remainingVcardIdsOfAllUsers = dslContextProvider.ppc(shard)
                .select(VCARDS.VCARD_ID)
                .from(VCARDS)
                .where(VCARDS.VCARD_ID.in(vcardIds))
                .fetchSet(VCARDS.VCARD_ID);

        Set<Long> deletedVcardIds = new HashSet<>(vcardIds);
        deletedVcardIds.removeAll(remainingVcardIdsOfAllUsers);

        return deletedVcardIds;
    }

    private void deleteVcardsFromVcardsTable(int shard, @Nullable Long clientUid, Collection<Long> vcardIds) {
        final String innerTableName = "unused_vcards_inner_table";

        Table<Record> unusedVcardIdsTable = createSelectStepForUnusedVcards(shard, clientUid, vcardIds)
                .asTable(innerTableName);

        Select<Record1<Long>> unusedVcardIdsSelect = dslContextProvider.ppc(shard)
                .select(VCARDS.as(innerTableName).VCARD_ID)
                .from(unusedVcardIdsTable);

        dslContextProvider.ppc(shard)
                .delete(VCARDS)
                .where(VCARDS.VCARD_ID.in(unusedVcardIdsSelect))
                .and(VCARDS.VCARD_ID.in(vcardIds))
                .execute();
    }

    /**
     * Массовая смена владельца визитки
     * Вызывается при смене главного представителя клиента и при удалении передставителей
     *
     * @param shard      шард
     * @param sourceUids - список uid-ов, которые нужно поменять
     * @param targetUid  - uid нового владельца визитки (проставляется chiefUid)
     */
    public void updateVcardUids(int shard, Collection<Long> sourceUids, Long targetUid) {
        if (sourceUids.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard)
                .update(VCARDS)
                .set(VCARDS.UID, targetUid)
                .where(VCARDS.UID.in(sourceUids))
                .execute();
    }

    /**
     * Массовая смена владельца организации
     * Вызывается при смене главного представителя клиента и при удалении передставителей
     */
    public void updateOrgDetailsUids(int shard, Collection<Long> sourceUids, Long targetUid) {
        dslContextProvider.ppc(shard)
                .update(ORG_DETAILS)
                .set(ORG_DETAILS.UID, targetUid)
                .where(ORG_DETAILS.UID.in(sourceUids))
                .execute();
    }

    private SelectConditionStep<Record> createSelectStepForUnusedVcards(
            int shard, @Nullable Long clientUid, Collection<Long> vcardIds) {
        return dslContextProvider.ppc(shard)
                .select(singletonList(VCARDS.VCARD_ID))
                .from(VCARDS)
                .leftJoin(BANNERS).on(BANNERS.VCARD_ID.eq(VCARDS.VCARD_ID))
                .leftJoin(MEDIAPLAN_BANNERS).on(MEDIAPLAN_BANNERS.VCARD_ID.eq(VCARDS.VCARD_ID))
                .where(clientUid != null ? VCARDS.UID.eq(clientUid) : DSL.trueCondition())
                .and(VCARDS.VCARD_ID.in(vcardIds))
                .and(BANNERS.BID.isNull())
                .and(MEDIAPLAN_BANNERS.MBID.isNull());
    }

    private JooqMapperWithSupplier<Vcard> createVcardMapper() {
        return JooqMapperWithSupplierBuilder.builder(Vcard::new)
                .map(property(Vcard.ID, VCARDS.VCARD_ID))
                .map(property(Vcard.CAMPAIGN_ID, VCARDS.CID))
                .map(property(Vcard.UID, VCARDS.UID))
                .map(property(Vcard.LAST_CHANGE, VCARDS.LAST_CHANGE))
                .map(property(Vcard.LAST_DISSOCIATION, VCARDS.LAST_DISSOCIATION))
                .map(property(Vcard.COMPANY_NAME, VCARDS.NAME))
                .map(property(Vcard.CONTACT_PERSON, VCARDS.CONTACTPERSON))
                .map(property(Vcard.EMAIL, VCARDS.CONTACT_EMAIL))
                .map(convertibleProperty(Vcard.PHONE, VCARDS.PHONE, VcardMappings::phoneFromDb,
                        VcardMappings::phoneToDb))
                .readProperty(Vcard.INSTANT_MESSENGER, fromFields(VCARDS.IM_CLIENT, VCARDS.IM_LOGIN)
                        .by(VcardMappings::instantMessengerByClientAndLoginFromDb))
                .writeField(VCARDS.IM_CLIENT, fromProperty(Vcard.INSTANT_MESSENGER)
                        .by(VcardMappings::clientByImToDb))
                .writeField(VCARDS.IM_LOGIN, fromProperty(Vcard.INSTANT_MESSENGER)
                        .by(VcardMappings::loginByImToDb))
                .map(property(Vcard.GEO_ID, VCARDS.GEO_ID))
                .map(property(Vcard.ADDRESS_ID, VCARDS.ADDRESS_ID))
                .map(property(Vcard.COUNTRY, VCARDS.COUNTRY))
                .map(property(Vcard.CITY, VCARDS.CITY))
                .map(property(Vcard.STREET, VCARDS.STREET))
                .map(property(Vcard.HOUSE, VCARDS.HOUSE))
                .map(property(Vcard.BUILD, VCARDS.BUILD))
                .map(property(Vcard.APART, VCARDS.APART))
                .map(property(Vcard.METRO_ID, VCARDS.METRO))
                .readProperty(Vcard.MANUAL_POINT, fromField(ADDRESSES.MAP_ID)
                        .by(VcardMappings::pointOnMapByIdFromDb))
                .readProperty(Vcard.AUTO_POINT, fromField(ADDRESSES.MAP_ID_AUTO)
                        .by(VcardMappings::pointOnMapByIdFromDb))
                .readProperty(Vcard.POINT_TYPE, fromField(ADDRESSES.KIND)
                        .by(PointType::fromSource))
                .readProperty(Vcard.PRECISION, fromField(ADDRESSES.PRECISION)
                        .by(PointPrecision::fromSource))
                .map(property(Vcard.WORK_TIME, VCARDS.WORKTIME))
                .map(property(Vcard.EXTRA_MESSAGE, VCARDS.EXTRA_MESSAGE))
                .map(property(Vcard.ORG_DETAILS_ID, VCARDS.ORG_DETAILS_ID))
                .readProperty(Vcard.OGRN, fromField(ORG_DETAILS.OGRN))
                .build();
    }

    private Field<String> createMd5Field() {
        return md5(DSL.concat(
                concat(ifnull(cast(VCARDS.UID, String.class), PLACEHOLDER), SEPARATOR),
                concat(ifnull(cast(VCARDS.CID, String.class), PLACEHOLDER), SEPARATOR),
                concat(ifnull(cast(VCARDS.ADDRESS_ID, String.class), PLACEHOLDER), SEPARATOR),
                concat(ifnull(cast(VCARDS.GEO_ID, String.class), PLACEHOLDER), SEPARATOR),
                concat(ifnull(cast(VCARDS.ORG_DETAILS_ID, String.class), PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.NAME, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.COUNTRY, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.CITY, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.STREET, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.HOUSE, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.BUILD, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.APART, PLACEHOLDER), SEPARATOR),
                concat(ifnull(cast(VCARDS.METRO, String.class), PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.CONTACTPERSON, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.CONTACT_EMAIL, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.PHONE, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.IM_CLIENT, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.IM_LOGIN, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.WORKTIME, PLACEHOLDER), SEPARATOR),
                concat(ifnull(VCARDS.EXTRA_MESSAGE, PLACEHOLDER), SEPARATOR)));
    }

    private String calcVcardMd5Hash(Vcard vcard) {
        String str = "" +
                nvl(vcard.getUid(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getCampaignId(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getAddressId(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getGeoId(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getOrgDetailsId(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getCompanyName(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getCountry(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getCity(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getStreet(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getHouse(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getBuild(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getApart(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getMetroId(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getContactPerson(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getEmail(), PLACEHOLDER) + SEPARATOR +
                nvl(phoneToDb(vcard.getPhone()), PLACEHOLDER) + SEPARATOR +
                (vcard.getInstantMessenger() != null && vcard.getInstantMessenger().getType() != null ?
                        vcard.getInstantMessenger().getType() : PLACEHOLDER) + SEPARATOR +
                (vcard.getInstantMessenger() != null && vcard.getInstantMessenger().getLogin() != null ?
                        vcard.getInstantMessenger().getLogin() : PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getWorkTime(), PLACEHOLDER) + SEPARATOR +
                nvl(vcard.getExtraMessage(), PLACEHOLDER) + SEPARATOR;
        return getMd5HashAsHexString(str.getBytes(StandardCharsets.UTF_8));
    }
}
