package ru.yandex.direct.core.entity.vcard.repository.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BinaryOperator;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Field;
import org.jooq.types.ULong;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.TextUtils;
import ru.yandex.direct.core.entity.vcard.model.PointOnMap;
import ru.yandex.direct.core.entity.vcard.model.PointPrecision;
import ru.yandex.direct.core.entity.vcard.model.PointType;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.dbschema.ppc.tables.records.AddressesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.AddedModelId;

import static java.time.LocalDateTime.now;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.common.util.TextUtils.smartStrip;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDRESSES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.HashingUtils.getMd5HalfHashUtf8;

@Repository
public class AddressesRepository {

    public final JooqMapperWithSupplier<DbAddress> addressMapper;
    private final Collection<Field<?>> fieldsToRead;

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public AddressesRepository(DslContextProvider dslContextProvider,
                               ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.addressMapper = createAddressMapper();

        this.fieldsToRead = addressMapper.getFieldsToRead();
    }

    public Map<Long, DbAddress> getAddresses(int shard, List<Long> addressesIds) {
        if (addressesIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ADDRESSES)
                .where(ADDRESSES.AID.in(addressesIds))
                .fetchMap(ADDRESSES.AID, addressMapper::fromDb);
    }

    /**
     * Для всех переданных визиток ищутся соответствующие записи в базе.
     * Если запись найдена, то у неё обновляются точки, если нет - создается новая запись.
     */
    public List<AddedModelId> getOrCreateAddresses(int shard, ClientId clientId, List<Vcard> vcards) {
        List<DbAddress> addresses = mapList(vcards, vcard -> vcardToDbAddress(vcard, clientId));

        Map<String, DbAddress> existingAddresses = getExistingAddresses(shard, clientId, addresses);

        int idCount = (int) StreamEx.of(addresses).map(DbAddress::getAddress).distinct().count()
                - existingAddresses.size();
        Iterator<Long> ids = shardHelper.generateAddressIds(idCount).iterator();

        // хэлпер для создания новых записей (или обновления существующих с помощью on duplicate key update)
        InsertHelper<AddressesRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), ADDRESSES);

        // список AppliedChanges для обновления существующих записей
        List<AppliedChanges<DbAddress>> appliedChangesList = new ArrayList<>(vcards.size());

        // мапа для запоминания, какие адреса уже подготовлены к созданию в InsertHelper'е
        Map<String, DbAddress> readyForInsertAddresses = new HashMap<>(vcards.size());
        Set<Long> changedAddressesIds = new HashSet<>(vcards.size());

        /*
            Так как insert выполняется с on duplicate key update,
            сгенерированные id нам не помогут найти соответствующие записи,
            поэтому они не запоминаются. Вместо этого, после выполнения
            запроса осуществляется повторный поиск записей.
         */
        for (DbAddress addressToSave : addresses) {
            // Мы используем 0 (т.е. не null), для того чтобы сказать что адрес не надо сохранять
            // (См. CommonMaps.pm::_save_address)
            if (addressToSave.getId() != null && addressToSave.getId() == 0) {
                continue;
            }

            String addressStr = addressToSave.getAddress();

            /*
                Если такой адрес не существовал, но мы уже подготовили его к созданию
                в одной из предыдущих итераций, то продолжаем
             */
            DbAddress addressReadyToInsert = readyForInsertAddresses.get(addressStr);
            if (addressReadyToInsert != null) {
                continue;
            }

            /*
                Если такой адрес уже существует в БД, то обновляем тип точки и точность,
                а если нет, то генерируем id и создаем новую запись
             */
            DbAddress existingAddress = existingAddresses.get(addressStr);
            if (existingAddress != null) {
                long existingId = existingAddress.getId();

                ModelChanges<DbAddress> addressChanges = new ModelChanges<>(existingId, DbAddress.class);
                addressChanges.process(addressToSave.getMapId(), DbAddress.MAP_ID);
                addressChanges.process(addressToSave.getMapIdAuto(), DbAddress.MAP_ID_AUTO);
                addressChanges.process(addressToSave.getPointType(), DbAddress.POINT_TYPE);
                addressChanges.process(addressToSave.getPrecision(), DbAddress.PRECISION);

                AppliedChanges<DbAddress> appliedChanges = addressChanges.applyTo(existingAddress);
                if (appliedChanges.hasActuallyChangedProps()) {
                    appliedChanges.modify(DbAddress.LAST_CHANGE, now());
                    appliedChangesList.add(appliedChanges);
                    changedAddressesIds.add(existingId);
                }
            } else {
                // такого адреса в базе нет, генерируем id и добавляем в InsertHelper
                addressToSave.setId(ids.next());

                // запоминаем, что адрес уже подготовили к созданию
                readyForInsertAddresses.put(addressStr, addressToSave);

                insertHelper.add(addressMapper, addressToSave)
                        .newRecord();
            }

        }

        JooqUpdateBuilder<AddressesRecord, DbAddress> updateBuilder =
                new JooqUpdateBuilder<>(ADDRESSES.AID, appliedChangesList);
        updateBuilder.processProperty(DbAddress.MAP_ID, ADDRESSES.MAP_ID);
        updateBuilder.processProperty(DbAddress.MAP_ID_AUTO, ADDRESSES.MAP_ID_AUTO);
        updateBuilder.processProperty(DbAddress.POINT_TYPE, ADDRESSES.KIND, PointType::toSource);
        updateBuilder.processProperty(DbAddress.PRECISION, ADDRESSES.PRECISION, PointPrecision::toSource);
        updateBuilder.processProperty(DbAddress.LAST_CHANGE, ADDRESSES.LOGTIME, localDateTime -> localDateTime);

        if (insertHelper.hasAddedRecords()) {
            /*
                Если с тех пор как мы проверяли существование адреса,
                он был создан в параллельном потоке, то просто апдейтим
                существующую запись.
             */
            insertHelper.onDuplicateKeyUpdate()
                    .set(ADDRESSES.MAP_ID, MySQLDSL.values(ADDRESSES.MAP_ID))
                    .set(ADDRESSES.MAP_ID_AUTO, MySQLDSL.values(ADDRESSES.MAP_ID_AUTO))
                    .set(ADDRESSES.KIND, MySQLDSL.values(ADDRESSES.KIND))
                    .set(ADDRESSES.PRECISION, MySQLDSL.values(ADDRESSES.PRECISION))
                    .set(ADDRESSES.LOGTIME, MySQLDSL.values(ADDRESSES.LOGTIME))
                    .execute();
        }

        if (!updateBuilder.getChangedIds().isEmpty()) {
            dslContextProvider.ppc(shard)
                    .update(ADDRESSES)
                    .set(updateBuilder.getValues())
                    .where(ADDRESSES.AID.in(updateBuilder.getChangedIds()))
                    .execute();
        }


        /*
            Нам нужно хранить информацию только о том, изменился ли существующий адрес. Поэтому, если
            адрес не изменился, возвращаем ofUnknown.
         */
        return findIdsOfAllAddressesOrFail(shard, clientId, addresses).stream()
                .map(id -> id == null ? null :
                        changedAddressesIds.contains(id) ? AddedModelId.ofChanged(id) : AddedModelId.ofUnknown(id))
                .collect(toList());
    }

    private List<Long> findIdsOfAllAddressesOrFail(int shard, ClientId clientId, List<DbAddress> addresses) {
        Map<String, Long> createdAddressesIds = getExistingAddressesIds(shard, clientId, addresses);
        return mapList(
                addresses,
                // Мы используем 0 (т.е. не null), для того чтобы сказать что адрес не надо сохранять, потому что он
                // некорректен. Для таких адресов в качестве address_id в таблице vcard должен быть NULL
                // (См. DIRECT-72073)
                addr -> {
                    if (addr.getId() != null && addr.getId() == 0) {
                        return null;
                    }
                    Long addressId = createdAddressesIds.get(addr.getAddress());
                    if (addressId == null) {
                        throw new IllegalStateException(
                                String.format(
                                        "Не удалось найти идентификатор для адреса %s", addr.getAddress()));
                    }
                    return addressId;
                });
    }

    private Map<String, DbAddress> getExistingAddresses(int shard, ClientId clientId, List<DbAddress> addresses) {
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ADDRESSES)
                .where(ADDRESSES.CLIENT_ID.eq(clientId.asLong()))
                .and(ADDRESSES.ADDRESS.in(
                        addresses.stream()
                                .map(DbAddress::getAddress)
                                .collect(toList())))
                .stream()
                .collect(
                        toMap(
                                rec -> smartStrip(rec.getValue(ADDRESSES.ADDRESS)),
                                addressMapper::fromDb,
                                // Берем адрес с минимальным id в случае конфликта hash-ей
                                BinaryOperator.minBy(Comparator.comparing(DbAddress::getId))));
    }

    private Map<String, Long> getExistingAddressesIds(int shard, ClientId clientId, List<DbAddress> addresses) {
        return dslContextProvider.ppc(shard)
                .select(asList(ADDRESSES.AID, ADDRESSES.ADDRESS))
                .from(ADDRESSES)
                .where(ADDRESSES.CLIENT_ID.eq(clientId.asLong()))
                .and(ADDRESSES.ADDRESS.in(mapList(addresses, DbAddress::getAddress)))
                .stream()
                .collect(
                        toMap(
                                rec -> smartStrip(rec.getValue(ADDRESSES.ADDRESS)),
                                rec -> rec.getValue(ADDRESSES.AID),
                                // Берем адрес с минимальным id в случае конфликта hash-ей
                                BinaryOperator.minBy(Comparator.<Long>naturalOrder())));
    }

    DbAddress vcardToDbAddress(Vcard vcard, ClientId clientId) {
        List<String> addressElements =
                asList(vcard.getCountry(), vcard.getCity(), vcard.getStreet(), vcard.getHouse(), vcard.getBuild());
        addressElements = filterList(addressElements, Objects::nonNull);
        String address = smartStrip(StringUtils.lowerCase(StringUtils.join(addressElements, ",")));
        return new DbAddress()
                .withId(vcard.getAddressId())
                .withClientId(clientId.asLong())
                .withAddress(address)
                .withAddressHash(getMd5HalfHashUtf8(address))
                .withMetroId(vcard.getMetroId())
                .withMapId(getIfNotNull(vcard.getManualPoint(), PointOnMap::getId, null))
                .withMapIdAuto(getIfNotNull(vcard.getAutoPoint(), PointOnMap::getId, null))
                .withPointType(vcard.getPointType())
                .withPrecision(vcard.getPrecision())
                .withLastChange(now());
    }

    private <T, V> V getIfNotNull(T value, Function<T, V> fn, V defaultValue) {
        return value != null ? fn.apply(value) : defaultValue;
    }

    private JooqMapperWithSupplier<DbAddress> createAddressMapper() {
        return JooqMapperWithSupplierBuilder.builder(DbAddress::new)
                .map(property(DbAddress.ID, ADDRESSES.AID))
                .map(property(DbAddress.CLIENT_ID, ADDRESSES.CLIENT_ID))
                .map(property(DbAddress.LAST_CHANGE, ADDRESSES.LOGTIME))
                .map(property(DbAddress.MAP_ID, ADDRESSES.MAP_ID))
                .map(property(DbAddress.MAP_ID_AUTO, ADDRESSES.MAP_ID_AUTO))
                .map(convertibleProperty(DbAddress.ADDRESS, ADDRESSES.ADDRESS, TextUtils::smartStrip, TextUtils::smartStrip))
                .map(convertibleProperty(DbAddress.ADDRESS_HASH, ADDRESSES.AHASH, ULong::toBigInteger, ULong::valueOf))
                .map(property(DbAddress.METRO_ID, ADDRESSES.METRO))
                .map(convertibleProperty(DbAddress.POINT_TYPE, ADDRESSES.KIND, PointType::fromSource, PointType::toSource))
                .map(convertibleProperty(DbAddress.PRECISION, ADDRESSES.PRECISION, PointPrecision::fromSource, PointPrecision::toSource))
                .build();
    }
}
