package ru.yandex.direct.core.entity.vcard.repository.internal;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.dbschema.ppc.tables.records.OrgDetailsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Arrays.asList;
import static ru.yandex.direct.dbschema.ppc.Tables.ORG_DETAILS;
import static ru.yandex.direct.dbschema.ppc.Tables.VCARDS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class OrgDetailsRepository {

    private final JooqMapperWithSupplier<DbOrgDetails> orgDetailsMapper;

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public OrgDetailsRepository(DslContextProvider dslContextProvider,
                                ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.orgDetailsMapper = createOrgDetailsMapper();
    }

    public void deleteClientOrgDetails(DSLContext dslContext, Collection<Long> orgDetailIds) {
        dslContext
                .delete(ORG_DETAILS)
                .where(ORG_DETAILS.ORG_DETAILS_ID.in(orgDetailIds))
                .execute();
    }

    public List<Long> getUnusedClientOrgDetails(DSLContext dslContext, long clientUid) {
        return dslContext
                .select(ORG_DETAILS.ORG_DETAILS_ID)
                .from(ORG_DETAILS)
                .leftJoin(VCARDS)
                .on(ORG_DETAILS.UID.eq(VCARDS.UID).and(ORG_DETAILS.ORG_DETAILS_ID.eq(VCARDS.ORG_DETAILS_ID)))
                .where(ORG_DETAILS.UID.eq(clientUid).and(VCARDS.ORG_DETAILS_ID.isNull()))
                .fetch(ORG_DETAILS.ORG_DETAILS_ID);

    }

    public Map<Long, DbOrgDetails> getOrgDetails(int shard, long clientUid, Collection<Long> orgDetailsIds) {
        return dslContextProvider.ppc(shard)
                .select(asList(ORG_DETAILS.ORG_DETAILS_ID, ORG_DETAILS.UID, ORG_DETAILS.OGRN))
                .from(ORG_DETAILS)
                .where(ORG_DETAILS.UID.eq(clientUid))
                .and(ORG_DETAILS.ORG_DETAILS_ID.in(orgDetailsIds))
                .fetchMap(ORG_DETAILS.ORG_DETAILS_ID, orgDetailsMapper::fromDb);
    }

    public List<Long> getOrCreateOrgDetails(int shard, long clientUid, ClientId clientId, List<Vcard> vcards) {
        List<String> ogrns = mapList(vcards, Vcard::getOgrn);

        Set<String> ogrnsNonNull = StreamEx.of(ogrns).nonNull().toSet();
        Map<String, Long> existingOrgDetailsIds = getExistingOrgDetailsIdsByOgrns(shard, clientUid, ogrnsNonNull);

        int idCount = ogrnsNonNull.size() - existingOrgDetailsIds.size();
        Iterator<Long> ids = shardHelper.generateOrgDetailsIds(clientId.asLong(), idCount).iterator();

        InsertHelper<OrgDetailsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), ORG_DETAILS);

        List<Long> orgDetailsIds = mapList(ogrns, ogrn -> {
            if (ogrn == null) {
                return null;
            }

            Long orgDetailsId = existingOrgDetailsIds.get(ogrn);
            if (orgDetailsId != null) {
                return orgDetailsId;
            }

            orgDetailsId = ids.next();

            // добавляем в мапу существующих id только что сгенерированный, чтобы избежать дублирования
            existingOrgDetailsIds.put(ogrn, orgDetailsId);

            DbOrgDetails dbOrgDetails = new DbOrgDetails()
                    .withId(orgDetailsId)
                    .withOgrn(ogrn)
                    .withUid(clientUid);
            insertHelper.add(orgDetailsMapper, dbOrgDetails).newRecord();

            return orgDetailsId;
        });

        insertHelper.executeIfRecordsAdded();

        return orgDetailsIds;
    }

    private Map<String, Long> getExistingOrgDetailsIdsByOgrns(int shard, long uid, Collection<String> ogrns) {
        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(ORG_DETAILS.ORG_DETAILS_ID, ORG_DETAILS.OGRN))
                .from(ORG_DETAILS)
                .where(ORG_DETAILS.UID.eq(uid))
                .and(ORG_DETAILS.OGRN.in(ogrns))
                .fetch();
        return StreamEx.of(result)
                .distinct(rec -> rec.getValue(ORG_DETAILS.OGRN))
                .mapToEntry(rec -> rec.getValue(ORG_DETAILS.ORG_DETAILS_ID))
                .mapKeys(rec -> rec.getValue(ORG_DETAILS.OGRN))
                .toMap();
    }

    private JooqMapperWithSupplier<DbOrgDetails> createOrgDetailsMapper() {
        return JooqMapperWithSupplierBuilder.builder(DbOrgDetails::new)
                .map(property(DbOrgDetails.ID, ORG_DETAILS.ORG_DETAILS_ID))
                .map(property(DbOrgDetails.UID, ORG_DETAILS.UID))
                .map(property(DbOrgDetails.OGRN, ORG_DETAILS.OGRN))
                .build();
    }
}
