package ru.yandex.direct.core.entity.vcard.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService;
import ru.yandex.direct.core.entity.vcard.service.validation.VcardDefects;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.operation.AddedModelId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.AbstractAddOperation;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.operation.OperationsUtils.fillListFromMap;
import static ru.yandex.direct.operation.OperationsUtils.recoverMapFromList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Следующие поля визитки являются пользовательскими и могут быть заполнены в передаваемых визитках
 * (остальные поля считаются служебными и будут проигнорированы):
 * campaignId, companyName, workTime, contactPerson, phone, email, instantMessenger, extraMessage, ogrn,
 * geoId, country, city, street, house, build, apart, metroId, manualPoint.
 */
public class AddVcardOperation extends AbstractAddOperation<Vcard, Long> {

    private final AddVcardValidationService validationService;
    private final VcardRepository vcardRepository;
    private final VcardHelper vcardHelper;
    private final BannerCommonRepository bannerRepository;
    private final Long operatorUid;
    private final UidClientIdShard client;

    private final boolean partOfComplexOperation;

    public AddVcardOperation(Applicability applicability, boolean partOfComplexOperation,
                             List<Vcard> models, AddVcardValidationService validationService,
                             VcardRepository vcardRepository, VcardHelper vcardHelper,
                             Long operatorUid, UidClientIdShard client,
                             BannerCommonRepository bannerRepository) {
        super(applicability, models);
        this.validationService = validationService;
        this.vcardRepository = vcardRepository;
        this.vcardHelper = vcardHelper;
        this.operatorUid = operatorUid;
        this.client = client;
        this.bannerRepository = bannerRepository;

        this.partOfComplexOperation = partOfComplexOperation;
    }

    @Override
    protected void validate(ValidationResult<List<Vcard>, Defect> preValidationResult) {
        new ItemValidationBuilder<>(preValidationResult)
                .checkBy(vcards -> validationService.validate(
                        operatorUid, !partOfComplexOperation, vcards, client));
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<Vcard> modelsValidatedStep) {
        prepareSystemFields(modelsValidatedStep.getValidModelsMap().values());
    }

    @Override
    protected Map<Integer, Long> execute(Map<Integer, Vcard> validModelsMapToApply) {
        List<Vcard> validModels = new ArrayList<>();
        Map<Integer, Integer> localToSourceIndexMap = fillListFromMap(validModels, validModelsMapToApply);
        List<AddedModelId> addedModelIds =
                vcardRepository.addVcards(client.getShard(), client.getUid(), client.getClientId(), validModels);
        List<Long> changedVcardIds = new ArrayList<>(addedModelIds.size());
        for (int i = 0; i < addedModelIds.size(); ++i) {
            if (addedModelIds.get(i) != null && !addedModelIds.get(i).isAdded()) {
                validationResult.getSubResults()
                        .get(index(localToSourceIndexMap.get(i)))
                        .addWarning(VcardDefects.vcardIsDuplicated());
                if (addedModelIds.get(i).isChanged()) {
                    changedVcardIds.add(addedModelIds.get(i).getId());
                }
            }
        }
        bannerRepository.updateStatusBsSyncedAndLastChangeByVcardId(client.getShard(), changedVcardIds,
                StatusBsSynced.NO);
        List<Long> resultIds = mapList(addedModelIds, AddedModelId::getId);
        return recoverMapFromList(resultIds, localToSourceIndexMap);
    }

    private void prepareSystemFields(Collection<Vcard> vcards) {
        VcardHelper.fillSystemDateTimeFields(vcards);
        vcards.forEach(vcard -> {
            vcard.withId(null)
                    .withUid(client.getUid())
                    .withAddressId(null)
                    .withOrgDetailsId(null);

            if (vcard.getManualPoint() != null) {
                vcard.getManualPoint().withId(null);
            }

            if (vcard.getAutoPoint() != null) {
                vcard.getAutoPoint().withId(null);
            }
        });

        vcardHelper.fillVcardsWithRegionIds(vcards);
        vcardHelper.fillVcardsWithGeocoderData(vcards);
    }

}
