package ru.yandex.direct.core.entity.vcard.service;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.geo.model.GeoRegionType;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.Metro;
import ru.yandex.direct.regions.Region;

import static java.util.Arrays.asList;

/**
 * Отвечает за поиск региона по названию. Специфичен для визитки.
 * Инкапсулирует порядок поиска по языкам, правила обработки регионов с одинаковыми именами.
 * <p>
 * На данный момент для ситуации, когда существует несколько городов,
 * имеющих одинаковые имена на одном языке, методы поиска ничего не возвращают,
 * чтобы не было неоднозначности.
 */
@Service
@ParametersAreNonnullByDefault
public class GeoRegionLookup {

    private final Map<String, Region> ruCityMap;
    private final Map<String, Region> enCityMap;
    private final Map<String, Region> uaCityMap;
    private final Map<String, Region> trCityMap;

    private final List<Map<String, Region>> mapsList;
    private final List<Map<String, Region>> countryMapsList;
    private final Map<Long, String> metroNames;

    @Autowired
    public GeoRegionLookup(GeoTreeFactory geoTreeFactory) {
        GeoTree geoTree = geoTreeFactory.getGlobalGeoTree();
        List<Region> cities = EntryStream.of(geoTree.getRegions())
                .filterValues(region -> region.getType() == GeoRegionType.CITY.getTypedValue())
                .values()
                .toList();

        ruCityMap = getMapWithUniqueCities(cities, Region::getNameRu);
        enCityMap = getMapWithUniqueCities(cities, Region::getNameEn);
        uaCityMap = getMapWithUniqueCities(cities, Region::getNameUa);
        trCityMap = getMapWithUniqueCities(cities, Region::getNameTr);

        mapsList = asList(ruCityMap, enCityMap, uaCityMap, trCityMap);

        List<Region> countries = EntryStream.of(geoTree.getRegions())
                .filterValues(region -> region.getType() == GeoRegionType.COUNTRY.getTypedValue())
                .values()
                .toList();

        Map<String, Region> ruCountryMap = getMapWithUniqueCities(countries, Region::getNameRu);
        Map<String, Region> enCountryMap = getMapWithUniqueCities(countries, Region::getNameEn);
        Map<String, Region> uaCountryMap = getMapWithUniqueCities(countries, Region::getNameUa);
        Map<String, Region> trCountryMap = getMapWithUniqueCities(countries, Region::getNameTr);

        countryMapsList = asList(ruCountryMap, enCountryMap, uaCountryMap, trCountryMap);

        metroNames = EntryStream.of(geoTree.getMetroMap()).mapValues(Metro::getName).toMap();
    }

    private Map<String, Region> getMapWithUniqueCities(List<Region> cities, Function<Region, String> getNameFn) {
        Set<String> duplicatedCities = new HashSet<>();
        Map<String, Region> map = new HashMap<>(cities.size());

        for (Region city : cities) {
            String name = getNameFn.apply(city);
            if (map.put(name, city) != null) {
                duplicatedCities.add(name);
            }
        }

        duplicatedCities.forEach(map::remove);

        return ImmutableMap.copyOf(map);
    }

    @Nullable
    public Region getRegionByCity(String city) {
        return getRegionByName(city, mapsList);
    }

    @Nullable
    private Region getRegionByName(String name, List<Map<String, Region>> listOfMaps) {
        for (Map<String, Region> map : listOfMaps) {
            Region region = map.get(name);
            if (region != null) {
                return region;
            }
        }
        return null;
    }

    @Nonnull
    public Map<String, Long> getRegionIdsByCities(Set<String> cities) {
        return getRegionIdsByNames(cities, mapsList);
    }

    /**
     * Возвращает region_id стран по их названиям
     *
     * @param countries названия стран
     * @return map название страны - region_id
     */
    @Nonnull
    public Map<String, Long> getRegionIdsByCountries(Set<String> countries) {
        return getRegionIdsByNames(countries, countryMapsList);
    }

    @Nonnull
    private Map<String, Long> getRegionIdsByNames(Set<String> names, List<Map<String, Region>> listOfMaps) {
        Map<String, Long> regionByNames = new HashMap<>(names.size());
        for (String name : names) {
            Region region = getRegionByName(name, listOfMaps);
            if (region != null) {
                regionByNames.put(name, region.getId());
            }
        }
        return regionByNames;
    }

    /**
     * Возвращает названия метро по их id
     *
     * @param metroIds id-ки метро
     * @return map id метро - название
     */
    public Map<Long, String> getMetroNamesByIds(Set<Long> metroIds) {
        return StreamEx.of(metroIds).mapToEntry(metroNames::get).nonNullValues().toMap();
    }
}
