package ru.yandex.direct.core.entity.vcard.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.container.AdsSelectionCriteria;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.DynamicBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.vcard.container.AssignVcardRequest;
import ru.yandex.direct.core.entity.vcard.container.SaveVcardRequest;
import ru.yandex.direct.core.entity.vcard.container.UnassignVcardRequest;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentBannerCidAndCampaignId;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredButEmptyHrefOrTurboOrVcardIdOrPermalink;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.unsupportedBannerType;
import static ru.yandex.direct.core.entity.vcard.service.validation.VcardDefects.bannersWithVcardIdNotFound;
import static ru.yandex.direct.core.entity.vcard.service.validation.VcardDefects.inconsistentVcardCidAndCampaignId;
import static ru.yandex.direct.core.entity.vcard.service.validation.VcardDefects.vcardHasValidationErrors;
import static ru.yandex.direct.multitype.entity.LimitOffset.maxLimited;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.minListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.absentRequiredField;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

/**
 * Для страницы "Мастер заполнения виртуальных визиток"
 */
@Service
@ParametersAreNonnullByDefault
public class ManageVcardService {
    private final RbacService rbacService;
    private final VcardService vcardService;
    private final ShardHelper shardHelper;
    private final VcardRepository vcardRepository;
    private final BannerService bannerService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final FeatureService featureService;

    private static final Set<Class<?>> SUPPORTED_BANNER_TYPES =
            ImmutableSet.of(TextBanner.class, DynamicBanner.class);


    @Autowired
    public ManageVcardService(
            RbacService rbacService,
            VcardService vcardService,
            ShardHelper shardHelper,
            VcardRepository vcardRepository,
            BannerService bannerService,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            FeatureService featureService) {
        this.rbacService = rbacService;
        this.vcardService = vcardService;
        this.shardHelper = shardHelper;
        this.vcardRepository = vcardRepository;
        this.bannerService = bannerService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.featureService = featureService;
    }

    /**
     * Сохранить визитку и назначить ее баннерам
     * <p>
     * Параметры запроса (request)
     * campaignId -- обязательно (номер кампании, баннерам в которой назначится новая визитка)
     * bannerIds -- опционально. Список id объявлений, которым надо назначить новую визитку
     * (должны относится к одной кампании с номером campaignId)
     * vcardId -- опционально. Если указан, то баннерам с таким vcardId и campaignId назначится новая визитка
     * (визитка должна относиться к кампании campaignId)
     * vcard -- новые значения для полей визитки
     * (должно быть указано либо bannerIds либо vcardId)
     *
     * @param operatorUid Uid оператора
     * @param clientId    Идентификатор клиента
     * @param request     запрос (содержит параметры campaignId, bannerIds, vcardId, vcard)
     * @return id сохранненой визитки (или ошибки валидации)
     */
    public Result<Long> saveVcard(Long operatorUid, ClientId clientId,
                                  SaveVcardRequest request) {
        if (request.getBannerIds() != null) {
            return saveVcardByBannerIds(operatorUid, clientId, request);
        } else {
            return saveVcardByVcardId(operatorUid, clientId, request);
        }
    }

    private Result<Long> saveVcardByBannerIds(Long operatorUid, ClientId clientId,
                                              SaveVcardRequest request) {
        ValidationResult<SaveVcardRequest, Defect> preValidationResult =
                preValidateSaveVcardRequestByBannerIds(request);
        if (preValidationResult.hasAnyErrors()) {
            return Result.broken(preValidationResult);
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Long clientChiefUid = rbacService.getChiefByClientId(clientId);
        UidClientIdShard client = UidClientIdShard.of(clientChiefUid, clientId, shard);

        ValidationResult<SaveVcardRequest, Defect> validationResult =
                validateSaveVcardRequestByBannerIds(operatorUid, client, request);

        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        Long campaignId = request.getCampaignId();
        List<Long> bannerIds = request.getBannerIds();

        // тк все баннеры относятся к одной кампании, то нужно создать только одну визитку
        // если визитка с такими полями уже есть в БД новая не должна создаваться
        Vcard vcard = request.getVcard().withCampaignId(campaignId);
        Result<Long> addVcardResult = addVcardToBanners(shard, operatorUid, clientId, bannerIds, vcard);
        ValidationResult<?, Defect> vcardValidationResult = addVcardResult.getValidationResult();

        Result<Long> operationResult;
        if (vcardValidationResult != null && vcardValidationResult.hasAnyErrors()) {
            ModelItemValidationBuilder<SaveVcardRequest> vcardVB = ModelItemValidationBuilder.of(request);
            //noinspection unchecked
            vcardVB.item(SaveVcardRequest.VCARD)
                    .checkBy(v -> (ValidationResult<Vcard, Defect>) vcardValidationResult);
            operationResult = Result.broken(vcardVB.getResult());
        } else {
            operationResult = Result.successful(addVcardResult.getResult());
        }
        return operationResult;
    }

    private Result<Long> saveVcardByVcardId(Long operatorUid, ClientId clientId,
                                            SaveVcardRequest request) {
        ValidationResult<SaveVcardRequest, Defect> preValidationResult =
                preValidateSaveVcardRequestByVcardId(request);
        if (preValidationResult.hasAnyErrors()) {
            return Result.broken(preValidationResult);
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Long clientChiefUid = rbacService.getChiefByClientId(clientId);
        UidClientIdShard client = UidClientIdShard.of(clientChiefUid, clientId, shard);

        Map<Long, Long> bannerIdsToVcardIds = vcardRepository.getBannerIdsToVcardIds(shard, clientChiefUid,
                null, singletonList(request.getVcardId()));
        List<Long> bannerIds = new ArrayList<>(bannerIdsToVcardIds.keySet());

        Long campaignId = request.getCampaignId();
        ValidationResult<List<Long>, Defect> bannerIdsValidationResult =
                validateBannerIds(operatorUid, clientId, campaignId, bannerIds, false);
        // будем обновлять визитку только в валидных баннерах с таким vcardId (архивные и тд пропускаем)
        List<Long> validBannerIds = getValidItems(bannerIdsValidationResult);

        // проверим что визитка существует и есть хотя бы один валидный баннер с таким vcardId
        ValidationResult<SaveVcardRequest, Defect> validationResult =
                validateSaveVcardRequestByVcardId(operatorUid, client, request, validBannerIds.isEmpty());
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }

        // тк все баннеры относятся к одной кампании, то нужно создать только одну визитку
        // если визитка с такими полями уже есть в БД новая не должна создаваться
        Vcard vcard = request.getVcard().withCampaignId(campaignId);
        Result<Long> addVcardResult = addVcardToBanners(shard, operatorUid, clientId, validBannerIds, vcard);
        ValidationResult<?, Defect> vcardValidationResult = addVcardResult.getValidationResult();

        Result<Long> operationResult;
        if (vcardValidationResult != null && vcardValidationResult.hasAnyErrors()) {
            ModelItemValidationBuilder<SaveVcardRequest> vcardVB = ModelItemValidationBuilder.of(request);
            //noinspection unchecked
            vcardVB.item(SaveVcardRequest.VCARD)
                    .checkBy(v -> (ValidationResult<Vcard, Defect>) vcardValidationResult);
            operationResult = Result.broken(vcardVB.getResult());
        } else {
            operationResult = Result.successful(addVcardResult.getResult());
        }
        return operationResult;
    }

    /**
     * Назначить визитку объявлениям (одному/нескольким)
     * <p>
     * Параметры запроса (request)
     * campaignId -- номер кампании, баннерам в которой назначится новая визитка
     * bannerIds -- список id объявлений, которым надо назначить визитку;
     * (все должны быть из одной кампании (с номером campaignId))
     * vcardId -- может быть и из той же кампании, и из другой того же пользователя
     *
     * @param operatorUid Uid оператора
     * @param clientId    id клиента
     * @param request     запрос
     * @return id новой визитки (или ошибки валидации)
     */
    public Result<Long> assignVcard(Long operatorUid, ClientId clientId,
                                    AssignVcardRequest request) {
        ValidationResult<AssignVcardRequest, Defect> preValidationResult =
                preValidateAssignVcardRequest(request);
        if (preValidationResult.hasAnyErrors()) {
            return Result.broken(preValidationResult);
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Long clientChiefUid = rbacService.getChiefByClientId(clientId);
        UidClientIdShard client = UidClientIdShard.of(clientChiefUid, clientId, shard);

        Long oldVcardId = request.getVcardId();
        Vcard oldVcard = getVCardById(operatorUid, clientId, oldVcardId);

        ValidationResult<AssignVcardRequest, Defect> validationResult =
                validateAssignVcardRequest(operatorUid, client, request, oldVcard != null);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }

        Long campaignId = request.getCampaignId();
        List<Long> bannerIds = request.getBannerIds();
        checkState(oldVcard != null, "vcard not found");
        Vcard vcard = oldVcard.withCampaignId(campaignId).withLastChange(null).withLastDissociation(null);

        Result<Long> addVcardResult = addVcardToBanners(shard, operatorUid, clientId, bannerIds, vcard);
        ValidationResult<?, Defect> vcardValidationResult = addVcardResult.getValidationResult();

        Result<Long> operationResult;
        if (vcardValidationResult != null && vcardValidationResult.hasAnyErrors()) {
            ModelItemValidationBuilder<AssignVcardRequest> vcardVB = ModelItemValidationBuilder.of(request);
            vcardVB.item(AssignVcardRequest.VCARD_ID).check(fromPredicate(v -> false, vcardHasValidationErrors()));
            operationResult = Result.broken(vcardVB.getResult());
        } else {
            operationResult = Result.successful(addVcardResult.getResult());
        }
        return operationResult;
    }

    /**
     * Удаление визитки из объявлений
     *
     * @param operatorUid Uid оператора
     * @param clientId    id клиента
     * @param request     запрос
     * @return ошибки валидации (если есть)
     */
    public Result<Long> unassignVcard(Long operatorUid, ClientId clientId,
                                      UnassignVcardRequest request) {
        ValidationResult<UnassignVcardRequest, Defect> preValidationResult =
                preValidateUnassignVcardRequest(request);
        if (preValidationResult.hasAnyErrors()) {
            return Result.broken(preValidationResult);
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Long clientChiefUid = rbacService.getChiefByClientId(clientId);
        UidClientIdShard client = UidClientIdShard.of(clientChiefUid, clientId, shard);

        ValidationResult<UnassignVcardRequest, Defect> validationResult =
                validateUnassignVcardRequest(operatorUid, client, request);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }

        List<Long> bannerIds = request.getBannerIds();
        deleteVcardFromBanners(shard, operatorUid, clientId, bannerIds);

        return Result.successful(null);
    }

    /**
     * Получение списка визиток
     * если не переданы id визиток, то будут все визитки пользователя,
     * которые используются хотя бы в одном баннере (в том числе в архивных)
     * если не передан Uid оператора (null), то не будет проверки прав оператора на визитки
     *
     * @param operatorUid uid оператора
     * @param clientId    id клиента
     * @param vcardIds    id визиток
     * @return список визиток
     */
    public List<Vcard> getVcards(@Nullable Long operatorUid, ClientId clientId, @Nullable Collection<Long> vcardIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Long clientChiefUid = rbacService.getChiefByClientId(clientId);

        Collection<Long> vcardIdsToGet = (vcardIds != null) ? vcardIds
                : vcardRepository.getVcardsIdsFromBannersOnly(shard, clientChiefUid);

        List<Vcard> vcards = (operatorUid != null)
                // визитки из кампаний видимых оператору
                ? vcardService.getVcards(operatorUid, clientId, vcardIdsToGet, maxLimited())
                // визитки из любых кампаний
                : vcardRepository.getVcards(shard, clientChiefUid, vcardIdsToGet, maxLimited());

        vcardService.setVcardsAdditionalFields(vcards);
        return vcards;
    }

    /**
     * Получение визитки по id
     *
     * @param operatorUid uid оператора
     * @param clientId    id клиента
     * @param vcardId     id визитки
     * @return визитка (или null если такой визитки нет)
     */
    @Nullable
    public Vcard getVCardById(long operatorUid, ClientId clientId, @Nullable Long vcardId) {
        if (vcardId == null) {
            return null;
        }
        List<Vcard> vcards = getVcards(operatorUid, clientId, singletonList(vcardId));
        return vcards.isEmpty() ? null : vcards.get(0);
    }

    /**
     * сколько раз используются визитки
     *
     * @param clientId id клиента
     * @param vcardIds список id визиток
     * @return map id визитки - количество баннеров с такой визиткой
     */
    public Map<Long, Long> getVcardsUsesCount(ClientId clientId, List<Long> vcardIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Long clientChiefUid = rbacService.getChiefByClientId(clientId);
        return vcardRepository.getVcardsUsesCount(shard, clientChiefUid, vcardIds);
    }

    private Result<Long> addVcardToBanners(int shard, Long operatorUid, ClientId clientId, List<Long> bannerIds,
                                           Vcard vcard) {
        Result<Long> vcardAddResult = vcardService.addVcardsFull(singletonList(vcard),
                operatorUid, clientId).get(0);

        ValidationResult<?, Defect> vcardValidationResult = vcardAddResult.getValidationResult();
        if (vcardValidationResult != null && vcardValidationResult.hasAnyErrors()) {
            return Result.broken(vcardValidationResult);
        }
        Long newVcardId = vcardAddResult.getResult();
        checkState(newVcardId != null, "added vcardId must be valid");

        Map<Long, Long> vcardIdByBannerId = StreamEx.of(bannerIds).mapToEntry(bannerId -> newVcardId).toMap();
        bannerService.updateBannerVcardIds(shard, operatorUid, clientId, bannerIds, vcardIdByBannerId);
        return Result.successful(newVcardId);
    }

    private void deleteVcardFromBanners(int shard, Long operatorUid, ClientId clientId, List<Long> bannerIds) {
        bannerService.updateBannerVcardIds(shard, operatorUid, clientId, bannerIds, new HashMap<>());
    }

    private ValidationResult<SaveVcardRequest, Defect> preValidateSaveVcardRequestByBannerIds(
            SaveVcardRequest request) {
        ModelItemValidationBuilder<SaveVcardRequest> ivb =
                ModelItemValidationBuilder.of(request);

        ivb.item(SaveVcardRequest.CAMPAIGN_ID)
                .check(notNull())
                .check(validId(), When.isValid());

        ivb.item(SaveVcardRequest.VCARD)
                .check(notNull());

        ivb.list(SaveVcardRequest.BANNER_IDS)
                .checkBy(this::preValidateBannerIds);

        return ivb.getResult();
    }

    private ValidationResult<SaveVcardRequest, Defect> preValidateSaveVcardRequestByVcardId(
            SaveVcardRequest request) {
        ModelItemValidationBuilder<SaveVcardRequest> ivb =
                ModelItemValidationBuilder.of(request);

        ivb.item(SaveVcardRequest.CAMPAIGN_ID)
                .check(notNull())
                .check(validId(), When.isValid());

        ivb.item(SaveVcardRequest.VCARD)
                .check(notNull());

        ivb.item(SaveVcardRequest.VCARD_ID)
                .check(notNull())
                .check(validId(), When.isValid());

        return ivb.getResult();
    }

    private ValidationResult<AssignVcardRequest, Defect> preValidateAssignVcardRequest(
            AssignVcardRequest request) {
        ModelItemValidationBuilder<AssignVcardRequest> ivb =
                ModelItemValidationBuilder.of(request);

        ivb.item(AssignVcardRequest.CAMPAIGN_ID)
                .check(notNull())
                .check(validId(), When.isValid());

        ivb.list(AssignVcardRequest.BANNER_IDS)
                .checkBy(this::preValidateBannerIds);

        ivb.item(AssignVcardRequest.VCARD_ID)
                .check(notNull())
                .check(validId(), When.isValid());

        return ivb.getResult();
    }

    private ValidationResult<UnassignVcardRequest, Defect> preValidateUnassignVcardRequest(
            UnassignVcardRequest request) {
        ModelItemValidationBuilder<UnassignVcardRequest> ivb =
                ModelItemValidationBuilder.of(request);

        ivb.item(UnassignVcardRequest.CAMPAIGN_ID)
                .check(notNull())
                .check(validId(), When.isValid());

        ivb.list(UnassignVcardRequest.BANNER_IDS)
                .checkBy(this::preValidateBannerIds);

        return ivb.getResult();
    }

    private ValidationResult<SaveVcardRequest, Defect> validateSaveVcardRequestByBannerIds(
            Long operatorUid, UidClientIdShard client, SaveVcardRequest request) {
        ModelItemValidationBuilder<SaveVcardRequest> ivb =
                ModelItemValidationBuilder.of(request);

        ivb.item(SaveVcardRequest.CAMPAIGN_ID)
                .checkBy(cid -> validateCampaignId(operatorUid, client, cid));

        Long campaignId = request.getCampaignId();
        ivb.list(SaveVcardRequest.BANNER_IDS)
                .checkBy(bids -> validateBannerIds(operatorUid, client.getClientId(), campaignId, bids, false));

        return ivb.getResult();
    }

    private ValidationResult<SaveVcardRequest, Defect> validateSaveVcardRequestByVcardId(
            Long operatorUid, UidClientIdShard client, SaveVcardRequest request, boolean validBannerIdsIsEmpty) {
        ModelItemValidationBuilder<SaveVcardRequest> ivb =
                ModelItemValidationBuilder.of(request);

        ivb.item(SaveVcardRequest.CAMPAIGN_ID)
                .checkBy(cid -> validateCampaignId(operatorUid, client, cid));

        Long campaignId = request.getCampaignId();
        ivb.item(SaveVcardRequest.VCARD_ID)
                .checkBy(vcardId -> validateVcardId(client.getShard(), client.getUid(), campaignId, vcardId))
                .check(fromPredicate(v -> !validBannerIdsIsEmpty, bannersWithVcardIdNotFound()), When.isValid());

        return ivb.getResult();
    }

    private ValidationResult<AssignVcardRequest, Defect> validateAssignVcardRequest(
            Long operatorUid, UidClientIdShard client, AssignVcardRequest request, boolean vcardExists) {
        ModelItemValidationBuilder<AssignVcardRequest> ivb =
                ModelItemValidationBuilder.of(request);

        ivb.item(AssignVcardRequest.CAMPAIGN_ID)
                .checkBy(cid -> validateCampaignId(operatorUid, client, cid));

        Long campaignId = request.getCampaignId();
        ivb.list(AssignVcardRequest.BANNER_IDS)
                .checkBy(bids -> validateBannerIds(operatorUid, client.getClientId(), campaignId, bids, false));

        ivb.item(AssignVcardRequest.VCARD_ID)
                .check(fromPredicate(v -> vcardExists, CommonDefects.objectNotFound()));

        return ivb.getResult();
    }

    private ValidationResult<UnassignVcardRequest, Defect> validateUnassignVcardRequest(
            Long operatorUid, UidClientIdShard client, UnassignVcardRequest request) {
        ModelItemValidationBuilder<UnassignVcardRequest> ivb =
                ModelItemValidationBuilder.of(request);

        ivb.item(UnassignVcardRequest.CAMPAIGN_ID)
                .checkBy(cid -> validateCampaignId(operatorUid, client, cid));

        Long campaignId = request.getCampaignId();
        ivb.list(UnassignVcardRequest.BANNER_IDS)
                .checkBy(bids -> validateBannerIds(operatorUid, client.getClientId(), campaignId, bids, true));

        return ivb.getResult();
    }

    private ValidationResult<Long, Defect> validateCampaignId(
            Long operatorUid, UidClientIdShard client, Long campaignId) {
        CampaignSubObjectAccessChecker campaignAccessChecker =
                campaignSubObjectAccessCheckerFactory.newCampaignChecker(operatorUid, client,
                        singletonList(campaignId));
        CampaignSubObjectAccessValidator campaignAccessValidator =
                campaignAccessChecker.createValidator(CampaignAccessType.READ_WRITE);

        return ItemValidationBuilder.<Long, Defect>of(campaignId)
                .checkBy(campaignAccessValidator)
                .getResult();
    }

    private ValidationResult<Long, Defect> validateVcardId(
            int shard, Long clientUid, Long campaignId, Long vcardId) {
        Map<Long, Long> vcardIdToCampaignId = vcardRepository.getVcardIdsToCampaignIds(shard, clientUid);

        Set<Long> existingVcardIds = StreamEx.of(vcardId).filter(vcardIdToCampaignId::containsKey).toSet();
        // если номер кампании из запроса отличается от номера кампании из визитки вернем ошибку
        Set<Long> vcardIdsWithCorrectCampaignId = StreamEx.of(existingVcardIds)
                .filter(v -> campaignId.equals(vcardIdToCampaignId.get(v))).toSet();

        return ItemValidationBuilder.<Long, Defect>of(vcardId)
                .check(inSet(existingVcardIds), CommonDefects.objectNotFound())
                .check(inSet(vcardIdsWithCorrectCampaignId), inconsistentVcardCidAndCampaignId(), When.isValid())
                .getResult();
    }

    private ValidationResult<List<Long>, Defect> preValidateBannerIds(List<Long> bannerIds) {
        return ListValidationBuilder.<Long, Defect>of(bannerIds)
                .check(notNull())
                .check(minListSize(1), absentRequiredField(), When.isValid())
                .checkEach(notNull())
                .checkEach(validId(), When.isValid())
                .checkEach(unique(), duplicatedObject(), When.isValid())
                .getResult();
    }

    private ValidationResult<List<Long>, Defect> validateBannerIds(long operatorUid, ClientId clientId,
                                                                   Long campaignId, List<Long> bannerIds,
                                                                   boolean checkThatBannerHasHref) {
        AdsSelectionCriteria criteria = new AdsSelectionCriteria().withAdIds(new HashSet<>(bannerIds));
        List<BannerWithSystemFields> existingBanners = bannerService.getBannersBySelectionCriteria(operatorUid,
                clientId, criteria, LimitOffset.maxLimited());
        Set<Long> existingBannerIds = StreamEx.of(existingBanners)
                .map(Banner::getId)
                .toSet();

        Set<Long> bannerIdsWithSupportedType = StreamEx.of(existingBanners)
                .filter(b -> SUPPORTED_BANNER_TYPES.contains(b.getClass())).map(Banner::getId).toSet();

        Set<Long> bannerIdsWithCorrectCampaignId = StreamEx.of(existingBanners)
                .filter(b -> b.getCampaignId().equals(campaignId)).map(Banner::getId).toSet();

        Boolean clientHasDesktopTurbolandingFeature = featureService.getEnabledForClientId(clientId)
                .contains(FeatureName.DESKTOP_LANDING.getName());
        // в текстовом баннере должна быть или визитка или href (если href нет то визитку удалить нельзя)
        // у динамического нет href (поэтому можно удалить визитку)
        Set<Long> bannerIdsWithoutHref = StreamEx.of(existingBanners)
                .select(TextBanner.class)
                .filter(b -> !clientHasDesktopTurbolandingFeature && Strings.isNullOrEmpty(b.getHref()))
                .map(Banner::getId)
                .toSet();
        return ListValidationBuilder.<Long, Defect>of(bannerIds)
                .checkEach(inSet(existingBannerIds), CommonDefects.objectNotFound())
                .checkEach(inSet(bannerIdsWithSupportedType), unsupportedBannerType(), When.isValid())
                .checkEach(inSet(bannerIdsWithCorrectCampaignId), inconsistentBannerCidAndCampaignId(), When.isValid())
                .checkEach(notInSet(bannerIdsWithoutHref), requiredButEmptyHrefOrTurboOrVcardIdOrPermalink(),
                        When.isValidAnd(When.isTrue(checkThatBannerHasHref)))
                .getResult();
    }
}
