package ru.yandex.direct.core.entity.vcard.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ListMultimap;
import com.google.common.collect.MultimapBuilder;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.vcard.model.PointOnMap;
import ru.yandex.direct.core.entity.vcard.model.PointPrecision;
import ru.yandex.direct.core.entity.vcard.model.PointType;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.geosearch.GeosearchClient;
import ru.yandex.direct.geosearch.model.Address;
import ru.yandex.direct.geosearch.model.GeoObject;
import ru.yandex.direct.regions.Region;

import static ru.yandex.direct.core.entity.vcard.service.VcardServiceMappings.pointTypeFromGeocoder;
import static ru.yandex.direct.core.entity.vcard.service.VcardServiceMappings.precisionFromGeocoder;

@Component
@ParametersAreNonnullByDefault
public class VcardHelper {
    private final GeosearchClient geosearchClient;
    private final GeoRegionLookup geoRegionLookup;


    public VcardHelper(GeosearchClient geosearchClient, GeoRegionLookup geoRegionLookup) {
        this.geosearchClient = geosearchClient;
        this.geoRegionLookup = geoRegionLookup;
    }

    public void fillVcardsWithRegionIds(Collection<Vcard> vcards) {
        ListMultimap<String, Vcard> citiesForVcardsWithNullRegionId =
                getUniqueCitiesForVcardsWithNullRegionId(vcards);
        Set<String> citiesToLookupRegionId = citiesForVcardsWithNullRegionId.keySet();

        // не для всех городов могут быть найдены id регионов
        Map<String, Long> regionIdsByCities = geoRegionLookup.getRegionIdsByCities(citiesToLookupRegionId);

        citiesForVcardsWithNullRegionId.asMap().forEach((city, vcardsWithCity) -> {
            // если id региона не найден, то выставляем id корневого региона
            Long regionId = regionIdsByCities.getOrDefault(city, Region.GLOBAL_REGION_ID);
            for (Vcard vcard : vcardsWithCity) {
                vcard.setGeoId(regionId);
            }
        });
    }


    private ListMultimap<String, Vcard> getUniqueCitiesForVcardsWithNullRegionId(Collection<Vcard> vcards) {
        ListMultimap<String, Vcard> uniqueCities = MultimapBuilder
                .hashKeys()
                .arrayListValues()
                .build();

        for (Vcard vcard : vcards) {
            if (vcard.getGeoId() == null && vcard.getCity() != null) {
                uniqueCities.put(vcard.getCity(), vcard);
            }
        }
        return uniqueCities;
    }


    /**
     * Заполняем визитки данными из геокодера (автоматически определенная точка,
     * тип точки и точность). Из всех визиток извлекаем уникальные адреса и
     * обращаемся с ними в геокодер, чтобы не делать дублирующихся вызовов
     * с одинаковыми данными.
     *
     * @param vcards валидные визитки
     */
    public void fillVcardsWithGeocoderData(Collection<Vcard> vcards) {
        ListMultimap<Address, Vcard> uniqueAddresses = getUniqueGeocoderAddresses(vcards);

        uniqueAddresses.asMap().forEach((address, vcardsWithThisAddress) -> {
            GeoObject geoObject = geosearchClient.getMostRelevantGeoData(address);

            PointOnMap autoPoint = new PointOnMap()
                    .withX(geoObject.getX())
                    .withY(geoObject.getY())
                    .withX1(geoObject.getX1())
                    .withY1(geoObject.getY1())
                    .withX2(geoObject.getX2())
                    .withY2(geoObject.getY2());
            PointType pointType = pointTypeFromGeocoder(geoObject.getKind());
            PointPrecision pointPrecision = precisionFromGeocoder(geoObject.getPrecision());

            vcardsWithThisAddress.forEach(
                    vcard -> {
                        // Если местонахождение адреса не задано пользователем и геокодер не смог определить
                        // его по адресу с достаточной точностью, то ничего не задаем в качестве адреса
                        // (См. CommonMaps.pm::_save_address)
                            if (vcard.getManualPoint() == null && !isPointPrecisionAccurate(pointPrecision)) {
                            vcard.setAddressId(0L);
                        } else {
                            // Если точка вручную не задана, то берем определенную геокодером (См. CommonMaps.pm::_save_address)
                            if (vcard.getManualPoint() == null) {
                                vcard.setManualPoint(autoPoint);
                            }
                            vcard.withAutoPoint(autoPoint)
                                    .withPointType(pointType)
                                    .withPrecision(pointPrecision);
                        }
                    });
        });
    }

    /**
     * Заполнить системные поля со временами. Нужно, чтобы обойти ограничения на NULL при записи в таблицу.
     * Данные в lastChange и lastDissociationTime будут перезатёрты.
     * @param vcards визитки для заполнения.
     */
    public static void fillSystemDateTimeFields(Collection<Vcard> vcards) {
        LocalDateTime now = LocalDateTime.now();
        vcards.forEach(vcard -> {
                    vcard.setLastChange(now);
                    vcard.setLastDissociation(now);
        });
    }

    /**
     * Адрес разрешен с приемлимой точностью (См. CommonMaps.pm::_save_address)
     */
    private static boolean isPointPrecisionAccurate(@Nonnull PointPrecision precision) {
        return precision == PointPrecision.EXACT
                || precision == PointPrecision.NUMBER
                || precision == PointPrecision.NEAR
                || precision == PointPrecision.STREET;
    }

    private ListMultimap<Address, Vcard> getUniqueGeocoderAddresses(Collection<Vcard> vcards) {
        ListMultimap<Address, Vcard> uniqueAddresses = MultimapBuilder
                .hashKeys()
                .arrayListValues()
                .build();

        for (Vcard vcard : vcards) {
            Address address = new Address()
                    .withCountry(vcard.getCountry())
                    .withCity(vcard.getCity())
                    .withStreet(vcard.getStreet())
                    .withHouse(vcard.getHouse())
                    .withBuilding(vcard.getBuild());
            uniqueAddresses.put(address, vcard);
        }
        return uniqueAddresses;
    }

}
