package ru.yandex.direct.core.entity.vcard.service;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.type.vcard.BannerWithVcardUtils;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService;
import ru.yandex.direct.core.entity.vcard.service.validation.DeleteVcardValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static ru.yandex.direct.core.entity.vcard.service.validation.VcardDefects.vcardIsInUse;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.result.PathHelper.index;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Service
@ParametersAreNonnullByDefault
public class VcardService implements EntityService<Vcard, Long> {

    private final AddVcardValidationService addVcardValidationService;
    private final DeleteVcardValidationService deleteVcardValidationService;
    private final VcardRepository vcardRepository;
    private final GeoRegionLookup geoRegionLookup;
    private final VcardHelper vcardHelper;
    private final RbacService rbacService;
    private final ShardHelper shardHelper;
    private final ShardSupport shardSupport;
    private final BannerCommonRepository bannerCommonRepository;
    private final BannerTypedRepository bannerTypedRepository;

    @Autowired
    public VcardService(AddVcardValidationService addVcardValidationService,
                        DeleteVcardValidationService deleteVcardValidationService, VcardRepository vcardRepository,
                        GeoRegionLookup geoRegionLookup, VcardHelper vcardHelper, RbacService rbacService,
                        ShardHelper shardHelper,
                        ShardSupport shardSupport,
                        BannerCommonRepository bannerCommonRepository,
                        BannerTypedRepository bannerTypedRepository) {
        this.addVcardValidationService = addVcardValidationService;
        this.deleteVcardValidationService = deleteVcardValidationService;
        this.vcardRepository = vcardRepository;
        this.geoRegionLookup = geoRegionLookup;
        this.vcardHelper = vcardHelper;
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;
        this.shardSupport = shardSupport;
        this.bannerCommonRepository = bannerCommonRepository;
        this.bannerTypedRepository = bannerTypedRepository;
    }

    public List<Vcard> getVcards(long operatorUid, ClientId clientId, @Nullable Collection<Long> ids,
                                 LimitOffset limitOffset) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        long clientUid = rbacService.getChiefByClientId(clientId);

        Map<Long, Long> vcardIdToCampaignId = vcardRepository.getVcardIdsToCampaignIds(shard, clientUid);

        Set<Long> uniqCampaignIds = new HashSet<>(vcardIdToCampaignId.values());
        Set<Long> visibleCampaignIds = rbacService.getVisibleCampaigns(operatorUid, uniqCampaignIds);

        Collection<Long> filteredIds;
        if (ids != null) {
            filteredIds = ids.stream()
                    .filter(id -> vcardIdToCampaignId.containsKey(id) &&
                            visibleCampaignIds.contains(vcardIdToCampaignId.get(id)))
                    .collect(toList());
        } else {
            filteredIds = vcardIdToCampaignId.entrySet().stream()
                    .filter(e -> visibleCampaignIds.contains(e.getValue()))
                    .map(Map.Entry::getKey)
                    .collect(toList());
        }

        return vcardRepository.getVcards(shard, clientUid, filteredIds, limitOffset);
    }

    public Map<Long, Vcard> getVcardsById(ClientId clientId, @Nullable Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        long clientUid = rbacService.getChiefByClientId(clientId);

        List<Vcard> vcards = vcardRepository.getVcards(shard, clientUid, ids);
        return listToMap(vcards, Vcard::getId);
    }

    /**
     * Проставляет доп поля (metro_name, country_geo_id) в переданных визитках
     *
     * @param vcards визитные карточки
     */
    public void setVcardsAdditionalFields(List<Vcard> vcards) {
        Set<String> countries = StreamEx.of(vcards).map(Vcard::getCountry).toSet();
        Map<String, Long> regionIdsByCountries = geoRegionLookup.getRegionIdsByCountries(countries);

        Set<Long> metroIds = StreamEx.of(vcards).map(Vcard::getMetroId).filter(v -> v != null && v > 0L).toSet();
        Map<Long, String> metroNamesByMetroIds = geoRegionLookup.getMetroNamesByIds(metroIds);

        vcards.forEach(v -> v.withCountryGeoId(regionIdsByCountries.get(v.getCountry()))
                .withMetroName(metroNamesByMetroIds.get(defaultIfNull(v.getMetroId(), 0L))));
    }

    /**
     * Добавить визитные карточки
     * <p>
     * Часть визитных карточек может быть не валидна. Операция в любом случае выполнится
     *
     * @param vcards      Список визитных карточек
     * @param operatorUid Uid оператора
     * @param clientId    Идентификатор клиента
     */
    public MassResult<Long> addVcardsPartial(
            List<Vcard> vcards, Long operatorUid, ClientId clientId) {
        AddVcardOperation addOperation =
                createAddVcardOperation(false, Applicability.PARTIAL, vcards, operatorUid,
                        clientId, bannerCommonRepository);
        return addOperation.prepareAndApply();
    }

    /**
     * Добавить визитные карточки
     * <p>
     * Для выполнения операции все визитные карточки должны быть валидны
     *
     * @param vcards      Список визитных карточек
     * @param operatorUid Uid оператора
     * @param clientId    Идентификатор клиента
     */
    public MassResult<Long> addVcardsFull(
            List<Vcard> vcards, Long operatorUid, ClientId clientId) {
        AddVcardOperation addOperation =
                createAddVcardOperation(false, Applicability.FULL, vcards, operatorUid, clientId,
                        bannerCommonRepository);
        return addOperation.prepareAndApply();
    }

    public AddVcardOperation createFullAddVcardOperationAsPartOfComplexOperation(
            List<Vcard> vcards, Long operatorUid, ClientId clientId) {
        // Так как в данном случае валидировать кампанию не нужно, то мы передаем в качестве campaignAccessTypeSets
        // константу ALL
        return createAddVcardOperation(true, Applicability.FULL, vcards, operatorUid, clientId,
                bannerCommonRepository);
    }

    public MassResult<Long> deleteVcards(List<Long> vcardIds, long operatorUid, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        long clientUid = rbacService.getChiefByClientId(clientId);

        ValidationResult<List<Long>, Defect> validationResult =
                deleteVcardValidationService.validate(vcardIds, operatorUid, clientId, clientUid, shard);
        Collection<Long> validItems = getValidItems(validationResult);
        if (validationResult.hasErrors() || validItems.isEmpty()) {
            return MassResult.brokenMassAction(vcardIds, validationResult);
        }

        Set<Long> deletedVcardIds = vcardRepository.deleteUnusedVcards(shard, clientUid, validItems);
        shardSupport.deleteValues(ShardKey.VCARD_ID, deletedVcardIds);

        fixValidationResultOverActuallyDeletedVcardIds(vcardIds, deletedVcardIds, validationResult);
        return MassResult.successfulMassAction(vcardIds, validationResult);
    }

    /**
     * Существует ситуация, когда в промежутке между валидацией списка id визиток на удаление
     * и моментом фактического удаления визитка привязывается к баннеру и не удаляется.
     * В этом случае необходимо добавить в результат валидации соответствующие ошибки уже
     * после попытки удаления.
     * <p>
     * Данный метод на основе переданного на удаление списка id визиток и набора фактически
     * удаленных визиток исправляет результат валидации, добавляя в него ошибки
     * для тех id из входного списка, которые не имеют ошибок (на этапе основной валидации)
     * и при этом не были удалены.
     *
     * @param vcardIds                список id визиток, переданный на удаление извне.
     * @param actuallyDeletedVcardIds фактически удаленные визитки.
     * @param validationResult        результат валидации.
     */
    private void fixValidationResultOverActuallyDeletedVcardIds(List<Long> vcardIds,
                                                                Set<Long> actuallyDeletedVcardIds,
                                                                ValidationResult<List<Long>, Defect> validationResult) {
        for (int i = 0; i < vcardIds.size(); i++) {
            Long vcardId = vcardIds.get(i);
            ValidationResult<Long, Defect> subValidationResult =
                    validationResult.getOrCreateSubValidationResult(index(i), vcardId);

            if (!subValidationResult.hasAnyErrors() && !actuallyDeletedVcardIds.contains(vcardId)) {
                subValidationResult.addError(vcardIsInUse());
            }
        }
    }

    private AddVcardOperation createAddVcardOperation(
            boolean partOfComplexOperation,
            Applicability applicability,
            List<Vcard> vcards,
            Long operatorUid,
            ClientId clientId,
            BannerCommonRepository bannerRepository) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return createAddVcardOperation(
                partOfComplexOperation, applicability, vcards, shard, operatorUid, clientId, bannerRepository);
    }

    private AddVcardOperation createAddVcardOperation(
            boolean partOfComplexOperation,
            Applicability applicability,
            List<Vcard> vcards,
            int shard,
            Long operatorUid,
            ClientId clientId,
            BannerCommonRepository bannerRepository) {
        UidClientIdShard client = UidClientIdShard.of(rbacService.getChiefByClientId(clientId), clientId, shard);

        return new AddVcardOperation(
                applicability,
                partOfComplexOperation,
                vcards,
                addVcardValidationService,
                vcardRepository,
                vcardHelper,
                operatorUid,
                client,
                bannerRepository
        );
    }

    public Map<Long, Long> getLastVcardIdByCampaignId(ClientId clientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return vcardRepository.getLastVcardIdByCampaignId(shard, campaignIds);
    }

    public Map<Long, Vcard> getCommonVcardByCampaignId(UidAndClientId uidAndClientId, Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(uidAndClientId.getClientId());
        List<BannerWithSystemFields> banners =
                bannerTypedRepository.getBannersByCampaignIdsAndClass(shard, campaignIds,
                        BannerWithSystemFields.class);

        Map<Long, Long> commonVcardIdByCampaignId = StreamEx.of(banners)
                .filter(banner -> !banner.getStatusArchived())
                .mapToEntry(BannerWithSystemFields::getCampaignId, Function.identity())
                .selectValues(BannerWithVcard.class)
                .sortedBy(Map.Entry::getKey)
                .collapseKeys()
                .filterValues(BannerWithVcardUtils::hasCommonVcard)
                .mapValues(bannersWithVcard -> bannersWithVcard.get(0))
                .mapValues(BannerWithVcard::getVcardId)
                .toMap();

        List<Vcard> commonVcards = vcardRepository.getVcards(shard, uidAndClientId.getUid(),
                commonVcardIdByCampaignId.values());

        return listToMap(commonVcards, Vcard::getCampaignId, Function.identity());
    }

    @Override
    public List<Vcard> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        Map<Long, Vcard> vcardsByIds = getVcardsById(clientId, ids);
        return List.copyOf(vcardsByIds.values());
    }

    @Override
    public MassResult<Long> add(ClientId clientId, Long operatorUid, List<Vcard> entities,
                                Applicability applicability) {
        AddVcardOperation addOperation =
                createAddVcardOperation(false, applicability, entities, operatorUid,
                        clientId, bannerCommonRepository);
        return addOperation.prepareAndApply();
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer, List<Vcard> entities, Applicability applicability) {
        AddVcardOperation addOperation =
                createAddVcardOperation(
                        false,
                        applicability,
                        entities,
                        copyContainer.getShardTo(),
                        copyContainer.getOperatorUid(),
                        copyContainer.getClientIdTo(),
                        bannerCommonRepository);
        return addOperation.prepareAndApply();
    }
}
