package ru.yandex.direct.core.entity.vcard.service.validation;

import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.dbutil.model.UidClientIdShard;
import ru.yandex.direct.utils.TextConstants;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.AdmissibleCharsConstraint;
import ru.yandex.direct.validation.defect.params.StringDefectParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.StringDefects.mustContainLetters;

@Service
@ParametersAreNonnullByDefault
public class AddVcardValidationService {

    public static final int COMPANY_NAME_MAX_LENGTH = 255;
    public static final int CONTACT_PERSON_MAX_LENGTH = 155;
    public static final int CONTACT_EMAIL_MAX_LENGTH = 255;
    public static final int EXTRA_MESSAGE_MAX_LENGTH = 200;
    public static final int COUNTRY_MAX_LENGTH = 50;
    public static final int CITY_MAX_LENGTH = 55;
    public static final int STREET_MAX_LENGTH = 55;
    public static final int HOUSE_MAX_LENGTH = 30;
    public static final int BUILD_MAX_LENGTH = 10;
    public static final int APART_MAX_LENGTH = 100;

    public static final Constraint<String, Defect> ALLOWED_COUNTRY_CITY_CHARS =
            new AdmissibleCharsConstraint(
                    TextConstants.LETTERS_AND_NUMBERS + "[].- ()", mustContainLetters());
    public static final Constraint<String, Defect> ALLOWED_HOUSE_BUILDING_CHARS =
            new AdmissibleCharsConstraint(
                    TextConstants.LETTERS_AND_NUMBERS + "[].\\/-, ", mustContainLetters());

    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final MetroIdValidator metroIdValidator;

    @Autowired
    public AddVcardValidationService(
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            MetroIdValidator metroIdValidator) {
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.metroIdValidator = metroIdValidator;
    }

    /**
     * Провалидировать заданный список визитных карточек
     *
     * @param operatorUid          Uid оператора
     * @param needValidateCampaign Валидировать кампанию визитной карточки
     * @param vcards               Список визитных карточек
     * @param client               Клиент для которого создаются визитные карточки
     */
    public ValidationResult<List<Vcard>, Defect> validate(
            Long operatorUid,
            boolean needValidateCampaign,
            List<Vcard> vcards,
            UidClientIdShard client) {
        ListValidationBuilder<Vcard, Defect> lvb =
                ListValidationBuilder.of(vcards, Defect.class);

        lvb.checkEachBy(this::validateSimple);

        if (needValidateCampaign) {
            Set<Long> campaignIds = vcards.stream()
                    .map(Vcard::getCampaignId)
                    .filter(Objects::nonNull)
                    .collect(toSet());

            CampaignSubObjectAccessChecker campaignAccessChecker =
                    campaignSubObjectAccessCheckerFactory.newCampaignChecker(operatorUid, client, campaignIds);
            CampaignSubObjectAccessValidator campaignAccessValidator =
                    campaignAccessChecker.createValidator(CampaignAccessType.READ_WRITE);

            lvb.checkEachBy(campaignValidator(campaignAccessValidator));
        }
        return lvb.getResult();
    }

    private Validator<Vcard, Defect> campaignValidator(
            Validator<Long, Defect> campaignAccessValidator) {
        return vcard -> {
            ModelItemValidationBuilder<Vcard> vb = ModelItemValidationBuilder.of(vcard);

            // кампания
            vb.item(Vcard.CAMPAIGN_ID)
                    .check(notNull())
                    .checkBy(campaignAccessValidator, When.isValid());

            return vb.getResult();
        };
    }

    private ValidationResult<Vcard, Defect> validateSimple(Vcard vcard) {
        ModelItemValidationBuilder<Vcard> vb = ModelItemValidationBuilder.of(vcard);

        vb.item(Vcard.CAMPAIGN_ID)
                .check(notNull());

        vb.checkBy(VcardValidator.build(metroIdValidator));
        return vb.getResult();
    }

    public enum VoidDefectIds implements DefectId<Void> {
        CONTACT_EMAIL_IS_EMPTY,

        CONTACT_EMAIL_INVALID_FORMAT,
    }

    public enum StringLengthDefectIds implements DefectId<StringDefectParams> {
        CONTACT_EMAIL_TOO_LONG
    }


    public static class DefectDefinitions {

        // ContactEmail
        public static Defect<Void> contactEmailIsEmpty() {
            return new Defect<>(VoidDefectIds.CONTACT_EMAIL_IS_EMPTY);
        }

        public static Defect<StringDefectParams> contactEmailTooLong() {
            return new Defect<>(StringLengthDefectIds.CONTACT_EMAIL_TOO_LONG,
                    new StringDefectParams().withMaxLength(CONTACT_EMAIL_MAX_LENGTH));
        }

        public static Defect<Void> invalidContactEmailFormat() {
            return new Defect<>(VoidDefectIds.CONTACT_EMAIL_INVALID_FORMAT);
        }
    }
}
