package ru.yandex.direct.core.entity.vcard.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSortedMap;

import ru.yandex.direct.core.entity.vcard.model.InstantMessenger;
import ru.yandex.direct.utils.TextConstants;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.defect.params.StringDefectParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.misc.lang.StringUtils;

import static ru.yandex.direct.core.entity.vcard.service.validation.InstantMessengerValidator.DefectDefinitions.invalidIcqLoginFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.InstantMessengerValidator.DefectDefinitions.invalidJabberLoginFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.InstantMessengerValidator.DefectDefinitions.invalidMailAgentLoginFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.InstantMessengerValidator.DefectDefinitions.invalidSkypeOrMsnLoginFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.InstantMessengerValidator.DefectDefinitions.loginIsNull;
import static ru.yandex.direct.core.entity.vcard.service.validation.InstantMessengerValidator.DefectDefinitions.tooLongLogin;
import static ru.yandex.direct.core.entity.vcard.service.validation.InstantMessengerValidator.DefectDefinitions.typeIsNull;
import static ru.yandex.direct.core.entity.vcard.service.validation.InstantMessengerValidator.DefectDefinitions.unsupportedType;
import static ru.yandex.direct.utils.StringUtils.countDigits;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;

public class InstantMessengerValidator implements Validator<InstantMessenger, Defect> {
    static final String ICQ = "icq";
    static final String MAIL_AGENT = "mail_agent";
    static final String JABBER = "jabber";
    static final String SKYPE = "skype";
    static final String MSN = "msn";

    public static final int LOGIN_MAX_LENGTH = 255;

    private static final InstantMessengerValidator INSTANCE = new InstantMessengerValidator();

    private static final List<String> ALLOWED_MAIL_AGENT_DOMAINS = ImmutableList.of(
            "mail.ru", "inbox.ru", "bk.ru", "list.ru");

    private static final Pattern VALID_ICQ_UIN_PATTERN = Pattern.compile("[\\d\\-]+");
    private static final Pattern VALID_SKYPE_OR_MSN_PATTERN = Pattern.compile(
            "^[" + TextConstants.LETTERS_AND_NUMBERS + "\\-\\._@]+$");

    private static final Map<String, Constraint<String, Defect>> CONSTRAINTS_BY_CLIENT_TYPE =
            createClientTypes();

    /**
     * В качестве UIN ICQ разрешаем циферки (не меньше 5, не больше 10) и дефисы.
     */
    private static Constraint<String, Defect> icqLoginValidator() {
        return Constraint.fromPredicate(
                Predicates.notEmpty().and(
                        login -> {
                            int digitCount = countDigits(login);
                            return (5 <= digitCount && digitCount <= 10)
                                    && VALID_ICQ_UIN_PATTERN.matcher(login).matches();
                        }),
                invalidIcqLoginFormat());
    }

    /**
     * Разрешаем email-адрес something@mail.ru
     */
    private static Constraint<String, Defect> mailAgentLoginValidator() {
        return Constraint.fromPredicate(
                Predicates.notEmpty()
                        .and(Predicates.validEmail())
                        .and(login -> ALLOWED_MAIL_AGENT_DOMAINS.stream()
                                .anyMatch(d -> StringUtils.endsWithIgnoreCase(login, d))),
                invalidMailAgentLoginFormat());
    }

    /**
     * Разрешаем любой валидный email-адрес
     */
    private static Constraint<String, Defect> jabberLoginValidator() {
        return Constraint.fromPredicate(
                Predicates.notEmpty().and(Predicates.validEmail()),
                invalidJabberLoginFormat());
    }

    /**
     * Разрешаем [a-z A-Z а-я А-Я .(точку) -(дефис) _(подчеркивание) @ 0-9]
     */
    private static Constraint<String, Defect> skypeOrMsnAgentLoginValidator() {
        return Constraint.fromPredicate(
                Predicates.notEmpty()
                        .and(VALID_SKYPE_OR_MSN_PATTERN.asPredicate()
                                .or(Predicates.validEmail())),
                invalidSkypeOrMsnLoginFormat());
    }

    private static Map<String, Constraint<String, Defect>> createClientTypes() {
        Constraint<String, Defect> skypeOrMsnLoginValidator = skypeOrMsnAgentLoginValidator();

        // Так как в perl-е типы клиентов сравниваются без учета регистра и также впоследствии
        // сохраняются в базе данных, то используем SortedMap, кастомизированное правильным comparator-ом
        // см. Validation/VCards.pm
        return new ImmutableSortedMap.Builder<String, Constraint<String, Defect>>(
                String.CASE_INSENSITIVE_ORDER)
                .put(ICQ, icqLoginValidator())
                .put(MAIL_AGENT, mailAgentLoginValidator())
                .put(JABBER, jabberLoginValidator())
                .put(SKYPE, skypeOrMsnLoginValidator)
                .put(MSN, skypeOrMsnLoginValidator)
                .build();
    }

    public static InstantMessengerValidator instantMessengerValidator() {
        return INSTANCE;
    }

    @Override
    public ValidationResult<InstantMessenger, Defect> apply(InstantMessenger instantMessenger) {
        ModelItemValidationBuilder<InstantMessenger> vb = ModelItemValidationBuilder.of(instantMessenger);

        vb.item(InstantMessenger.TYPE)
                .check(notNull(), typeIsNull())
                .check(inSet(CONSTRAINTS_BY_CLIENT_TYPE.keySet()), unsupportedType(), When.isValid());

        vb.item(InstantMessenger.LOGIN)
                .check(notNull(), loginIsNull())
                .check(maxStringLength(LOGIN_MAX_LENGTH), tooLongLogin(), When.isValid())
                .check(
                        Optional.ofNullable(instantMessenger.getType())
                                .map(CONSTRAINTS_BY_CLIENT_TYPE::get)
                                .orElse(CommonConstraints.success()),
                        When.isValid());

        return vb.getResult();
    }

    public enum VoidDefectIds implements DefectId<Void> {
        TYPE_IS_NULL,

        UNSUPPORTED_TYPE,

        LOGIN_IS_NULL,

        INVALID_ICQ_LOGIN_FORMAT,
        INVALID_MAIL_AGENT_LOGIN_FORMAT,
        INVALID_JABBER_LOGIN_FORMAT,
        INVALID_SKYPE_OR_MSN_LOGIN_FORMAT
    }

    public enum StringDefectIds implements DefectId<StringDefectParams> {
        LOGIN_IS_TOO_LONG
    }

    public static class DefectDefinitions {

        // type

        public static Defect<Void> typeIsNull() {
            return new Defect<>(VoidDefectIds.TYPE_IS_NULL);
        }

        public static Defect<Void> unsupportedType() {
            return new Defect<>(VoidDefectIds.UNSUPPORTED_TYPE);
        }

        // login

        public static Defect<Void> loginIsNull() {
            return new Defect<>(VoidDefectIds.LOGIN_IS_NULL);
        }

        public static Defect<Void> invalidIcqLoginFormat() {
            return new Defect<>(VoidDefectIds.INVALID_ICQ_LOGIN_FORMAT);
        }

        public static Defect<Void> invalidJabberLoginFormat() {
            return new Defect<>(VoidDefectIds.INVALID_JABBER_LOGIN_FORMAT);
        }

        public static Defect<Void> invalidMailAgentLoginFormat() {
            return new Defect<>(VoidDefectIds.INVALID_MAIL_AGENT_LOGIN_FORMAT);
        }

        public static Defect<Void> invalidSkypeOrMsnLoginFormat() {
            return new Defect<>(VoidDefectIds.INVALID_SKYPE_OR_MSN_LOGIN_FORMAT);
        }

        public static Defect<StringDefectParams> tooLongLogin() {
            return new Defect<>(StringDefectIds.LOGIN_IS_TOO_LONG,
                    new StringDefectParams().withMaxLength(LOGIN_MAX_LENGTH));
        }
    }
}
