package ru.yandex.direct.core.entity.vcard.service.validation;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;

import static ru.yandex.direct.core.entity.vcard.service.validation.MetroIdValidator.DefectDefinitions.invalidMetro;
import static ru.yandex.direct.core.entity.vcard.service.validation.MetroIdValidator.DefectDefinitions.metroCityIsRequired;

/**
 * Валидатор станции метро
 */
@Component
public class MetroIdValidator {
    private final GeoTreeFactory geoTreeFactory;

    @Autowired
    public MetroIdValidator(GeoTreeFactory geoTreeFactory) {
        this.geoTreeFactory = geoTreeFactory;
    }

    /**
     * Создать ограничение для заданного города
     * <p>
     * Станция метро должна существовать в заданном городе
     *
     * @param city Город для которого будет валидироваться станция метро
     */
    Constraint<Long, Defect> createConstraintFor(@Nullable String city) {
        return metroId -> {
            // Соотвествует случаю "Не выбрано".
            // Perl-ый код допускает подобный id-ик (См. Validation/VCards.pm::validate_metro)
            if (metroId == 0) {
                return null;
            }

            if (city == null) {
                return metroCityIsRequired();
            }

            if (!getGeoTree().isCityHasMetro(city, metroId)) {
                return invalidMetro();
            }

            return null;
        };
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getGlobalGeoTree();
    }

    public enum DefectIds implements DefectId<Void> {
        /**
         * Если указана станция метро, то обязательно требуется город
         */
        METRO_CITY_IS_REQUIRED,

        /**
         * Указана несуществующая станция метро или станция метро отсутствует в заданном городе
         */
        INVALID_METRO
    }

    public static class DefectDefinitions {
        static Defect<Void> metroCityIsRequired() {
            return new Defect<>(DefectIds.METRO_CITY_IS_REQUIRED);
        }

        static Defect<Void> invalidMetro() {
            return new Defect<>(DefectIds.INVALID_METRO);
        }
    }
}
