package ru.yandex.direct.core.entity.vcard.service.validation;

import ru.yandex.direct.utils.StringUtils;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;

import static ru.yandex.direct.core.entity.vcard.service.validation.OgrnConstraint.DefectDefinitions.invalidOgrn;

/**
 * Валидатор для ОГРН/ОГРНИП (См. Validation/VCards.pm::validate_ogrn)
 * <p>
 * Основной государственный регистрационный номер состоит из 13 цифр. Последняя, тринадцатая, и есть контрольная.
 * Делим число, состоящее из первых 12 знаков, на 11. Потом целую часть полученного результата умножаем на 11
 * и сравниваем с первоначальным 12-значным числом. Разница должна совпадать с 13-й цифрой ОГРН.
 * <p>
 * Допустим, приходит регистрационный номер 1037556120338.
 * <p>
 * Проверяем:
 * <p>
 * 103755612033 : 11 = 9432328366,636
 * 9432328366 x 11 = 103755612026.
 * 103755612033 – 103755612026 = 7.
 * <p>
 * Последняя цифра в заявленном ОГРН равна 8. Значит такого ОГРН не существует.
 * Примеры существующих ОГРН:
 * 1087746113344
 * 1087746385320
 * 1055406282513
 * 1067760833810 (внимание, они настоящие, взятые из интернета)
 * <p>
 * ОГРНИП - основной государственный регистрационный номер индивидуального предпринимателя.
 * Состоит из 15 цифр, последняя - контрольная. От ОГРН отличается тем, что:
 * 1. Под номер записи в реестре выделено семь, а не пять цифр
 * 2. Контрольная цифра равна последней цифре остатка от деления на 13,а не на 11, предыдущего 14-значного числа
 * Примеры существующих ОГРНИП:
 * 304540707500034
 * 304540220800032
 * 309774611900857
 * 310253706100022 (внимание, они настоящие, взятые из интернета)
 */
public class OgrnConstraint implements Constraint<String, Defect> {

    private static final int OGRN_LENGTH = 13;
    private static final int OGRN_CONTROL_DIV = 11;
    private static final int OGRNIP_LENGTH = 15;
    private static final int OGRNIP_CONTROL_DIV = 13;

    private static final OgrnConstraint INSTANCE = new OgrnConstraint();

    public static OgrnConstraint ogrnIsValid() {
        return INSTANCE;
    }

    @Override
    public Defect apply(String s) {
        if (s == null) {
            return null;
        }

        if (s.length() != OGRN_LENGTH && s.length() != OGRNIP_LENGTH) {
            return invalidOgrn();
        }

        // Первая цифра ОГРН может быть только цифрами: 1, 2, 3, 5
        if ("1235".indexOf(s.charAt(0)) == -1) {
            return invalidOgrn();
        }

        if (!Predicates.isPositiveWholeNumber().test(s)) {
            return invalidOgrn();
        }

        long controlNumber = StringUtils.fastParseUnsignedLong(s, 0, s.length() - 1);
        long mod = controlNumber % (s.length() == OGRN_LENGTH ? OGRN_CONTROL_DIV : OGRNIP_CONTROL_DIV);

        long checkDigit = mod % 10;
        long expectedCheckDigit = s.charAt(s.length() - 1) - '0';

        if (checkDigit != expectedCheckDigit) {
            return invalidOgrn();
        }

        return null;
    }

    public enum VoidDefectIds implements DefectId<Void> {
        INVALID_OGRN
    }

    public static class DefectDefinitions {

        public static Defect<Void> invalidOgrn() {
            return new Defect<>(VoidDefectIds.INVALID_OGRN);
        }
    }
}
