package ru.yandex.direct.core.entity.vcard.service.validation;

import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.vcard.model.Phone;
import ru.yandex.direct.core.entity.vcard.repository.VcardMappings;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.params.StringDefectParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.emptyCityCode;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.emptyCountryCode;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.emptyPhoneNumber;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.invalidCityCodeFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.invalidEntirePhoneLength;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.invalidExtensionFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.invalidPhoneNumberFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.nullCityCode;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.nullCountryCode;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.nullPhoneNumber;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.tooLongCityCode;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.tooLongCountryCode;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.tooLongEntirePhoneWithExtension;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.tooLongExtension;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.tooLongPhoneNumber;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.DefectDefinitions.tooShortPhoneNumber;
import static ru.yandex.direct.utils.StringUtils.countDigits;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.Predicates.nullOrEmpty;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.isPositiveWholeNumber;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty;

public class PhoneValidator implements Validator<Phone, Defect> {

    private static final PhoneValidator INSTANCE = new PhoneValidator();

    public static final Integer COUNTRY_CODE_MAX_LENGTH = 1 + 4; // Ожидается формат +?\d{1, 4}
    public static final Integer CITY_CODE_MAX_LENGTH = 5;
    public static final Integer PHONE_NUMBER_MIN_LENGTH = 5;
    public static final Integer PHONE_NUMBER_MAX_LENGTH = 9;
    public static final Integer EXTENSION_MAX_LENGTH = 6;
    public static final Integer ENTIRE_PHONE_MIN_LENGTH = 8;
    public static final Integer ENTIRE_PHONE_MAX_LENGTH = 17;
    public static final Integer ENTIRE_PHONE_WITH_EXTENSION_MAX_LENGTH = 25;

    public static PhoneValidator phoneIsValid() {
        return INSTANCE;
    }

    private static final String TURKEY_COUNTRY_CODE = "+90";
    private static final String TURKEY_PHONE_NUMBER_WITHOUT_CITY_CODE_PREFIX = "444";
    private static final Pattern COUNTRY_CODE_PATTERN = Pattern.compile("\\+?\\d{1,4}");
    private static final Pattern PHONE_NUMBER_PATTERN = Pattern.compile("\\d+(?:[ \\-]\\d+)*");

    @Override
    public ValidationResult<Phone, Defect> apply(Phone phone) {
        ModelItemValidationBuilder<Phone> vb = ModelItemValidationBuilder.of(phone);

        int phoneNumberDigitCount = countDigits(phone.getPhoneNumber());

        vb.item(Phone.COUNTRY_CODE)
                .check(notNull(), nullCountryCode())
                .check(notEmpty(), emptyCountryCode(), When.isValid())
                .check(maxStringLength(COUNTRY_CODE_MAX_LENGTH), tooLongCountryCode(), When.isValid())
                .checkByFunction(countryCodeValidator(phone.getCityCode()), When.isValid());

        vb.item(Phone.CITY_CODE)
                .check(notNull(), nullCityCode())
                .check(notEmpty(), emptyCityCode(),
                        When.isTrue(isCityCodeRequired(phone.getCountryCode(), phone.getPhoneNumber())))
                .check(maxStringLength(CITY_CODE_MAX_LENGTH), tooLongCityCode(), When.isValid())
                .check(cityCodeFormatIsValid(), When.isValid());

        vb.item(Phone.PHONE_NUMBER)
                .check(notNull(), nullPhoneNumber())
                .check(notEmpty(), emptyPhoneNumber(), When.isValid())
                .check(minPhoneNumberLength(phoneNumberDigitCount), When.isValid())
                .check(maxPhoneNumberLength(phoneNumberDigitCount), When.isValid())
                .check(matchPattern(PHONE_NUMBER_PATTERN), invalidPhoneNumberFormat(), When.isValid());

        vb.item(Phone.EXTENSION)
                .check(maxStringLength(EXTENSION_MAX_LENGTH), tooLongExtension(), When.isValid())
                .check(
                        isPositiveWholeNumber(),
                        invalidExtensionFormat(),
                        When.isValidAnd(When.valueIs(not(nullOrEmpty()))));

        vb.check(entirePhoneLengthIsValid(phoneNumberDigitCount), When.isValid());
        vb.check(maxEntirePhoneWithExtensionLength(), When.isValid());

        return vb.getResult();
    }

    private static Function<String, Defect> countryCodeValidator(String cityCode) {
        return countryCode -> {
            if (("8".equals(countryCode) && ("800".equals(cityCode) || "804".equals(cityCode)))
                    || ("0".equals(countryCode) && "800".equals(cityCode))) {
                return null;
            }

            if (!COUNTRY_CODE_PATTERN.matcher(countryCode).matches()) {
                return DefectDefinitions.invalidCountryCodeFormat();
            }

            if (!countryCode.startsWith("+")) {
                return DefectDefinitions.countryCodeMustStartWithPlus();
            }

            if (("+8".equals(countryCode) && ("800".equals(cityCode) || "804".equals(cityCode)))
                    || ("+0".equals(countryCode) && "800".equals(cityCode))) {
                return DefectDefinitions.countryCodeMustNotStartWithPlus(cityCode);
            }

            return null;
        };
    }

    private static boolean isCityCodeRequired(String countryCode, String phoneNumber) {
        if (TURKEY_COUNTRY_CODE.equals(countryCode)
                && StringUtils.startsWith(phoneNumber, TURKEY_PHONE_NUMBER_WITHOUT_CITY_CODE_PREFIX)) {
            return false;
        }
        return true;
    }

    private static Constraint<String, Defect> cityCodeFormatIsValid() {
        // Код города считается допустимым, если он пустая строка (Predicates.empty()) или
        // состоит только из цифр (Predicates.isPositiveWholeNumber()). Т.е. здесь код города может быть пустым.
        // Если эту проверку на пустоту убрать, то валидатор будет ругаться на допустимые номера.
        // Проверка допустимости пустоты для кода города сделана ранее
        return Constraint.fromPredicate(
                Predicates.empty().or(
                        Predicates.isPositiveWholeNumber().and(v -> !"0".equals(v))),
                invalidCityCodeFormat());
    }

    private static Constraint<String, Defect> minPhoneNumberLength(int phoneNumberDigitCount) {
        return Constraint.fromPredicate(
                v -> phoneNumberDigitCount >= PHONE_NUMBER_MIN_LENGTH,
                tooShortPhoneNumber());
    }

    private static Constraint<String, Defect> maxPhoneNumberLength(int phoneNumberDigitCount) {
        return Constraint.fromPredicate(
                v -> phoneNumberDigitCount <= PHONE_NUMBER_MAX_LENGTH,
                tooLongPhoneNumber());
    }

    private static Constraint<Phone, Defect> entirePhoneLengthIsValid(int phoneNumberDigitCount) {
        Predicate<Phone> predicate = phone -> {
            int entirePhoneLength = phone.getCountryCode().length() +
                    phone.getCityCode().length() +
                    phoneNumberDigitCount;
            if (phone.getCountryCode().startsWith("+")) {
                entirePhoneLength--;
            }
            return entirePhoneLength >= ENTIRE_PHONE_MIN_LENGTH &&
                    entirePhoneLength <= ENTIRE_PHONE_MAX_LENGTH;
        };
        return Constraint.fromPredicate(predicate, invalidEntirePhoneLength());
    }

    private static Constraint<Phone, Defect> maxEntirePhoneWithExtensionLength() {
        Predicate<Phone> predicate = phone -> {
            // поле `phone` varchar(25),
            // проверим что длина строки (с разделителями) не превышает 25 (как в perl)
            // и что после преобразования для записи в БД она не превысит это значение
            int originalPhoneLength = StreamEx.of(
                    phone.getCountryCode(), phone.getCityCode(), phone.getPhoneNumber(), phone.getExtension()
            ).map(v -> v == null ? "" : v).joining("#").length();

            int dbPhoneLength = VcardMappings.phoneToDb(phone).length();

            return originalPhoneLength <= ENTIRE_PHONE_WITH_EXTENSION_MAX_LENGTH &&
                    dbPhoneLength <= ENTIRE_PHONE_WITH_EXTENSION_MAX_LENGTH;
        };
        return Constraint.fromPredicate(predicate, tooLongEntirePhoneWithExtension());
    }

    public enum VoidDefectIds implements DefectId<Void> {
        COUNTRY_CODE_IS_NULL,
        /**
         * Код города должен начинаться с '+' (Все кроме 8 800 ... и 8 804 ...)
         */
        COUNTRY_CODE_MUST_START_WITH_PLUS,
        COUNTRY_CODE_FORMAT_IS_INVALID,
        CITY_CODE_IS_NULL,
        CITY_CODE_FORMAT_IS_INVALID,
        PHONE_NUMBER_IS_NULL,
        PHONE_NUMBER_FORMAT_IS_INVALID,
        EXTENSION_FORMAT_IS_INVALID
    }

    public enum CityCodeDefectIds implements DefectId<String> {

        /**
         * Код города не должен начинаться с '+' для 8 800 ... и 8 804 ...
         */
        COUNTRY_CODE_MUST_NOT_START_WITH_PLUS
    }

    public enum StringLengthDefectIds implements DefectId<StringDefectParams> {
        COUNTRY_CODE_IS_EMPTY,
        COUNTRY_CODE_IS_TOO_LONG,

        CITY_CODE_IS_EMPTY,
        CITY_CODE_IS_TOO_LONG,

        PHONE_NUMBER_IS_EMPTY,
        PHONE_NUMBER_IS_TOO_SHORT,
        PHONE_NUMBER_IS_TOO_LONG,

        EXTENSION_IS_TOO_LONG,

        ENTIRE_PHONE_LENGTH_IS_INVALID,
        ENTIRE_PHONE_WITH_EXTENSION_IS_TOO_LONG,
    }

    public static class DefectDefinitions {

        // country code

        public static Defect<Void> nullCountryCode() {
            return new Defect<>(VoidDefectIds.COUNTRY_CODE_IS_NULL);
        }

        public static Defect<StringDefectParams> emptyCountryCode() {
            return new Defect<>(StringLengthDefectIds.COUNTRY_CODE_IS_EMPTY, new StringDefectParams());
        }

        public static Defect<StringDefectParams> tooLongCountryCode() {
            return new Defect<>(StringLengthDefectIds.COUNTRY_CODE_IS_TOO_LONG,
                    new StringDefectParams().withMaxLength(COUNTRY_CODE_MAX_LENGTH));
        }

        public static Defect<Void> invalidCountryCodeFormat() {
            return new Defect<>(VoidDefectIds.COUNTRY_CODE_FORMAT_IS_INVALID);
        }

        public static Defect countryCodeMustStartWithPlus() {
            return new Defect<>(VoidDefectIds.COUNTRY_CODE_MUST_START_WITH_PLUS);
        }

        public static Defect countryCodeMustNotStartWithPlus(String cityCode) {
            return new Defect<>(CityCodeDefectIds.COUNTRY_CODE_MUST_NOT_START_WITH_PLUS, cityCode);
        }

        // city code

        public static Defect<Void> nullCityCode() {
            return new Defect<>(VoidDefectIds.CITY_CODE_IS_NULL);
        }

        public static Defect<StringDefectParams> emptyCityCode() {
            return new Defect<>(StringLengthDefectIds.CITY_CODE_IS_EMPTY, new StringDefectParams());
        }

        public static Defect<StringDefectParams> tooLongCityCode() {
            return new Defect<>(StringLengthDefectIds.CITY_CODE_IS_TOO_LONG,
                    new StringDefectParams().withMaxLength(CITY_CODE_MAX_LENGTH));
        }

        public static Defect<Void> invalidCityCodeFormat() {
            return new Defect<>(VoidDefectIds.CITY_CODE_FORMAT_IS_INVALID);
        }

        // phone number

        public static Defect<Void> nullPhoneNumber() {
            return new Defect<>(VoidDefectIds.PHONE_NUMBER_IS_NULL);
        }

        public static Defect<StringDefectParams> emptyPhoneNumber() {
            return new Defect<>(StringLengthDefectIds.PHONE_NUMBER_IS_EMPTY);
        }

        public static Defect<StringDefectParams> tooShortPhoneNumber() {
            return new Defect<>(StringLengthDefectIds.PHONE_NUMBER_IS_TOO_SHORT,
                    new StringDefectParams().withMinLength(PHONE_NUMBER_MIN_LENGTH));
        }

        public static Defect<StringDefectParams> tooLongPhoneNumber() {
            return new Defect<>(StringLengthDefectIds.PHONE_NUMBER_IS_TOO_LONG,
                    new StringDefectParams().withMaxLength(PHONE_NUMBER_MAX_LENGTH));
        }

        public static Defect<Void> invalidPhoneNumberFormat() {
            return new Defect<>(VoidDefectIds.PHONE_NUMBER_FORMAT_IS_INVALID);
        }

        // extension

        public static Defect<StringDefectParams> tooLongExtension() {
            return new Defect<>(StringLengthDefectIds.EXTENSION_IS_TOO_LONG,
                    new StringDefectParams().withMaxLength(EXTENSION_MAX_LENGTH));
        }

        public static Defect<Void> invalidExtensionFormat() {
            return new Defect<>(VoidDefectIds.EXTENSION_FORMAT_IS_INVALID);
        }

        // entire phone

        public static Defect<StringDefectParams> invalidEntirePhoneLength() {
            return new Defect<>(StringLengthDefectIds.ENTIRE_PHONE_LENGTH_IS_INVALID,
                    new StringDefectParams()
                            .withMinLength(ENTIRE_PHONE_MIN_LENGTH)
                            .withMaxLength(ENTIRE_PHONE_MAX_LENGTH));
        }

        public static Defect<StringDefectParams> tooLongEntirePhoneWithExtension() {
            return new Defect<>(StringLengthDefectIds.ENTIRE_PHONE_WITH_EXTENSION_IS_TOO_LONG,
                    new StringDefectParams().withMaxLength(ENTIRE_PHONE_WITH_EXTENSION_MAX_LENGTH));
        }
    }
}
