package ru.yandex.direct.core.entity.vcard.service.validation;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.ALLOWED_COUNTRY_CITY_CHARS;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.ALLOWED_HOUSE_BUILDING_CHARS;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.APART_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.BUILD_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.CITY_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.COMPANY_NAME_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.CONTACT_EMAIL_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.CONTACT_PERSON_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.COUNTRY_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.DefectDefinitions.contactEmailIsEmpty;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.DefectDefinitions.contactEmailTooLong;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.DefectDefinitions.invalidContactEmailFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.EXTRA_MESSAGE_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.HOUSE_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.AddVcardValidationService.STREET_MAX_LENGTH;
import static ru.yandex.direct.core.entity.vcard.service.validation.InstantMessengerValidator.instantMessengerValidator;
import static ru.yandex.direct.core.entity.vcard.service.validation.OgrnConstraint.ogrnIsValid;
import static ru.yandex.direct.core.entity.vcard.service.validation.PhoneValidator.phoneIsValid;
import static ru.yandex.direct.core.entity.vcard.service.validation.PointOnMapValidator.pointIsValid;
import static ru.yandex.direct.core.entity.vcard.service.validation.WorkTimeConstraint.workTimeIsValid;
import static ru.yandex.direct.core.validation.constraints.Constraints.allowedBannerLetters;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.notEmpty;
import static ru.yandex.direct.validation.constraint.StringConstraints.validEmail;

@ParametersAreNonnullByDefault
public class VcardValidator implements Validator<Vcard, Defect> {
    private final MetroIdValidator metroIdValidator;

    private VcardValidator(MetroIdValidator metroIdValidator) {
        this.metroIdValidator = metroIdValidator;
    }

    public static VcardValidator build(MetroIdValidator metroIdValidator) {
        return new VcardValidator(metroIdValidator);
    }

    @Override
    public ValidationResult<Vcard, Defect> apply(Vcard vcard) {
        ModelItemValidationBuilder<Vcard> vb = ModelItemValidationBuilder.of(vcard);

        vb.item(Vcard.COMPANY_NAME)
                .check(notNull())
                .check(notEmpty(), When.isValid())
                .check(maxStringLength(COMPANY_NAME_MAX_LENGTH), When.isValid())
                .check(allowedBannerLetters(), When.isValid());

        vb.item(Vcard.WORK_TIME)
                .check(notNull())
                .check(workTimeIsValid(), When.isValid());

        vb.item(Vcard.CONTACT_PERSON)
                .check(notEmpty())
                .check(maxStringLength(CONTACT_PERSON_MAX_LENGTH), When.isValid())
                .check(allowedBannerLetters(), When.isValid());

        vb.item(Vcard.PHONE)
                .check(notNull())
                .checkBy(phoneIsValid(), When.notNull());

        vb.item(Vcard.EMAIL)
                .check(notBlank(), contactEmailIsEmpty())
                .check(maxStringLength(CONTACT_EMAIL_MAX_LENGTH), contactEmailTooLong(), When.isValid())
                .check(validEmail(true), invalidContactEmailFormat(), When.isValid());

        vb.item(Vcard.INSTANT_MESSENGER)
                .checkBy(instantMessengerValidator(), When.notNull());

        vb.item(Vcard.EXTRA_MESSAGE)
                .check(notEmpty())
                .check(maxStringLength(EXTRA_MESSAGE_MAX_LENGTH), When.isValid())
                .check(allowedBannerLetters(), When.isValid());

        vb.item(Vcard.OGRN)
                .check(notEmpty())
                .check(ogrnIsValid(), When.isValid());

        // адрес

        vb.item(Vcard.COUNTRY)
                .check(notNull())
                .check(notEmpty(), When.isValid())
                .check(maxStringLength(COUNTRY_MAX_LENGTH), When.isValid())
                .check(ALLOWED_COUNTRY_CITY_CHARS, When.isValid());

        vb.item(Vcard.CITY)
                .check(notNull())
                .check(notEmpty(), When.isValid())
                .check(maxStringLength(CITY_MAX_LENGTH), When.isValid())
                .check(ALLOWED_COUNTRY_CITY_CHARS, When.isValid());

        vb.item(Vcard.STREET)
                .check(notEmpty())
                .check(maxStringLength(STREET_MAX_LENGTH), When.isValid())
                .check(allowedBannerLetters(), When.isValid());

        vb.item(Vcard.HOUSE)
                .check(notEmpty())
                .check(maxStringLength(HOUSE_MAX_LENGTH), When.isValid())
                .check(ALLOWED_HOUSE_BUILDING_CHARS, When.isValid());

        vb.item(Vcard.BUILD)
                .check(notEmpty())
                .check(maxStringLength(BUILD_MAX_LENGTH), When.isValid())
                .check(ALLOWED_HOUSE_BUILDING_CHARS, When.isValid());

        vb.item(Vcard.APART)
                .check(notEmpty())
                .check(maxStringLength(APART_MAX_LENGTH), When.isValid())
                .check(allowedBannerLetters(), When.isValid());

        vb.item(Vcard.METRO_ID)
                .check(metroIdValidator.createConstraintFor(vcard.getCity()), When.notNull());

        vb.item(Vcard.MANUAL_POINT)
                .checkBy(pointIsValid(), When.notNull());

        return vb.getResult();
    }
}
