package ru.yandex.direct.core.entity.vcard.service.validation;

import java.util.List;
import java.util.Objects;
import java.util.function.Predicate;

import com.google.common.base.Splitter;
import one.util.streamex.StreamEx;

import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.params.StringDefectParams;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.vcard.service.validation.WorkTimeConstraint.DefectDefinitions.dayScheduleDuplicated;
import static ru.yandex.direct.core.entity.vcard.service.validation.WorkTimeConstraint.DefectDefinitions.emptyWorktime;
import static ru.yandex.direct.core.entity.vcard.service.validation.WorkTimeConstraint.DefectDefinitions.invalidFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.WorkTimeConstraint.DefectDefinitions.invalidTimeFormat;
import static ru.yandex.direct.core.entity.vcard.service.validation.WorkTimeConstraint.DefectDefinitions.minuteNotDivisibleBy15;
import static ru.yandex.direct.core.entity.vcard.service.validation.WorkTimeConstraint.DefectDefinitions.tooLongWorktime;
import static ru.yandex.direct.validation.Predicates.isPositiveWholeNumber;

/**
 * Описание скопировано со страницы:
 * https://tech.yandex.ru/direct/doc/ref-v5/vcards/add-docpage/
 * <p>
 * Режим работы организации или режим обслуживания клиентов.
 * Задается как строка, в которой указан диапазон дней недели, рабочих часов и минут.
 * Дни недели обозначаются цифрами от 0 до 6, где 0 — понедельник, 6 — воскресенье.
 * Минуты задают кратно 15: 0, 15, 30 или 45. Формат строки: “день_с;день_по;час_с;минуты_с;час_до;мин_до“.
 * <p>
 * Например, строка “0;4;10;0;18;0“ задает такой режим:
 * 0;4 — с понедельника по пятницу;
 * 10;0 — с 10 часов 0 минут;
 * 18;0 — до 18 часов 0 минут.
 * <p>
 * Режим может состоять из нескольких строк указанного формата, например: “0;4;10;0;18;0;5;6;11;0;16;0“.
 * Здесь в дополнение к предыдущему примеру задан режим:
 * 5;6 — с субботы по воскресенье;
 * 11;0 — с 11 часов 0 минут;
 * 16;0 — до 16 часов 0 минут.
 * <p>
 * Круглосуточный режим работы задается строкой “0;6;00;00;00;00“.
 * <p>
 * Не более 255 символов.
 * <p>
 * См. Validation/VCards.pm::validate_vcard_work_time
 */
public class WorkTimeConstraint implements Constraint<String, Defect> {
    static final int WORKTIME_MAX_LENGTH = 255;

    private static final WorkTimeConstraint INSTANCE = new WorkTimeConstraint();

    private static final Splitter OUTER_SPLITTER = Splitter.on(';');
    private static final Splitter INNER_SPLITTER = Splitter.on('#');

    // не может быть валидного времени работы длиной менее 11 символов
    private static final int MINIMAL_VALID_LENGTH = 11;
    private static final String SEPARATOR = ";";
    private static final int POSITIONS_COUNT = 6;
    private static final int POS_DAY_FROM = 0;
    private static final int POS_DAY_TO = 1;
    private static final int POS_HOUR_FROM = 2;
    private static final int POS_HOUR_TO = 4;
    private static final int POS_MINUTE_FROM = 3;
    private static final int POS_MINUTE_TO = 5;

    public static WorkTimeConstraint workTimeIsValid() {
        return INSTANCE;
    }

    /**
     * @return null, если все расписания валидны
     */
    private static Defect validateSchedules(List<DaySchedule> daySchedules) {
        return daySchedules.stream()
                .map(DaySchedule::validate)
                .filter(Objects::nonNull)
                .findFirst()
                .orElse(isDaysDuplicated(daySchedules) ? dayScheduleDuplicated() : null);
    }

    /**
     * @return true, если хотя бы один день повторяется дважды (См. Validation/VCards.pm::validate_vcard_work_time)
     */
    private static boolean isDaysDuplicated(List<DaySchedule> daySchedules) {
        int processedDays = 0;

        for (DaySchedule schedule : daySchedules) {
            int days = 0;

            int minDay = Math.min(schedule.dayFrom, schedule.dayTo);
            int maxDay = Math.max(schedule.dayFrom, schedule.dayTo);
            for (int day = minDay; day <= maxDay; day++) {
                days |= 1 << day;
            }

            // Если дата начала больше даты конца, то считаем что промежуток
            // является объединением двух промежутков [0, schedule.dayFrom] + [schedule.dayTo, 6]
            if (schedule.dayFrom > schedule.dayTo) {
                days = (~days) & 0b1111111;
            }

            // Проверяем встречался ли уже хотя бы один из обработанных дней ранее
            if ((processedDays & days) != 0) {
                return true;
            }

            processedDays |= days;
        }

        return false;
    }

    @Override
    public Defect apply(String workTime) {
        if (workTime == null) {
            return null;
        }

        if (workTime.isEmpty()) {
            return emptyWorktime();
        }

        if (workTime.length() < MINIMAL_VALID_LENGTH) {
            return invalidFormat();
        }

        if (workTime.length() > WORKTIME_MAX_LENGTH) {
            return tooLongWorktime();
        }

        // Разбиваем на части
        List<List<String>> parts = StreamEx.of(
                OUTER_SPLITTER.split(workTime).spliterator())
                .map(INNER_SPLITTER::splitToList)
                .toList();

        // Каждая часть должна содержать в точности POSITIONS_COUNT
        if (parts.stream().anyMatch(l -> l.size() != POSITIONS_COUNT)) {
            return invalidFormat();
        }

        // Каждая часть - целое число, помещающееся в Integer
        Predicate<String> isIntegerPredicate = isPositiveWholeNumber();
        if (parts.stream().anyMatch(l -> !l.stream().allMatch(isIntegerPredicate))) {
            return invalidFormat();
        }

        return validateSchedules(
                parts.stream().map(DaySchedule::fromList).collect(toList()));
    }

    public enum VoidDefectIds implements DefectId<Void> {
        WORKTIME_IS_EMPTY,
        /**
         * Соответствовало в perl-е ситуации, когда расписание не соотвествовало регуряному выражению
         */
        WORKTIME_FORMAT_IS_INVALID,
        /**
         * Если часы или минуты лежат вне допустимого диапазона
         */
        WORKTIME_TIME_FORMAT_IS_INVALID,

        /**
         * Если минуты деляться на 15
         */
        WORKTIME_MINUTE_NOT_DIVISIBLE_BY_15,

        WORKTIME_DAYS_DUPLICATED
    }

    public enum StringDefectIds implements DefectId<StringDefectParams> {
        WORKTIME_IS_TOO_LONG,
    }

    public static class DefectDefinitions {

        static Defect<Void> emptyWorktime() {
            return new Defect<>(VoidDefectIds.WORKTIME_IS_EMPTY);
        }

        static Defect<StringDefectParams> tooLongWorktime() {
            return new Defect<>(StringDefectIds.WORKTIME_IS_TOO_LONG,
                    new StringDefectParams().withMaxLength(WORKTIME_MAX_LENGTH));
        }

        static Defect<Void> invalidFormat() {
            return new Defect<>(VoidDefectIds.WORKTIME_FORMAT_IS_INVALID);
        }

        static Defect<Void> invalidTimeFormat() {
            return new Defect<>(VoidDefectIds.WORKTIME_TIME_FORMAT_IS_INVALID);
        }

        static Defect<Void> minuteNotDivisibleBy15() {
            return new Defect<>(VoidDefectIds.WORKTIME_MINUTE_NOT_DIVISIBLE_BY_15);
        }

        static Defect<Void> dayScheduleDuplicated() {
            return new Defect<>(VoidDefectIds.WORKTIME_DAYS_DUPLICATED);
        }
    }

    private static class DaySchedule {
        int dayFrom;
        int dayTo;
        int hourFrom;
        int hourTo;
        int minuteFrom;
        int minuteTo;

        static DaySchedule fromList(List<String> parts) {
            DaySchedule daySchedule = new DaySchedule();
            daySchedule.dayFrom = Integer.parseInt(parts.get(POS_DAY_FROM));
            daySchedule.dayTo = Integer.parseInt(parts.get(POS_DAY_TO));
            daySchedule.hourFrom = Integer.parseInt(parts.get(POS_HOUR_FROM));
            daySchedule.hourTo = Integer.parseInt(parts.get(POS_HOUR_TO));
            daySchedule.minuteFrom = Integer.parseInt(parts.get(POS_MINUTE_FROM));
            daySchedule.minuteTo = Integer.parseInt(parts.get(POS_MINUTE_TO));
            return daySchedule;
        }

        static boolean isDayInBounds(int day) {
            return day >= 0 && day <= 6;
        }

        static boolean isHourInBounds(int hour) {
            return hour >= 0 && hour <= 23;
        }

        static boolean isMinuteInBounds(int minute) {
            return minute >= 0 && minute <= 59;
        }

        Defect validate() {
            if (!isDayInBounds(dayFrom) || !isDayInBounds(dayTo)) {
                return invalidFormat();
            }

            if (!isHourInBounds(hourFrom)
                    || !isHourInBounds(hourTo)
                    || !isMinuteInBounds(minuteFrom)
                    || !isMinuteInBounds(minuteTo)) {
                return invalidTimeFormat();
            }

            if (minuteFrom % 15 != 0 || minuteTo % 15 != 0) {
                return minuteNotDivisibleBy15();
            }

            /*
            TODO: Возможно стоит раскомментировать (См. DIRECT-70000)
            В perl-е это не проверялось. Если расскомментировать, то будет падать тест AddVCardsFieldsWorkTimeTest
            if (dayTo < dayFrom) {
                return false;
            }

            // круглосуточно
            if (hourFrom == 0 && hourTo == 0 && minuteFrom == 0 && minuteTo == 0) {
                return true;
            }

            if (hourTo < hourFrom) {
                return false;
            }
            if (hourFrom == hourTo && minuteTo <= minuteFrom) {
                return false;
            }
            */

            return null;
        }
    }
}
