package ru.yandex.direct.core.entity.walletparams.repository;


import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.model.AggregatingSumStatus;
import ru.yandex.direct.core.entity.wallet.model.WalletParamsModel;
import ru.yandex.direct.core.entity.walletparams.container.WalletParams;
import ru.yandex.direct.dbschema.ppc.tables.records.WalletCampaignsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsMulticurrencySums.CAMPAIGNS_MULTICURRENCY_SUMS;
import static ru.yandex.direct.dbschema.ppc.tables.WalletCampaigns.WALLET_CAMPAIGNS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Информация о кампаниях типа "общий счёт"
 */
@Repository
@ParametersAreNonnullByDefault
public class WalletParamsRepository {

    private final JooqMapperWithSupplier<WalletParamsModel> walletParamsMapper;
    private final Collection<Field<?>> walletParamsFieldsToRead;

    private static final Field<BigDecimal> SUM_CHIPS_COST_FIELD =
            DSL.ifnull(
                    DSL.sum(CAMPAIGNS_MULTICURRENCY_SUMS.CHIPS_COST),
                    BigDecimal.ZERO
            ).as("sum_chips_cost");

    private final DslContextProvider dslContextProvider;

    @Autowired
    public WalletParamsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        this.walletParamsMapper = JooqMapperWithSupplierBuilder.builder(WalletParamsModel::new)
                .map(property(WalletParamsModel.ID, WALLET_CAMPAIGNS.WALLET_CID))
                .map(property(WalletParamsModel.TOTAL_BALANCE_TID, WALLET_CAMPAIGNS.TOTAL_BALANCE_TID))
                .map(property(WalletParamsModel.TOTAL_CHIPS_COST, WALLET_CAMPAIGNS.TOTAL_CHIPS_COST))
                .map(property(WalletParamsModel.TOTAL_SUM, WALLET_CAMPAIGNS.TOTAL_SUM))
                .map(convertibleProperty(WalletParamsModel.AGGREGATE_MIGRATE_STATUS, WALLET_CAMPAIGNS.IS_SUM_AGGREGATED,
                        AggregatingSumStatus::fromSource, AggregatingSumStatus::toSource))
                .build();

        this.walletParamsFieldsToRead = walletParamsMapper.getFieldsToRead(WalletParamsModel.allModelProperties());
    }

    /**
     * Обновляет запись значениями из {@code walletParams}.
     * currentWalletTid используется для корректной работы с транзакицями:
     * Запись обновится, если {@code currentWalletTid == record.TOTAL_BALANCE_TID and WALLET_CID == walletParams.getWalletId() }
     * <p>
     * deprecated - используется в нотификации из баланса, для новых случаев использовать модель WalletParamsModel
     *
     * @param shard            номер шарда
     * @param currentWalletTid текущее значение id транзакции
     * @param walletParams     запись с данными для обновления
     * @return true, если запись была успешно обновлена. Иначе false (к примеру, WALLET_CID верный, но TOTAL_BALANCE_TID не совпал)
     */
    @Deprecated
    public boolean updateWalletParams(int shard, Long currentWalletTid, WalletParams walletParams) {
        return dslContextProvider.ppc(shard)
                .update(WALLET_CAMPAIGNS)
                .set(WALLET_CAMPAIGNS.TOTAL_SUM, walletParams.getTotalSum())
                .set(WALLET_CAMPAIGNS.TOTAL_BALANCE_TID, walletParams.getTotalBalanceTid())
                .where(WALLET_CAMPAIGNS.WALLET_CID.eq(walletParams.getWalletId())
                        .and(WALLET_CAMPAIGNS.TOTAL_BALANCE_TID.eq(currentWalletTid)))
                .execute() > 0;
    }


    /**
     * Добавляет запись со значениями из {@code walletParams}.
     * <p>
     * deprecated - используется в нотификации из баланса, для новых случаев использовать модель WalletParamsModel
     *
     * @param shard        номер шарда
     * @param walletParams запись с данными для добавления
     */
    @Deprecated
    public void addWalletParams(int shard, WalletParams walletParams) {
        dslContextProvider.ppc(shard)
                .insertInto(WALLET_CAMPAIGNS, WALLET_CAMPAIGNS.TOTAL_SUM, WALLET_CAMPAIGNS.TOTAL_BALANCE_TID,
                        WALLET_CAMPAIGNS.WALLET_CID)
                .values(walletParams.getTotalSum(), walletParams.getTotalBalanceTid(),
                        walletParams.getWalletId())
                .execute();
    }

    /**
     * Количество фишек на общем счету и дочерних кампаниях.
     */
    public BigDecimal getChipsCostForUpdate(DSLContext context, Long walletId) {
        return context
                .select(SUM_CHIPS_COST_FIELD)
                .from(CAMPAIGNS_MULTICURRENCY_SUMS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(CAMPAIGNS_MULTICURRENCY_SUMS.CID))
                .where(CAMPAIGNS.WALLET_CID.eq(walletId).or(CAMPAIGNS.CID.eq(walletId)))
                .forUpdate()
                .fetchOne(SUM_CHIPS_COST_FIELD);
    }

    public boolean update(DSLContext context, Collection<AppliedChanges<WalletParamsModel>> appliedChanges) {
        JooqUpdateBuilder<WalletCampaignsRecord, WalletParamsModel> walletParamsUpdateBuilder =
                new JooqUpdateBuilder<>(WALLET_CAMPAIGNS.WALLET_CID, appliedChanges);

        walletParamsUpdateBuilder.processProperty(WalletParamsModel.TOTAL_CHIPS_COST,
                WALLET_CAMPAIGNS.TOTAL_CHIPS_COST);
        walletParamsUpdateBuilder.processProperty(WalletParamsModel.AGGREGATE_MIGRATE_STATUS,
                WALLET_CAMPAIGNS.IS_SUM_AGGREGATED, AggregatingSumStatus::toSource);

        int rowsUpdated = context.update(WALLET_CAMPAIGNS)
                .set(walletParamsUpdateBuilder.getValues())
                .where(WALLET_CAMPAIGNS.WALLET_CID.in(walletParamsUpdateBuilder.getChangedIds()))
                .execute();

        return rowsUpdated > 0;
    }

    public List<WalletParamsModel> get(int shard, Collection<Long> walletIds) {
        return dslContextProvider.ppc(shard)
                .select(walletParamsFieldsToRead)
                .from(WALLET_CAMPAIGNS)
                .where(WALLET_CAMPAIGNS.WALLET_CID.in(walletIds))
                .fetch(walletParamsMapper::fromDb);
    }
}
