package ru.yandex.direct.core.entity.walletparams.service;

import java.math.BigDecimal;
import java.util.List;

import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.wallet.model.WalletParamsModel;
import ru.yandex.direct.core.entity.walletparams.repository.WalletParamsRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;

@Service
public class WalletParamsService {

    private final WalletParamsRepository walletParamsRepository;
    private final CampaignRepository campaignRepository;
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    @Autowired
    public WalletParamsService(
            WalletParamsRepository walletParamsRepository,
            CampaignRepository campaignRepository,
            DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.walletParamsRepository = walletParamsRepository;
        this.campaignRepository = campaignRepository;
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }


    /**
     * Обновить общее количество фишек для общего счёта и сбросить статус в БК, если было изменение количества.
     *
     * @param walletId идентификатор общего счёта
     */
    public void updateTotalCost(Long walletId) {
        int shard = shardHelper.getShardByCampaignId(walletId);

        TransactionalRunnable transactionalTask = conf -> {
            DSLContext context = conf.dsl();

            List<WalletParamsModel> walletParamsList =
                    walletParamsRepository.get(shard, singletonList(walletId));
            checkState(!walletParamsList.isEmpty(), "params for wallet %d must be exist on update", walletId);
            WalletParamsModel walletParams = walletParamsList.get(0);

            BigDecimal chipsCost = walletParamsRepository.getChipsCostForUpdate(context, walletId);
            AppliedChanges<WalletParamsModel> appliedChanges =
                    createApplyChangesForTotalChipsCost(walletParams, chipsCost);

            boolean updated = walletParamsRepository.update(context, singleton(appliedChanges));
            if (updated) {
                campaignRepository.resetBannerSystemSyncStatus(conf, singleton(walletId));
            }
        };

        dslContextProvider.ppcTransaction(shard, transactionalTask);
    }

    private AppliedChanges<WalletParamsModel> createApplyChangesForTotalChipsCost(WalletParamsModel walletParams,
                                                                                  BigDecimal chipsCost) {
        ModelChanges<WalletParamsModel> changes = new ModelChanges<>(walletParams.getId(), WalletParamsModel.class);
        changes.process(chipsCost, WalletParamsModel.TOTAL_CHIPS_COST);

        return changes.applyTo(walletParams);
    }
}
