package ru.yandex.direct.core.entity.welcomeemail.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record;
import org.jooq.Result;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.welcomeemail.model.NewUsersData;
import ru.yandex.direct.dbschema.ppcdict.tables.records.NewUsersDataRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.NewUsersData.NEW_USERS_DATA;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Репозиторий для работы с welcome-данными пользователей.
 */
@ParametersAreNonnullByDefault
@Repository
public class NewUsersDataRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<NewUsersData> dataMapper;

    @Autowired
    public NewUsersDataRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.dataMapper = JooqMapperWithSupplierBuilder.builder(NewUsersData::new)
                .map(property(NewUsersData.UID, NEW_USERS_DATA.UID))
                .map(property(NewUsersData.LAST_CHANGE, NEW_USERS_DATA.LAST_CHANGE))
                .map(property(NewUsersData.EMAIL, NEW_USERS_DATA.WELCOME_EMAIL))
                .build();
    }

    /**
     * Возвращает временные данные новых пользователей старше {@code borderDateTime}
     *
     * @param borderDateTime
     * @return список данных старше {@code borderDateTime}
     */
    public List<NewUsersData> getEntriesOlderThan(LocalDateTime borderDateTime) {
        Result<Record> records = dslContextProvider.ppcdict()
                .select(dataMapper.getFieldsToRead())
                .from(NEW_USERS_DATA)
                .where(NEW_USERS_DATA.LAST_CHANGE.lessThan(borderDateTime))
                .fetch();
        return records.stream()
                .map(dataMapper::fromDb)
                .collect(Collectors.toList());
    }

    /**
     * Удаляет данные для пользователей из списка {@code uids} старше {@code borderDateTime}
     *
     * @param uids           uid'ы пользователей
     * @param borderDateTime
     * @return число удаленных записей
     */
    public int deleteEntriesForUids(Collection<Long> uids, LocalDateTime borderDateTime) {
        return dslContextProvider.ppcdict()
                .deleteFrom(NEW_USERS_DATA)
                .where(NEW_USERS_DATA.UID.in(uids).and(NEW_USERS_DATA.LAST_CHANGE.lessThan(borderDateTime)))
                .execute();
    }


    /**
     * Добавляет временные данные новых пользователей.
     *
     * @param data добавляемые данные.
     */
    public void addNewUserData(List<NewUsersData> data) {
        if (data.isEmpty()) {
            return;
        }
        InsertHelper<NewUsersDataRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppcdict(), NEW_USERS_DATA);
        for (NewUsersData datum : data) {
            insertHelper.add(dataMapper, datum).newRecord();
        }
        insertHelper.execute();
    }

}
