package ru.yandex.direct.core.entity.xiva;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.xiva.model.XivaPushesQueueItem;
import ru.yandex.direct.core.entity.xiva.repository.XivaPushesQueueRepository;
import ru.yandex.direct.dbschema.ppc.enums.XivaPushesQueuePushType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;

@Service
@ParametersAreNonnullByDefault
public class XivaPushesQueueService {
    private final XivaPushesQueueRepository xivaPushesQueueRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public XivaPushesQueueService(XivaPushesQueueRepository xivaPushQueueRepository, ShardHelper shardHelper) {
        this.xivaPushesQueueRepository = xivaPushQueueRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * Добавляет пуш в очередь.
     * */
    public void addPushToQueue(ClientId clientId, XivaPushesQueuePushType push) {
        int shard = shardHelper.getShardByClientId(clientId);
        xivaPushesQueueRepository.addPushToQueue(shard, clientId.asLong(), push);
    }

    /**
     * Добавляет несколько пушей в очередь.
     * */
    public void addPushesToQueue(Collection<XivaPushesQueueItem> pushes) {
        ShardedData<XivaPushesQueueItem> pushesByShard = shardHelper.groupByShard(
                pushes, ShardKey.CLIENT_ID, push -> ClientId.fromLong(push.getClientId()));

        pushesByShard.forEach(this::addPushesToQueueOnShard);
    }

    /**
     * Добавляет несколько пушей в очередь.
     * Предполагается, что все пуши (точнее получатели) относятся к данному шарду.
     * */
    public void addPushesToQueueOnShard(int shard, Collection<XivaPushesQueueItem> pushes) {
        xivaPushesQueueRepository.addPushesToQueue(shard, pushes);
    }

    /**
     * Получает верхние н пушей из очереди.
     * */
    public List<XivaPushesQueueItem> getPushes(int shard, int count) {
        return xivaPushesQueueRepository.getTopPushes(shard, count);
    }

    /**
     * Удаляет пуш из очереди.
     * */
    public void deletePushFromQueue(int shard, ClientId clientId, XivaPushesQueuePushType push) {
        xivaPushesQueueRepository.deletePush(shard, clientId.asLong(), push);
    }

    /**
     * Удаляет набор пушей из очереди.
     * */
    public void deletePushesFromQueue(int shard, Collection<XivaPushesQueueItem> pushes) {
        xivaPushesQueueRepository.deletePushes(shard, pushes);
    }
}
