package ru.yandex.direct.core.entity.xiva.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.xiva.model.XivaPushesQueueItem;
import ru.yandex.direct.dbschema.ppc.enums.XivaPushesQueuePushType;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static org.jooq.impl.DSL.falseCondition;
import static ru.yandex.direct.dbschema.ppc.tables.XivaPushesQueue.XIVA_PUSHES_QUEUE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
@ParametersAreNonnullByDefault
public class XivaPushesQueueRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<XivaPushesQueueItem> jooqMapper;

    @Autowired
    public XivaPushesQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(XivaPushesQueueItem::new)
                .map(property(XivaPushesQueueItem.CLIENT_ID, XIVA_PUSHES_QUEUE.CLIENT_ID))
                .map(convertibleProperty(XivaPushesQueueItem.PUSH_TYPE, XIVA_PUSHES_QUEUE.PUSH_TYPE,
                        ru.yandex.direct.core.entity.xiva.model.XivaPushesQueuePushType::fromSource,
                        ru.yandex.direct.core.entity.xiva.model.XivaPushesQueuePushType::toSource))
                .readProperty(XivaPushesQueueItem.ADD_TIME, fromField(XIVA_PUSHES_QUEUE.ADD_TIME))
                .build();
    }

    /**
     * Добавляет пуш в очередь.
     *
     * @param shard номер шарда (очереди), куда класть пуш
     * @param clientId id клиент в строковом формате
     * @param push сам пуш
     * */
    public int addPushToQueue(
            int shard, Long clientId, XivaPushesQueuePushType push) {
        return dslContextProvider.ppc(shard)
                .insertInto(XIVA_PUSHES_QUEUE,
                        XIVA_PUSHES_QUEUE.CLIENT_ID, XIVA_PUSHES_QUEUE.PUSH_TYPE, XIVA_PUSHES_QUEUE.ADD_TIME)
                .values(clientId, push, LocalDateTime.now())
                .onDuplicateKeyIgnore()
                .execute();
    }

    public int addPushesToQueue(int shard, Collection<XivaPushesQueueItem> pushes) {
        return new InsertHelper<>(dslContextProvider.ppc(shard), XIVA_PUSHES_QUEUE)
                .addAll(jooqMapper, pushes)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }

    @QueryWithoutIndex("")
    public List<XivaPushesQueueItem> getTopPushes(int shard, int count) {
        return dslContextProvider.ppc(shard)
                .select(XIVA_PUSHES_QUEUE.CLIENT_ID, XIVA_PUSHES_QUEUE.PUSH_TYPE, XIVA_PUSHES_QUEUE.ADD_TIME)
                .from(XIVA_PUSHES_QUEUE)
                .orderBy(XIVA_PUSHES_QUEUE.ADD_TIME.asc())
                .limit(count)
                .fetch(jooqMapper::fromDb);
    }

    public int deletePush(int shard, Long clientId, XivaPushesQueuePushType push) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(XIVA_PUSHES_QUEUE)
                .where(XIVA_PUSHES_QUEUE.CLIENT_ID.eq(clientId))
                .and(XIVA_PUSHES_QUEUE.PUSH_TYPE.eq(push))
                .execute();
    }

    public int deletePushes(int shard, Collection<XivaPushesQueueItem> pushes) {
        Condition deleteCondition = falseCondition();
        for (XivaPushesQueueItem push : pushes) {
            XivaPushesQueuePushType pushType = XivaPushesQueuePushType.valueOf(push.getPushType().toString());
            deleteCondition = deleteCondition.or(
                    XIVA_PUSHES_QUEUE.CLIENT_ID.eq(push.getClientId())
                            .and(XIVA_PUSHES_QUEUE.PUSH_TYPE.eq(pushType))
            );
        }

        return dslContextProvider.ppc(shard)
                .deleteFrom(XIVA_PUSHES_QUEUE)
                .where(deleteCondition)
                .execute();
    }
}
