package ru.yandex.direct.core.entity.yandexagencyorder.repository;

import java.util.List;

import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.yandexagencyorder.model.Status;
import ru.yandex.direct.core.entity.yandexagencyorder.model.YandexAgencyOrder;
import ru.yandex.direct.dbschema.ppc.tables.records.YandexAgencyOrdersRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.dbschema.ppc.tables.YandexAgencyOrders.YANDEX_AGENCY_ORDERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class YandexAgencyOrdersRepository {
    private final DslContextProvider databaseWrapperProvider;
    private final JooqMapperWithSupplier<YandexAgencyOrder> yandexAgencyOrderMapper;

    public YandexAgencyOrdersRepository(DslContextProvider databaseWrapperProvider) {
        this.databaseWrapperProvider = databaseWrapperProvider;
        this.yandexAgencyOrderMapper = JooqMapperWithSupplierBuilder.builder(YandexAgencyOrder::new)
                .map(property(YandexAgencyOrder.ID, YANDEX_AGENCY_ORDERS.YA_ORDER_ID))
                .map(property(YandexAgencyOrder.CLIENT_ID, YANDEX_AGENCY_ORDERS.CLIENT_ID))
                .map(property(YandexAgencyOrder.CREATED, YANDEX_AGENCY_ORDERS.CREATED))
                .map(property(YandexAgencyOrder.LAST_CHANGE, YANDEX_AGENCY_ORDERS.LAST_CHANGE))
                .map(property(YandexAgencyOrder.PRODUCT_TYPE, YANDEX_AGENCY_ORDERS.PRODUCT_TYPE))
                .map(convertibleProperty(YandexAgencyOrder.YA_ORDER_STATUS, YANDEX_AGENCY_ORDERS.YA_ORDER_STATUS,
                        Status::fromSource, Status::toSource))
                .build();
    }

    /**
     * Обновление заказов Яндекс.Агенств
     */
    public void updateYandexOrder(int shard, List<AppliedChanges<YandexAgencyOrder>> appliedChanges) {
        JooqUpdateBuilder<YandexAgencyOrdersRecord, YandexAgencyOrder> ub =
                new JooqUpdateBuilder<>(YANDEX_AGENCY_ORDERS.YA_ORDER_ID, appliedChanges);

        ub.processProperty(YandexAgencyOrder.YA_ORDER_STATUS, YANDEX_AGENCY_ORDERS.YA_ORDER_STATUS,
                Status::toSource);
        ub.processProperty(YandexAgencyOrder.LAST_CHANGE, YANDEX_AGENCY_ORDERS.LAST_CHANGE,
                localDateTime -> localDateTime);
        ub.processProperty(YandexAgencyOrder.PRODUCT_TYPE, YANDEX_AGENCY_ORDERS.PRODUCT_TYPE);


        databaseWrapperProvider.ppc(shard)
                .update(YANDEX_AGENCY_ORDERS)
                .set(ub.getValues())
                .where(YANDEX_AGENCY_ORDERS.YA_ORDER_ID.in(ub.getChangedIds()))
                .execute();
    }

    /**
     * Добавление заказов Яндекс.Агенств
     */
    public List<Long> addYandexAgencyOrders(int shard, List<YandexAgencyOrder> yandexAgencyOrderList) {
        addYandexAgencyOrdersWithId(shard, yandexAgencyOrderList);
        return mapList(yandexAgencyOrderList, YandexAgencyOrder::getId);
    }

    /**
     * Получение заказов Яндекс.Агенств по id заказа.
     * id появляется только при отправе в баланс. И этого id может не быть.
     */
    public List<YandexAgencyOrder> getYandexAgencyOrdersByOrderId(int shard, List<Long> ids) {
        return databaseWrapperProvider.ppc(shard)
                .select(yandexAgencyOrderMapper.getFieldsToRead())
                .from(YANDEX_AGENCY_ORDERS)
                .where(YANDEX_AGENCY_ORDERS.YA_ORDER_ID.in(ids))
                .fetch(yandexAgencyOrderMapper::fromDb);
    }

    private void addYandexAgencyOrdersWithId(int shard, List<YandexAgencyOrder> yandexAgencyOrderList) {
        InsertHelper<YandexAgencyOrdersRecord> insertHelper =
                new InsertHelper<>(databaseWrapperProvider.ppc(shard), YANDEX_AGENCY_ORDERS);
        yandexAgencyOrderList.forEach(adGroup -> insertHelper.add(yandexAgencyOrderMapper, adGroup).newRecord());
        insertHelper.execute();
    }

}
