package ru.yandex.direct.core.entity.zora;

import java.time.Duration;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import org.asynchttpclient.Request;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.utils.AutoCloseableList;
import ru.yandex.direct.utils.Transient;
import ru.yandex.direct.zorafetcher.ZoraFetcher;
import ru.yandex.direct.zorafetcher.ZoraFetcherSettings;
import ru.yandex.direct.zorafetcher.ZoraGoRequest;
import ru.yandex.direct.zorafetcher.ZoraGoRequestCreator;
import ru.yandex.direct.zorafetcher.ZoraOnlineRequest;
import ru.yandex.direct.zorafetcher.ZoraOnlineRequestCreator;
import ru.yandex.direct.zorafetcher.ZoraRequest;
import ru.yandex.direct.zorafetcher.ZoraRequestCreator;
import ru.yandex.direct.zorafetcher.ZoraResponse;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.common.db.PpcPropertyNames.USE_GO_ZORA;
import static ru.yandex.direct.tracing.util.TraceUtil.traceToHeader;
import static ru.yandex.direct.tvm.TvmService.ZORA_GO;
import static ru.yandex.direct.tvm.TvmService.ZORA_ONLINE;

@Service
public class ZoraService {

    private static final Logger logger = LoggerFactory.getLogger(ZoraService.class);

    private final ZoraFetcher zoraFetcher;
    private final TvmIntegration tvmIntegration;
    private final ZoraGoRequestCreator zoraGoRequestCreator;
    private final ZoraOnlineRequestCreator zoraOnlineRequestCreator;

    private final Duration defaultRequestTimeout;

    private final PpcProperty<Boolean> useGoZoraProperty;

    @Autowired
    public ZoraService(ZoraFetcher zoraFetcher,
                       TvmIntegration tvmIntegration,
                       PpcPropertiesSupport ppcPropertiesSupport,
                       ZoraGoRequestCreator zoraGoRequestCreator,
                       ZoraOnlineRequestCreator zoraOnlineRequestCreator,
                       ZoraFetcherSettings zoraFetcherSettings) {
        this.zoraFetcher = zoraFetcher;
        this.tvmIntegration = tvmIntegration;
        this.zoraGoRequestCreator = zoraGoRequestCreator;
        this.zoraOnlineRequestCreator = zoraOnlineRequestCreator;

        this.defaultRequestTimeout = zoraFetcherSettings.getRequestTimeout();

        this.useGoZoraProperty = ppcPropertiesSupport.get(USE_GO_ZORA, Duration.ofMinutes(2));
    }

    public Map<String, Result<ZoraResponse>> fetchByUrl(List<String> urls,
                                                        boolean followRedirects,
                                                        @Nullable String userAgent) {
        logger.debug("Urls to fetch: {}", urls);

        boolean useGoZora = useGoZora();

        String tvmTicket = getTvmTicket(useGoZora);
        try (TraceProfile traceProfile = Trace.current().profile("zora", "fetch", urls.size());
             AutoCloseableList<TraceChild> traces = new AutoCloseableList<>()) {
            List<ZoraRequest> requests = EntryStream.of(urls)
                    .mapKeyValue((id, url) -> {
                        TraceChild child = Trace.current().child("zora", "get");
                        traces.add(new Transient<>(child));
                        return createZoraRequest(id,
                                getRequestCreator(useGoZora).createGetRequest(url, traceToHeader(child),
                                        defaultRequestTimeout, tvmTicket, followRedirects, userAgent), useGoZora);
                    })
                    .toList();
            logger.debug("Requests: {}", requests);

            Map<Long, Result<ZoraResponse>> resultMap = zoraFetcher.fetch(requests, useGoZora);
            logger.debug("Response: {}", resultMap);

            return EntryStream.of(resultMap)
                    .mapKeys(idx -> urls.get(idx.intValue()))
                    .toMap();
        }
    }

    public Map<String, Result<ZoraResponse>> fetchByUrl(List<String> urls, boolean followRedirects) {
        return fetchByUrl(urls, followRedirects, null);
    }

    public Result<ZoraResponse> fetchByUrl(String url, boolean followRedirects) {
        return fetchByUrl(url, followRedirects, null);
    }

    public Result<ZoraResponse> fetchByUrl(String url, boolean followRedirects, @Nullable String userAgent) {
        return this.fetchByUrl(singletonList(url), followRedirects, userAgent).get(url);
    }

    public Result<ZoraResponse> fetch(ZoraRequest request, boolean useGoZora) {
        return zoraFetcher.fetch(request, useGoZora);
    }

    public Request createGetRequest(String url,
                                    String yandexTraceHeaderValue,
                                    Duration responseTimeout,
                                    String tvmTicket,
                                    boolean followRedirects,
                                    boolean useGoZora) {
        return createGetRequest(url, yandexTraceHeaderValue, responseTimeout, tvmTicket, followRedirects, useGoZora, null);
    }

    public Request createGetRequest(String url,
                                    String yandexTraceHeaderValue,
                                    Duration responseTimeout,
                                    String tvmTicket,
                                    boolean followRedirects,
                                    boolean useGoZora,
                                    String userAgent) {
        return getRequestCreator(useGoZora).createGetRequest(url, yandexTraceHeaderValue, responseTimeout, tvmTicket,
                followRedirects, userAgent);
    }

    private ZoraRequestCreator getRequestCreator(boolean useGoZora) {
        if (useGoZora) {
            return zoraGoRequestCreator;
        } else {
            return zoraOnlineRequestCreator;
        }
    }

    public String getTvmTicket(boolean useGoZora) {
        return tvmIntegration.getTicket(getTvmService(useGoZora));
    }

    private TvmService getTvmService(boolean useGoZora) {
        if (useGoZora) {
            return ZORA_GO;
        } else {
            return ZORA_ONLINE;
        }
    }

    public ZoraRequest createZoraRequest(long id,
                                         Request request,
                                         boolean useGoZora) {
        if (useGoZora) {
            return new ZoraGoRequest(id, request);
        } else {
            return new ZoraOnlineRequest(id, request);
        }
    }

    public boolean useGoZora() {
        return useGoZoraProperty.getOrDefault(false);
    }
}
