package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.buildRetargetingCondition
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toAssetLinkStatuses
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toAssetLinks
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toEDeviceTypes
import ru.yandex.direct.core.entity.uac.converter.UacGrutCampaignConverter.toTAdjustments
import ru.yandex.direct.core.entity.uac.converter.toUacAdGroupBrief
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.model.DeviceType
import ru.yandex.direct.core.entity.uac.model.Socdem
import ru.yandex.direct.core.entity.uac.model.UacAdjustment
import ru.yandex.direct.core.entity.uac.model.UacRetargetingCondition
import ru.yandex.direct.core.entity.uac.model.UacShowsFrequencyLimit
import ru.yandex.direct.core.entity.uac.model.relevance_match.UacRelevanceMatch
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacBrandsafety
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacCampaignMeasurer
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacCpmAsset
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacDisabledPlaces
import ru.yandex.direct.core.entity.uac.repository.ydb.model.UacYdbCampaignContent
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.AdGroupBrief
import ru.yandex.grut.objects.proto.AssetLink
import ru.yandex.grut.objects.proto.Campaign
import ru.yandex.grut.objects.proto.client.Schema

data class AdGroupBriefGrutModel(
    val id: Long? = null,
    val campaignId: Long,
    val name: String? = null,
    val url: String? = null,
    val regions: List<Long>? = null,
    val hyperGeoId: Long? = null,
    val minusRegions: List<Long>? = null,
    val adjustments: List<UacAdjustment>? = null,
    val keywords: List<String>? = null,
    val minusKeywords: List<String>? = null,
    val socdem: Socdem? = null,
    val deviceTypes: Set<DeviceType>? = null,
    val retargetingCondition: UacRetargetingCondition? = null,
    val videosAreNonSkippable: Boolean? = null,
    val catalogIds: List<Long>? = null,
    val brandSurveyId: String? = null,
    val showsFrequencyLimit: UacShowsFrequencyLimit? = null,
    val cpmAssets: Map<String, UacCpmAsset>? = null,
    val campaignMeasurers: List<UacCampaignMeasurer>? = null,
    val uacBrandsafety: UacBrandsafety? = null,
    val uacDisabledPlaces: UacDisabledPlaces? = null,
    val relevanceMatch: UacRelevanceMatch? = null,
    val assetLinks: List<UacYdbCampaignContent>? = null,
    val adGroupIds: List<Long>? = null,

    // Не используется в таблице GrUT'а.
    // Возможно костыльное решение и нужно поддержать честный тип заявки
    // https://st.yandex-team.ru/DIRECT-173973
    val isCpmBrief: Boolean,
)

/**
 * API для работы с групповыми заявками (таблица ad_group_brief в GRuT)
 */
class AdGroupBriefGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<AdGroupBriefGrutModel>(grutContext, Schema.EObjectType.OT_AD_GROUP_BRIEF, properties) {

    private companion object {
        const val DEFAULT_FETCH_LIMIT = 2000L
        const val PATCH_AD_GROUP_IDS = "/status/ad_group_ids"
        const val PATCH_CATALOG_IDS = "/spec/catalog_ids"
    }

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TAdGroupBriefMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TAdGroupBriefMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: AdGroupBriefGrutModel): ByteString {
        return Schema.TAdGroupBriefMeta.newBuilder().apply {
            obj.id?.let { id = it }
            campaignId = obj.campaignId
        }.build().toByteString()
    }

    override fun serializeSpec(obj: AdGroupBriefGrutModel): ByteString {
        return AdGroupBrief.TAdGroupBriefSpec.newBuilder().apply {
            obj.name?.let { name = it }
            obj.url?.let { url = it }
            obj.regions?.let { addAllRegions(it) }
            obj.hyperGeoId?.let { hypergeoId = it }
            obj.minusRegions?.let { addAllMinusRegions(it) }
            obj.adjustments?.let { addAllAdjustments(it.toTAdjustments()) }
            obj.keywords?.let { addAllKeywords(it) }
            obj.minusKeywords?.let { addAllMinusKeywords(it) }
            obj.socdem?.let { socdem = UacGrutCampaignConverter.buildSocdem(it) }
            obj.deviceTypes?.let { addAllDeviceType(it.toEDeviceTypes()) }
            obj.videosAreNonSkippable?.let { videosAreNonSkippable = it }
            obj.catalogIds?.let { addAllCatalogIds(it) }
            buildCpmData(obj)?.let { cpmData = it }
            obj.relevanceMatch?.let { relevanceMatch = UacGrutCampaignConverter.toTRelevanceMatch(it) }
            obj.retargetingCondition?.let { retargetingCondition = buildRetargetingCondition(it) }
            briefAssetLinks = AssetLink.TAssetLinks.newBuilder().apply {
                obj.assetLinks?.let { addAllLinks(it.toAssetLinks()) }
            }.build()
            briefAssetLinksStatuses = AssetLink.TAssetLinksStatuses.newBuilder().apply {
                obj.assetLinks?.let { addAllLinkStatuses(it.toAssetLinkStatuses()) }
            }.build()
        }.build().toByteString()
    }

    override fun serializeStatus(obj: AdGroupBriefGrutModel): ByteString? {
        return AdGroupBrief.TAdGroupBriefStatus.newBuilder().apply {
            obj.adGroupIds?.let { addAllAdGroupIds(it) }
        }.build().toByteString()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TAdGroupBrief.parseFrom(rawMeta).meta.id
    }

    fun getAdGroupBrief(id: Long): AdGroupBriefGrutModel? {
        return getObjectAs(id, ::transformToAdGroupBriefModel)
    }

    fun getAdGroupBriefs(ids: Set<Long>): List<AdGroupBriefGrutModel> {
        val objects = getObjectsByIds(ids)
        return objects.filter { it.protobuf.size() > 0 }.map { transformToAdGroupBriefModel(it)!! }
    }

    fun selectAdGroupBriefsByCampaignId(
        campaignId: Long,
        limit: Long = DEFAULT_FETCH_LIMIT
    ): List<AdGroupBriefGrutModel> {
        return selectObjectsAs(
            filter = "[/meta/campaign_id] = $campaignId",
            limit = limit,
            transform = ::transformToAdGroupBriefModel
        )
    }

    fun selectTAdGroupBriefsByCampaignIds(
        campaignIds: Set<Long>,
        attributeSelector: List<String> = listOf("/meta", "/spec", "/status"),
    ): List<Schema.TAdGroupBrief> = selectObjectsAs(
        filter = "[/meta/campaign_id] IN (${campaignIds.joinToString { "$it" }})",
        attributeSelector = attributeSelector,
        transform = ::transformToTAdGroupBriefModel,
    )

    fun selectAdGroupBriefs(
        filter: String,
        attributeSelector: List<String> = listOf("/meta", "/spec", "/status"),
        index: String? = null,
        limit: Long = DEFAULT_FETCH_LIMIT,
        continuationToken: String? = null,
        allowFullScan: Boolean = false
    ): List<AdGroupBriefGrutModel> {
        return selectObjectsAs(
            filter,
            attributeSelector,
            index,
            limit,
            continuationToken,
            allowFullScan,
            ::transformToAdGroupBriefModel
        )
    }

    fun createAdGroupBriefs(adGroupBriefs: List<AdGroupBriefGrutModel>): List<Long> {
        return createObjects(adGroupBriefs)
    }

    private val setPaths = listOf("/spec")
    fun createOrUpdateAdGroupBrief(adGroupBriefs: AdGroupBriefGrutModel) {
        createOrUpdateObject(adGroupBriefs, setPaths)
    }

    /**
     * Обновляет в заявке список каталогов и id групп
     */
    fun updateCatalogsAndGroups(uacAdGroupBriefs: AdGroupBriefGrutModel) {
        val removePaths = mutableListOf<String>()
        val addPaths = mutableListOf<String>()

        if (uacAdGroupBriefs.adGroupIds.isNullOrEmpty()) {
            removePaths.add(PATCH_AD_GROUP_IDS)
        } else {
            addPaths.add(PATCH_AD_GROUP_IDS)
        }

        if (uacAdGroupBriefs.catalogIds.isNullOrEmpty()) {
            removePaths.add(PATCH_CATALOG_IDS)
        } else {
            addPaths.add(PATCH_CATALOG_IDS)
        }

        createOrUpdateObject(
            uacAdGroupBriefs,
            addPaths,
            removePaths,
        )
    }

    /**
     * Обновляет в заявке список id групп
     */
    fun updateAdGroupIds(uacAdGroupBrief: AdGroupBriefGrutModel) {
        val removePaths = mutableListOf<String>()
        val addPaths = mutableListOf<String>()

        if (uacAdGroupBrief.adGroupIds.isNullOrEmpty()) {
            removePaths.add(PATCH_AD_GROUP_IDS)
        } else {
            addPaths.add(PATCH_AD_GROUP_IDS)
        }

        createOrUpdateObject(
            uacAdGroupBrief,
            addPaths,
            removePaths,
        )
    }

    fun updateRetargetingConditions(adGroupBriefs: List<AdGroupBriefGrutModel>) {
        updateObjects(
            objects = adGroupBriefs,
            setPaths = listOf("/spec/retargeting_condition"),
            removePaths = listOf()
        )
    }

    fun createOrUpdateAdGroupBriefs(adGroupBriefs: List<AdGroupBriefGrutModel>) {
        createOrUpdateObjects(adGroupBriefs, setPaths)
    }

    fun deleteAdGroupBriefs(ids: Collection<Long>) {
        deleteObjects(ids, true)
    }

    private fun buildCpmData(adGroupBrief: AdGroupBriefGrutModel): Campaign.TCampaignBrief.TCpmData? {
        if (!adGroupBrief.isCpmBrief) {
            return null
        }

        return Campaign.TCampaignBrief.TCpmData.newBuilder().apply {
            adGroupBrief.brandSurveyId?.let { brandSurveyId = it }
            adGroupBrief.retargetingCondition?.let { retargetingCondition ->
                retargetingConditionBuilder.apply {
                    retargetingCondition.name?.let { name = it }
                    retargetingCondition.id?.let { id = it }
                    addAllConditionRules(retargetingCondition.conditionRules.map {
                        UacGrutCampaignConverter.buildRetargetingConditionRule(
                            it
                        )
                    })
                }
            }
            adGroupBrief.showsFrequencyLimit?.let { showsFreqLimit ->
                showsFrequencyLimitBuilder.apply {
                    impressionRateCount = showsFreqLimit.impressionRateCount
                    impressionRateIntervalDays = showsFreqLimit.impressionRateIntervalDays
                }
            }
            adGroupBrief.cpmAssets?.let { assets ->
                putAllCpmAssets(assets.mapValues { UacGrutCampaignConverter.buildCpmAsset(it.value) })
            }

            adGroupBrief.campaignMeasurers?.let { measurers ->
                addAllCampaignMeasurers(
                    measurers.map { UacGrutCampaignConverter.buildCampaignMeasurer(it) }
                )
            }

            adGroupBrief.uacBrandsafety?.let {
                brandSafety = UacGrutCampaignConverter.buildBrandSafety(it)
            }

            adGroupBrief.uacDisabledPlaces?.let {
                disabledPlaces = UacGrutCampaignConverter.buildDisabledPlaces(it);
            }
        }.build()
    }

    private fun transformToTAdGroupBriefModel(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TAdGroupBrief? {
        if (raw == null) return null
        return Schema.TAdGroupBrief.parseFrom(raw.protobuf) ?: return null
    }

    private fun transformToAdGroupBriefModel(raw: ObjectApiServiceOuterClass.TVersionedPayload?): AdGroupBriefGrutModel? {
        if (raw == null) return null
        return Schema.TAdGroupBrief.parseFrom(raw.protobuf).toUacAdGroupBrief()
    }
}
