package ru.yandex.direct.core.grut.api

import java.time.Duration
import java.time.LocalDateTime
import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupDeviceTypeTargeting
import ru.yandex.direct.core.entity.adgroup.model.MobileContentAdGroupNetworkTargeting
import ru.yandex.direct.core.entity.adgroup.model.PageBlock
import ru.yandex.direct.core.entity.adgroupadditionaltargeting.model.AdGroupAdditionalTargeting
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatchCategory
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.utils.AdGroupAdditionalTargetingConverter.Companion.transformPageBlock
import ru.yandex.direct.core.grut.api.utils.AdGroupAdditionalTargetingConverter.Companion.transformTargeting
import ru.yandex.direct.core.grut.api.utils.MAX_RF_RESET
import ru.yandex.direct.core.grut.api.utils.moscowDateTimeToGrut
import ru.yandex.direct.mysql2grut.enummappers.AdGroupEnumMappers
import ru.yandex.direct.mysql2grut.enummappers.AdGroupEnumMappers.Companion.toGrutMobileDeviceType
import ru.yandex.direct.mysql2grut.enummappers.AdGroupEnumMappers.Companion.toGrutMobileNetworkType
import ru.yandex.direct.utils.TimeConvertUtils
import ru.yandex.grut.auxiliary.proto.RfOptions
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TVersionedPayload
import ru.yandex.grut.objects.proto.AdGroupV2
import ru.yandex.grut.objects.proto.client.Schema

class AdGroupGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<AdGroupGrut>(grutContext, Schema.EObjectType.OT_AD_GROUP_V2, properties) {

    companion object {
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)
        private val GET_TIMEOUT = Duration.ofMinutes(1)
    }

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TAdGroupV2Meta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TAdGroupV2Meta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: AdGroupGrut): ByteString {
        return Schema.TAdGroupV2Meta.newBuilder()
            .setId(obj.id)
            .setDirectType(AdGroupEnumMappers.toGrutAdGroupType(obj.type).number)
            .setCampaignId(obj.orderId)
            .build()
            .toByteString()
    }

    override fun serializeSpec(obj: AdGroupGrut): ByteString {
        return AdGroupV2.TAdGroupV2Spec.newBuilder()
            .apply {
                if (obj.name != null) {
                    name = obj.name
                }
                if (obj.regions.isNotEmpty()) {
                    addAllRegionsIds(obj.regions)
                }
                if (obj.internalLevel != null) {
                    internalLevel = obj.internalLevel.toInt()
                }
                if (obj.pageGroupTags.isNotEmpty()) {
                    addAllPageGroupTags(obj.pageGroupTags)
                }
                if (obj.targetTags.isNotEmpty()) {
                    addAllTargetTags(obj.targetTags)
                }
                if (obj.trackingParams != null) {
                    trackingParams = obj.trackingParams
                }
                if (obj.priority != null) {
                    matchPriority = obj.priority.toInt()
                }

                if (obj.mobileContent != null) {
                    mobileContentId = obj.mobileContent.mobileContentId
                    mobileContentDetails = AdGroupV2.TAdGroupV2Spec.TMobileContentDetails.newBuilder().apply {
                        if (obj.mobileContent.deviceTypeTargeting.isNotEmpty()) {
                            addAllDeviceTypes(obj.mobileContent.deviceTypeTargeting.map { toGrutMobileDeviceType(it).number })
                        }
                        if (obj.mobileContent.networkTargeting.isNotEmpty()) {
                            addAllNetworkTypes(obj.mobileContent.networkTargeting.map { toGrutMobileNetworkType(it).number })
                        }
                        storeUrl = obj.mobileContent.storeUrl
                        minOsVersion = obj.mobileContent.minOsVersion
                    }.build()
                }

                if (obj.relevanceMatchData.isNotEmpty()) {
                    relevanceMatch = AdGroupV2.TAdGroupV2Spec.TRelevanceMatch.newBuilder()
                        .addAllCategories(obj.relevanceMatchData.map {
                            AdGroupEnumMappers.toGrutRelevanceMatchCategory(it).number
                        })
                        .build()
                }
                if (obj.internalAdGroupOptions != null) {
                    rfOptions = RfOptions.TRfOptions.newBuilder().apply {
                        shows = RfOptions.TRfOptions.TLimit.newBuilder().apply {
                            count = obj.internalAdGroupOptions.rf ?: 0
                            val periodDays = obj.internalAdGroupOptions.rfReset ?: MAX_RF_RESET
                            period = TimeConvertUtils.daysToSecond(periodDays).toInt()
                        }.build()
                        clicks = RfOptions.TRfOptions.TLimit.newBuilder().apply {
                            count = obj.internalAdGroupOptions.maxClicksCount ?: 0
                            period = obj.internalAdGroupOptions.maxClicksPeriod ?: 0
                        }.build()
                        closes = RfOptions.TRfOptions.TLimit.newBuilder().apply {
                            count = obj.internalAdGroupOptions.maxStopsCount ?: 0
                            period = obj.internalAdGroupOptions.maxStopsPeriod ?: 0
                        }.build()
                    }.build()
                    if(obj.internalAdGroupOptions.startTime != null) {
                        startTime = moscowDateTimeToGrut(obj.internalAdGroupOptions.startTime)
                    }
                    if(obj.internalAdGroupOptions.finishTime != null) {
                        finishTime = moscowDateTimeToGrut(obj.internalAdGroupOptions.finishTime)
                    }
                }
                if (obj.pageTargets.isNotEmpty()) {
                    addAllPageBlocks(obj.pageTargets.map { transformPageBlock(it) }.toList())
                }
                if (obj.minusPhrasesIds.isNotEmpty()) {
                    addAllMinusPhrasesIds(obj.minusPhrasesIds)
                }
                if (obj.targeting.isNotEmpty()) {
                    addAllAdditionalTargetings(obj.targeting
                        .map { transformTargeting(it, obj.cryptaSegmentsMapping) }.toList())
                }
                if (obj.biddableShowConditionsIds.isNotEmpty()) {
                    addAllBiddableShowConditionsIds(obj.biddableShowConditionsIds)
                }
                if (obj.bidModifiersIds.isNotEmpty()) {
                    addAllBidModifiersIds(obj.bidModifiersIds)
                }
            }
            .build()
            .toByteString()
    }


    private val setPaths = listOf("/spec")
    fun createOrUpdateAdGroups(adGroups: List<AdGroupGrut>) {
        createOrUpdateObjects(adGroups, setPaths)
    }

    fun createOrUpdateAdGroup(adGroup: AdGroupGrut) {
        createOrUpdateObject(adGroup, setPaths)
    }

    fun createOrUpdateAdGroupsParallel(adGroups: List<AdGroupGrut>) {
        createOrUpdateObjectsParallel(adGroups, UPDATE_TIMEOUT, setPaths)
    }

    fun getAdGroup(id: Long): Schema.TAdGroupV2? {
        return getObjectAs(id, ::transformToAdGroup)
    }

    fun getAdGroups(ids: Collection<Long>): List<Schema.TAdGroupV2> {
        return getObjectsByIds(ids, skipNonexistent = true).map { transformToAdGroup(it)!! }
    }

    fun getExistingAdGroupsParallel(ids: Collection<Long>): List<Long> {
        return getExistingObjectsParallel(ids, GET_TIMEOUT)
    }

    private fun transformToAdGroup(raw: TVersionedPayload?): Schema.TAdGroupV2? {
        if (raw == null) return null
        return Schema.TAdGroupV2.parseFrom(raw.protobuf)
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TAdGroupV2.parseFrom(rawMeta).meta.id
    }
}

data class AdGroupGrut(
    val id: Long,
    val orderId: Long,
    val name: String? = null,
    val type: AdGroupType,
    val pageGroupTags: List<String> = emptyList(),
    val targetTags: List<String> = emptyList(),
    val relevanceMatchData: Set<RelevanceMatchCategory> = emptySet(),
    val internalAdGroupOptions: InternalAdGroupOptions? = null,
    val internalLevel: Long? = null,
    val minusPhrasesIds: Collection<Long> = emptyList(),
    val regions: List<Long> = emptyList(),
    val targeting: List<AdGroupAdditionalTargeting> = emptyList(),
    val mobileContent: MobileContentDetails? = null,
    val pageTargets: List<PageBlock> = emptyList(),
    val trackingParams: String? = null,
    val priority: Long? = null,
    val biddableShowConditionsIds: List<Long> = emptyList(), // идентификаторы уже в формате grut
    val cryptaSegmentsMapping: Map<Long, Goal>, // маппинг cryptaSegments goalId в Goal, нужен для additional targeting с типом ContentCategory
    val bidModifiersIds: Collection<Long> = emptyList(),

)

data class InternalAdGroupOptions(
    val rf: Int?,
    val rfReset: Int?,
    val maxClicksCount: Int?,
    val maxClicksPeriod: Int?,
    val maxStopsCount: Int?,
    val maxStopsPeriod: Int?,
    val startTime: LocalDateTime?,
    val finishTime: LocalDateTime?,
)

data class MobileContentDetails(
    val mobileContentId: Long,
    val deviceTypeTargeting: MutableSet<MobileContentAdGroupDeviceTypeTargeting>,
    val networkTargeting: MutableSet<MobileContentAdGroupNetworkTargeting>,
    val storeUrl: String,
    val minOsVersion: String,
)
