package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass
import ru.yandex.grut.objects.proto.Asset
import ru.yandex.grut.objects.proto.MediaType
import ru.yandex.grut.objects.proto.client.Schema

abstract class AssetGrut {
    abstract val id: Long
    abstract val clientId: Long
    abstract val mediaType: MediaType.EMediaType
}

data class TextAssetGrut(
    override val id: Long,
    override val clientId: Long,
    override val mediaType: MediaType.EMediaType = MediaType.EMediaType.MT_TEXT,
    val text: String,
) : AssetGrut()

data class TitleAssetGrut(
    override val id: Long,
    override val clientId: Long,
    override val mediaType: MediaType.EMediaType = MediaType.EMediaType.MT_TITLE,
    val title: String,
) : AssetGrut()

data class ImageAssetGrut(
    override val id: Long,
    override val clientId: Long,
    override val mediaType: MediaType.EMediaType = MediaType.EMediaType.MT_IMAGE,
    val mdsInfo: Asset.TMdsInfo,
    val imageHash: String?,
) : AssetGrut()

data class VideoAssetGrut(
    override val id: Long,
    override val clientId: Long,
    override val mediaType: MediaType.EMediaType = MediaType.EMediaType.MT_VIDEO,
    val mdsInfo: Asset.TMdsInfo,
    val duration: Int?,
) : AssetGrut()


data class Html5AssetGrut(
    override val id: Long,
    override val clientId: Long,
    override val mediaType: MediaType.EMediaType = MediaType.EMediaType.MT_HTML5,
    val mdsInfo: Asset.TMdsInfo,
) : AssetGrut()


data class SitelinkAssetGrut(
    override val id: Long,
    override val clientId: Long,
    override val mediaType: MediaType.EMediaType = MediaType.EMediaType.MT_SITELINK,
    val title: String,
    val href: String,
    val description: String?,
) : AssetGrut()

class AssetGrutApi(grutContext: GrutContext, properties: GrutApiProperties = DefaultGrutApiProperties()) :
    GrutApiBase<AssetGrut>(grutContext, Schema.EObjectType.OT_ASSET, properties) {
    override fun buildIdentity(id: Long): ByteString {
        return Schema.TAssetMeta.newBuilder().setId(id).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TAssetMeta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: AssetGrut): ByteString {
        return Schema.TAssetMeta.newBuilder().apply {
            id = obj.id
            clientId = obj.clientId
            mediaType = obj.mediaType
        }.build().toByteString()
    }

    override fun serializeSpec(obj: AssetGrut): ByteString {
        return Asset.TAssetSpec.newBuilder().apply {
            when (obj) {
                is TextAssetGrut -> {
                    text = obj.text
                }
                is TitleAssetGrut -> {
                    title = obj.title
                }
                is SitelinkAssetGrut -> {
                    sitelink = Asset.TSitelinkAsset.newBuilder().apply {
                        title = obj.title
                        href = obj.href
                        obj.description?.let { description = it }
                    }.build()
                }
                is ImageAssetGrut -> {
                    image = Asset.TImageAsset.newBuilder().apply {
                        mdsInfo = obj.mdsInfo
                        obj.imageHash?.let { directImageHash = it }
                    }.build()
                }
                is VideoAssetGrut -> {
                    video = Asset.TVideoAsset.newBuilder().apply {
                        mdsInfo = obj.mdsInfo
                        obj.duration?.let { videoDuration = it }
                    }.build()
                }
                is Html5AssetGrut -> {
                    html5 = Asset.THtml5Asset.newBuilder().apply {
                        mdsInfo = obj.mdsInfo
                    }.build()
                }
            }
        }.build().toByteString()
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TAsset.parseFrom(rawMeta).meta.id
    }

    fun selectAssets(
        filter: String,
        attributeSelector: List<String> = listOf("/meta", "/spec"),
        index: String? = null,
        limit: Long? = null,
        continuationToken: String? = null,
        allowFullScan: Boolean = false
    ): List<Schema.TAsset> {
        return selectObjectsAs(filter, attributeSelector, index, limit, continuationToken, allowFullScan, ::transformToAsset)
    }

    fun getAssets(ids: Collection<Long>): List<Schema.TAsset> {
        val rawClients = getObjectsByIds(ids)
        return rawClients.filter { it.protobuf.size() > 0 }.map { transformToAsset(it)!! }
    }

    fun getAsset(id: Long): Schema.TAsset? {
        return getAssets(listOf(id)).firstOrNull()
    }

    fun updateAssets(objects: Collection<UpdatedObject>) {
        updateObjects(objects)
    }

    private fun transformToAsset(raw: ObjectApiServiceOuterClass.TVersionedPayload?): Schema.TAsset? {
        if (raw == null) return null
        return Schema.TAsset.parseFrom(raw.protobuf)
    }
}
