package ru.yandex.direct.core.grut.api

import com.google.protobuf.ByteString
import ru.yandex.direct.core.entity.banner.model.BannerImageOpts.SINGLE_AD_TO_BS
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage
import ru.yandex.direct.core.entity.banner.model.BannerWithBody
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative
import ru.yandex.direct.core.entity.banner.model.BannerWithHref
import ru.yandex.direct.core.entity.banner.model.BannerWithIsMobile
import ru.yandex.direct.core.entity.banner.model.BannerWithMobileContent
import ru.yandex.direct.core.entity.banner.model.BannerWithOnlyGeoFlag
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganization
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganizationAndPhone
import ru.yandex.direct.core.entity.banner.model.BannerWithPixels
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.BannerWithTitle
import ru.yandex.direct.core.entity.banner.model.BannerWithTitleExtension
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLandingParams
import ru.yandex.direct.core.entity.banner.model.NewReflectedAttribute
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate
import ru.yandex.direct.core.entity.bs.common.service.BsBannerIdCalculator.calculateBsBannerId
import ru.yandex.direct.core.entity.image.converter.BannerImageConverter
import ru.yandex.direct.core.entity.image.model.BannerImageFormat
import ru.yandex.direct.core.entity.image.model.BannerImageFormatNamespace
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.grut.api.utils.colorStringToUint64
import ru.yandex.direct.core.grut.api.utils.crc64StringToLong
import ru.yandex.direct.core.grut.api.utils.extractVcardId
import ru.yandex.direct.mysql2grut.enummappers.BannerEnumMappers.Companion.avatarsHostToEnvironment
import ru.yandex.direct.mysql2grut.enummappers.BannerEnumMappers.Companion.imageTypeToGrut
import ru.yandex.direct.mysql2grut.enummappers.BannerEnumMappers.Companion.toGrutBannerType
import ru.yandex.direct.mysql2grut.enummappers.BannerEnumMappers.Companion.toGrutLanguage
import ru.yandex.direct.mysql2grut.enummappers.BannerEnumMappers.Companion.toGrutPrimaryAction
import ru.yandex.grut.auxiliary.proto.MdsInfo.TMdsFileInfo
import ru.yandex.grut.object_api.proto.ObjectApiServiceOuterClass.TVersionedPayload
import ru.yandex.grut.objects.proto.Banner
import ru.yandex.grut.objects.proto.BannerV2
import ru.yandex.grut.objects.proto.BannerV2.TBannerV2Spec.TFlags
import ru.yandex.grut.objects.proto.BannerV2.TBannerV2Spec.TImage
import ru.yandex.grut.objects.proto.BannerV2.TBannerV2Spec.TMobileContent
import ru.yandex.grut.objects.proto.BannerV2.TBannerV2Spec.TPermalink
import ru.yandex.grut.objects.proto.BannerV2.TBannerV2Spec.TSitelinkSet
import ru.yandex.grut.objects.proto.MdsInfo.TAvatarsImageMeta
import ru.yandex.grut.objects.proto.client.Schema
import java.time.Duration

open class BannerGrutApi(
    grutContext: GrutContext,
    properties: GrutApiProperties = DefaultGrutApiProperties(),
    objectType: Schema.EObjectType = Schema.EObjectType.OT_BANNER_CANDIDATE,
    private val setPaths: List<String> = listOf("/spec"),
) : GrutApiBase<BannerGrut>(grutContext, objectType, properties) {

    override fun buildIdentity(id: Long): ByteString {
        return Schema.TBannerV2Meta.newBuilder().setId(id).build().toByteString()
    }

    private fun buildDirectIdIdentity(directId: Long): ByteString {
        return Schema.TCampaignV2Meta.newBuilder().setDirectId(directId).build().toByteString()
    }

    override fun parseIdentity(identity: ByteString): Long {
        return Schema.TBannerV2Meta.parseFrom(identity).id
    }

    override fun serializeMeta(obj: BannerGrut): ByteString {
        return Schema.TBannerV2Meta.newBuilder().apply {
            directId = obj.banner.id
            adGroupId = obj.banner.adGroupId
            campaignId = obj.orderId
            directType = toGrutBannerType(obj.banner).number
            source = Banner.EBannerSource.BS_DIRECT.number
            imageBannerId = calculateImageBannerId(obj)
        }
            .build()
            .toByteString()
    }

    override fun serializeStatus(obj: BannerGrut): ByteString? {
        return BannerV2.TBannerV2Status.newBuilder().apply {
            skipModeration = obj.skipModeration
        }.build().toByteString()
    }

    fun updateBanners(banners: List<UpdatedObject>) {
        updateObjects(banners)
    }

    open fun createOrUpdateBanners(banners: List<BannerGrut>) {
        createOrUpdateObjects(banners, setPaths)
    }

    open fun createOrUpdateBannersParallel(banners: List<BannerGrut>) {
        createOrUpdateObjectsParallel(banners, UPDATE_TIMEOUT, setPaths)
    }

    fun deleteBannersByDirectIds(directIds: List<Long>) {
        val identities = directIds.map { buildDirectIdIdentity(it) }
        return deleteObjectsByIdentities(identities, true)
    }

    internal companion object {
        val bannerSpecFields = listOf(
            "status",
            "is_mobile",
            "href",
            "domain",
            "title",
            "title_extension",
            "body",
            "language",
            "flags",
            "pixels",
        )
        val bannerForeignSpecFields = listOf(
            "vcard_id",
            "images",
        )
        private val UPDATE_TIMEOUT = Duration.ofMinutes(1)

        fun toTImage(bannerWithImage: BannerWithBannerImage, bannerImageFormat: BannerImageFormat?): TImage {
            return TImage.newBuilder().apply {
                if (bannerWithImage.imageHash != null) {
                    assert(bannerImageFormat != null)

                    this.imageHash = bannerWithImage.imageHash
                    this.imageType = imageTypeToGrut(bannerImageFormat!!.imageType).number
                    this.mdsFileInfo = TMdsFileInfo.newBuilder().apply {
                        this.environment = avatarsHostToEnvironment(bannerImageFormat.avatarsHost).number
                        this.namespace = BannerImageFormatNamespace.toSource(bannerImageFormat.namespace)!!.literal
                        this.groupId = bannerImageFormat.mdsGroupId
                        this.mdsFileName = bannerWithImage.imageHash
                    }.build()
                    this.avatarsImageMeta

                    val mdsMeta = BannerImageConverter.toImageMdsMeta(bannerImageFormat.mdsMeta)!!
                    this.avatarsImageMeta = TAvatarsImageMeta.newBuilder().apply {
                        mdsMeta.meta.colorWizBack?.let { colorWizBackground = colorStringToUint64(it) }
                        mdsMeta.meta.colorWizButton?.let { colorWizButton = colorStringToUint64(it) }
                        mdsMeta.meta.colorWizButtonText?.let { colorWizButtonText = colorStringToUint64(it) }
                        mdsMeta.meta.averageColor?.let { averageColor = colorStringToUint64(it) }
                        mdsMeta.meta.mainColor?.let { mainColor = colorStringToUint64(it) }
                        mdsMeta.meta.crc64?.let { crc64 = crc64StringToLong(it) }

                        if (mdsMeta.meta.backgroundColors != null) {
                            this.backgroundColors = TAvatarsImageMeta.TBackgroundColors.newBuilder().apply {
                                mdsMeta.meta.backgroundColors.bottom?.let { bottom = colorStringToUint64(it) }
                                mdsMeta.meta.backgroundColors.left?.let { left = colorStringToUint64(it) }
                                mdsMeta.meta.backgroundColors.right?.let { right = colorStringToUint64(it) }
                                mdsMeta.meta.backgroundColors.top?.let { top = colorStringToUint64(it) }
                            }.build()
                        }

                        val formats = mdsMeta.sizes.map { size ->
                            val (sizeName, imageSizeMeta) = size
                            TAvatarsImageMeta.TFormat.newBuilder().apply {
                                this.formatName = sizeName
                                this.width = imageSizeMeta.width
                                this.height = imageSizeMeta.height

                                // смартцентры сортируются лексикографически по названию размера
                                if (imageSizeMeta.smartCenters != null) {
                                    val sortedSmartCenters: List<TAvatarsImageMeta.TFormat.TSmartCenter> =
                                        imageSizeMeta.smartCenters.mapValues {
                                            TAvatarsImageMeta.TFormat.TSmartCenter.newBuilder().apply {
                                                this.w = it.value.width
                                                this.h = it.value.height
                                                this.x = it.value.x
                                                this.y = it.value.y
                                            }.build()
                                        }.entries
                                            .toList()
                                            .sortedBy { it.key }
                                            .map { it.value }
                                    this.addAllSmartCenters(sortedSmartCenters)
                                }
                            }.build()
                        }.sortedBy { it.formatName }
                        this.addAllFormats(formats)
                    }.build()
                }
            }.build()
        }

        fun toTSitelinkSet(sitelinkSet: SitelinkSet, turbolandingsById: Map<Long, TurboLanding>): TSitelinkSet {
            return TSitelinkSet.newBuilder().apply {
                id = sitelinkSet.id
                val siteLinks = sitelinkSet.sitelinks.map { sitelink ->
                    TSitelinkSet.TSitelink.newBuilder().apply {
                        href = sitelink.href
                        title = sitelink.title
                        sitelink.description?.let { description = it }
                        sitelink.turboLandingId?.let {
                            turbolandingId = it
                            turbolandingHref = turbolandingsById[it]!!.url
                        }
                    }.build()
                }
                addAllSitelinks(siteLinks)
            }.build()
        }
    }

    /**
     * Сериализация полей самого баннера.
     * <p>
     * При редактировании метода — нужно актуализировать список bannerSpecFields выше
     */
    fun serializeBannerFields(
        builder: BannerV2.TBannerV2Spec.Builder,
        banner: BannerWithSystemFields,
    ): BannerV2.TBannerV2Spec.Builder {
        return builder.apply {
            status = calcBannerStatuses(banner).number

            // isMobile
            (banner as? BannerWithIsMobile)?.isMobile?.let { isMobile = it }

            // href
            (banner as? BannerWithHref)?.let { banner ->
                banner.href?.let { href = it }
                banner.domain?.let { domain = it }
            }
            // title
            (banner as? BannerWithTitle)?.title?.let { title = it }
            // titleExt
            (banner as? BannerWithTitleExtension)?.titleExtension?.let { titleExtension = it }
            // body
            (banner as? BannerWithBody)?.body?.let { body = it }
            // language
            banner.language?.let { language = toGrutLanguage(it).number }

            // flags
            flags = toFlags(banner)

            // pixels
            (banner as? BannerWithPixels)?.pixels?.let { addAllPixels(it) }
        }
    }

    /**
     * Сериализация внешних по отношению к баннеру сущностей. Общий код с репликацией.
     * При добавлении сущностей — нужно актуализировать список bannerForeignSpecFields выше
     */
    fun serializeForeignBannerFields(
        builder: BannerV2.TBannerV2Spec.Builder,
        obj: BannerGrut,
    ): BannerV2.TBannerV2Spec.Builder {
        val banner = obj.banner

        run {
            val vcardId = extractVcardId(banner)
            if (vcardId != null && !obj.dropVcard) {
                builder.vcardId = vcardId
            }
        }

        if (canSendImage(obj)) {
            // images
            builder.addAllImages(listOf(toTImage(banner as BannerWithBannerImage, obj.bannerImageFormat)))
        }

        return builder
    }

    protected fun canSendImage(obj: BannerGrut) =
        obj.banner is BannerWithBannerImage
            && !obj.dropImage
            && obj.banner.imageHash != null
            && obj.bannerImageFormat != null
            && obj.bannerImageFormat.mdsMeta != null

    override fun serializeSpec(obj: BannerGrut): ByteString {
        val banner = obj.banner
        return BannerV2.TBannerV2Spec.newBuilder().apply {
            serializeBannerFields(this, obj.banner)
            serializeForeignBannerFields(this, obj)

            // turbolanding
            (banner as? BannerWithTurboLanding)?.turboLandingId?.let {
                turbolandingId = it
                turbolandingHref = obj.turbolandingsById[it]!!.url
            }
            (banner as? BannerWithTurboLandingParams)?.turboLandingHrefParams?.let { turbolandingHrefParams = it }

            // sitelinkSet
            obj.sitelinkSet?.let { sitelinkSet = toTSitelinkSet(obj.sitelinkSet, obj.turbolandingsById) }
            // creatives
            (banner as? BannerWithCreative)?.let { banner ->
                banner.creativeId?.let { addCreativeIds(it) }
            }
            // organization
            (banner as? BannerWithOrganization)?.let { banner ->
                banner.permalinkId?.let { permalink = toTPermalink(banner) }
            }
            (banner as? BannerWithMobileContent)?.let { mobileContent = toTMobileContent(it) }
        }.build().toByteString()
    }

    private fun toFlags(banner: BannerWithSystemFields): TFlags {
        val flagsBuilder = TFlags.newBuilder()

        // geoflag пока пишем из Директа, но в будущем планируем вычислять на стороне Цезаря
        (banner as? BannerWithOnlyGeoFlag)?.geoFlag?.let { flagsBuilder.geoflag = it }

        return flagsBuilder.build()
    }

    private fun toTPermalink(bannerWithOrganization: BannerWithOrganization): TPermalink {
        return TPermalink.newBuilder().apply {
            bannerWithOrganization.permalinkId?.let { id = it }
            (bannerWithOrganization as? BannerWithOrganizationAndPhone)?.phoneId?.let { phoneId = it }
            bannerWithOrganization.preferVCardOverPermalink?.let { preferVcardOverPermalink = it }
            //TODO add assignType
        }.build()
    }

    private fun toTMobileContent(banner: BannerWithMobileContent): TMobileContent {
        return TMobileContent.newBuilder().apply {
            banner.primaryAction?.let { primaryAction = toGrutPrimaryAction(it).number }
            banner.reflectedAttributes?.let { addAllReflectedAttributes(listOf(toTReflectedAttribute(it))) }
            banner.impressionUrl?.let { impressionUrl = it }
        }.build()
    }

    private fun toTReflectedAttribute(reflectedAttributes: MutableMap<NewReflectedAttribute, Boolean>):
        TMobileContent.TReflectedAttribute {
        return TMobileContent.TReflectedAttribute.newBuilder().apply {
            reflectedAttributes[NewReflectedAttribute.RATING]?.let { rating = it }
            reflectedAttributes[NewReflectedAttribute.ICON]?.let { icon = it }
            reflectedAttributes[NewReflectedAttribute.PRICE]?.let { price = it }
            reflectedAttributes[NewReflectedAttribute.RATING_VOTES]?.let { ratingVotes = it }
        }.build()
    }

    private fun calcBannerStatuses(banner: BannerWithSystemFields): BannerV2.EBannerStatus {
        return if (banner.statusModerate == BannerStatusModerate.NEW) {
            BannerV2.EBannerStatus.BST_DRAFT
        } else if (banner.statusArchived != null && banner.statusArchived) {
            BannerV2.EBannerStatus.BST_ARCHIVED
        } else if (banner.statusShow != null && !banner.statusShow) {
            BannerV2.EBannerStatus.BST_STOPPED
        } else {
            BannerV2.EBannerStatus.BST_ACTIVE
        }
    }

    fun createOrUpdateBanner(obj: BannerGrut) {
        createOrUpdateObject(obj, setPaths)
    }

    override fun getMetaId(rawMeta: ByteString): Long {
        return Schema.TBannerV2.parseFrom(rawMeta).meta.id
    }

    fun getBanner(id: Long): Schema.TBannerV2? {
        return getObjectAs(id, ::transformToBanner)
    }

    fun getBanners(ids: Collection<Long>): List<Schema.TBannerV2> {
        val rawBanners = getObjectsByIds(ids)
        return rawBanners.filter { it.protobuf.size() > 0 }.map { transformToBanner(it)!! }
    }

    fun getBannersByDirectId(directIds: Collection<Long>): List<Schema.TBannerV2> {
        val rawBanners = getRawBannersByDirectId(directIds, false)
        return rawBanners.filter { it.protobuf.size() > 0 }
            .map { transformToBanner(it)!! }
    }

    fun getVersionedBannersByDirectId(directIds: List<Long>): List<VersionedGrutObject<Schema.TBannerV2>> {
        val rawBanners = getRawBannersByDirectId(directIds, true)
        return rawBanners.filter { it.protobuf.size() > 0 }
            .map { transformToVersionedBanner(it)!! }
    }

    private fun getRawBannersByDirectId(
        directIds: Collection<Long>,
        fetchTimestamps: Boolean,
    ): List<TVersionedPayload> {
        val identities = directIds.map { Schema.TBannerV2Meta.newBuilder().setDirectId(it).build().toByteString() }
        return getObjects(identities = identities, fetchTimestamps = fetchTimestamps)
    }

    // TODO(dimitrovsd): add index
    fun getBannersByAdGroupIds(adGroupIds: List<Long>): List<Schema.TBannerV2> {
        val rawBanners = selectObjects("[/meta/ad_group_id] IN (${adGroupIds.distinct().joinToString { "$it" }})")
        return rawBanners.filter { it.protobuf.size() > 0 }
            .map { transformToBanner(it)!! }
    }

    fun getBannerByDirectId(directId: Long): Schema.TBannerV2? {
        return getBannersByDirectId(listOf(directId)).firstOrNull()
    }

    private fun transformToBanner(raw: TVersionedPayload?): Schema.TBannerV2? {
        if (raw == null) return null
        return Schema.TBannerV2.parseFrom(raw.protobuf)
    }

    private fun transformToVersionedBanner(raw: TVersionedPayload?): VersionedGrutObject<Schema.TBannerV2>? {
        if (raw == null) return null
        return VersionedGrutObject(Schema.TBannerV2.parseFrom(raw.protobuf), raw.timestamp)
    }

    fun canSendImageId(banner: ru.yandex.direct.core.entity.banner.model.Banner): Boolean {
        if (banner !is BannerWithBannerImage) {
            // некартиночный баннер, потом пошлем 0
            return true
        }

        return if (banner.deletedImageBsBannerId != null) {
            // картинка уже удалена, у нее либо ненулевой BannerID и его можно отправить (и такая картинка не удалится)
            // либо нулевой (и тогда отправится все равно 0), и не так важно что она может удалиться из mysql
            true
        } else if (banner.imageBsBannerId == null) {
            // картинки нет (отправится 0)
            true
        } else if (banner.imageBsBannerId != 0L) {
            true
        } else {
            // если у картинки еще нет BannerID, проверим ее статус модерации — берем только принятые картинки
            // нужно, чтобы минимизировать возможность изменения /meta/image_banner_id, так как непринятую картинку
            // нельзя отправить в БК старым транспортом (который запишет ей BannerID)
            // и можно удалить (а новая будет создана со своим id)
            StatusBannerImageModerate.YES == banner.imageStatusModerate;
        }
    }

    private fun calculateImageBannerId(obj: BannerGrut): Long {
        if (obj.dropImage) {
            return 0
        }
        val banner = obj.banner
        if (banner !is BannerWithBannerImage) {
            return 0
        }
        if ((banner.imageId ?: 0) == 0L && (banner.deletedImageId ?: 0) == 0L) {
            // картинки точно нет
            return 0
        }

        if (banner.deletedImageBsBannerId != null) {
            // у баннера есть картинка когда-либо бывшая в БК, но она удалена.
            return banner.deletedImageBsBannerId
        }
        // не вычисляем BannerID из deletedImageId, так как в таком состоянии картинку все еще можно удалить
        return if ((banner.imageBsBannerId ?: 0L) == 0L) {
            val imageBanner = banner as BannerWithBannerImage
            val singleAdToBS = imageBanner.opts != null && imageBanner.opts.contains(SINGLE_AD_TO_BS)
            // В новой схеме картиночный баннер передаем в БК в родительском баннере
            // TODO: Поддержать единый баннер в отправке из грута
            if (singleAdToBS) {
                if (obj.banner.bsBannerId > 0) obj.banner.bsBannerId
                else calculateBsBannerId(imageBanner.id)
            } else {
                calculateBsBannerId(imageBanner.imageId)
            }
        } else banner.imageBsBannerId
    }
}

data class BannerGrut(
    val banner: BannerWithSystemFields,
    val orderId: Long,
    val sitelinkSet: SitelinkSet?,
    val turbolandingsById: Map<Long, TurboLanding>,
    val bannerImageFormat: BannerImageFormat?,
    // признак, что визитку нужно "сбросить" (не реплицировать, удалить)
    val dropVcard: Boolean = false,
    // признак, что репликацию картинки нужно пропустить
    val dropImage: Boolean = false,
    // признак, что Модерация в груте должна быть отключена (true) или нет (false)
    val skipModeration: Boolean = true,
)
